<?php

/*

 Copyright (C) 2005 - 2006, Domeo / Avisi B.V.

 Website Baker inline wrapper module is free software; you can
 redistribute it and/or modify it under the terms of the GNU
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your
 option) any later version.

 Website Baker is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Website Baker; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

// Direct request, process document.
if (!defined("WB_URL") && isset($_GET['wrapperurl'])) {

	require_once("functions.php");
	require_once("WebPage.php");
	require_once("HtmlPage.php");
	require_once("CssFile.php");

	$lUrl = urldecode($_GET['wrapperurl']);

	// Test filetype and forward to correspondending action class.
	$lPage = false;
	if (str_endswith(".css", strtolower($lUrl))) {
		// process CSS
		$lPage = new CssFile($lUrl);
	} else {
		// process HTML
		$lPage = new HtmlPage($lUrl);
	}

	$lPage->display();
}


/**
 * @package pages
 *
 * Contents web page.
 */
class WebPage {

	// Full URL of web page, including protocol, domain and request parameters.
	var $url;
	// ETag page
	var $eTag;
	// Page contents.
	var $contents;
	// Cache and cache meta info location.
	var $cachedir;
	// Contents rewritten?
	var $processed;
	// Is this a cached version or a fresh server document?
	var $cached;
	// Content-type
	var $contenttype;
	// Do not cache this page (HTTP POST reply will never be cached)
	var $docache;
	// Forward HTTP GET variables?
	var $forwardget;

	/**
	 * Constructor.
	 *
	 * @param $pUrl string  The absolute URL.
	 */
	function WebPage($pUrl) {

		global $IWCACHEDIR;

		$this->url = $pUrl;

		// Use WebsiteBakers temp dir for caching
		$this->cachedir = $IWCACHEDIR;

		// Default settings
		$this->eTag = "";
		$this->processed = false;
		$this->cached = false;
		$this->docache = true;
		$this->forwardget = false;

		// Add trailing slash, convert http://www.google.com to http://www.google.com/
		if (!str_contains("/", substr($pUrl, 8))) {
			$this->url .= "/";
		}
		if (str_contains(" ", $this->url)) {
			$this->url = str_replace(" ", "%20", $this->url);
		}
	}


	/**
	 * Get rewritten contents.
	 *
	 * @return The rewritten contents.
	 */
	function getContents() {

		// Process this document only ones.
		if (!$this->processed) {

			// Try to load external document.
			$this->loadRemoteDocument();

			// Can we use a cached version?
			if ($this->processed) {
				return;
			}

			// No, do the full document rewrite.
			$this->processDocument();
		}
	}


	/**
	 * Display the document
	 */
	function display() {

		$this->getContents();

		// Add some debugging
		$lDebugText = "";
		if (strtolower($this->contenttype) == "text/css") {
			$lDebugText = $this->cached ? "/* Inline wrapper: Cached version */" : "/* Inline wrapper: Fresh content */";
		} else {
			$lDebugText = $this->cached ? "<!-- Inline wrapper: Cached version -->" : "<!-- Inline wrapper: Fresh content -->";
		}

		print "\n<div id=\"wrapper\">\n";
		print $lDebugText . "\n" . $this->contents;
		print "</div>\n";

		// Flush Apache cache
		flush();

		// At last, save rewritten contents to cache.
		// HTTP POST reply will never be cached.
		$this->saveToCache();

		// Clean the cache once in a while
		$this->cleanCache();
	}


	// ======================================================
	//  Private methods
	// ======================================================


	/**
	 * Get contents external document or use cached version.
	 */
	function loadRemoteDocument() {

    // Detect force reload.
    $lForceReload = false;
    $lHeaders = apache_request_headers();
    if ((isset($lHeaders['Cache-Control']) && $lHeaders['Cache-Control'] == 'no-cache')
        || (isset($lHeaders['Pragma']) && $lHeaders['Pragma'] == 'no-cache')) {
      $lForceReload = true;
    }

		// Try to load the cached ETag.
		if (!$lForceReload) {
		  $this->loadCachedMetaInfo();
		}

		// Set cookie/session?
		$lCookie = false;
		if (isset($_SESSION['wrappercookie'])) {
			$lCookie = $_SESSION['wrappercookie'];
		}

		// Capture return headers.
		$lReturnHeaders = array();

		// If there are HTTP POST variables available, grab those variables and do a POST instead of a GET request.
		if (sizeof(array_keys($_POST))) {

			// Do a HTTP POST
			$this->contents = doPost($this->url, $lCookie, $_POST, $lReturnHeaders);

		} else {

			// Attach all HTTP GET variables to URL
			if ($this->forwardget) {
				foreach ($_GET as $lKey => $lValue) {
					$this->url .= str_contains("?", $this->url) ? "&" : "?";
					$this->url .= "$lKey=" . urlencode($lValue);
				}
			}
			
			// Do a HTTP GET request.
			$lRequestHeaders = array();
			if (!empty($this->eTag) && !$lForceReload) {
				// Check if page is modified.
				$lRequestHeaders[] = "If-None-Match: " . $this->eTag;
			}
			$this->contents = doGet($this->url, $lRequestHeaders, $lCookie, $lReturnHeaders);
		}

		
		// Do not cache by default.
		$this->docache = false;

		// Save cookies/session external page?
		if (isset($lReturnHeaders['set-cookie'])) {
			$_SESSION['wrappercookie'] = $lReturnHeaders['set-cookie'];
		}
		
		// If status Ok, remember the ETag.
		if (str_contains("200", $lReturnHeaders['status'])) {
			
			$this->eTag = $lReturnHeaders['etag'];
			$this->contenttype = $lReturnHeaders['content-type'];
			// Only cache GET requests.
			if (!sizeof(array_keys($_POST))) {
				$this->docache = true;
			}
		}
		
		// If status "302 Moved Temporarily" or "302 Object moved", forward to new page.
		elseif (str_contains("302", $lReturnHeaders['status'])) {
			
			$lForwardUrl = $lReturnHeaders['location'];
			
			// Make relative URL absolute
			if (!str_startswith("http", $lForwardUrl)) {
				if (str_startswith("/", $lForwardUrl)) {
					$lForwardUrl = $this->getRootPath() . substr($lForwardUrl, 1);
				} else {
					$lForwardUrl = $this->getPath() . $lForwardUrl;
				}
			}
			
			$this->url = $lForwardUrl;
			$this->eTag = "";
			$this->loadRemoteDocument();
			
			// Remove HTTP POST variables
			if (sizeof(array_keys($_POST))) {
				foreach ($_POST as $lKey => $lValue) {
					unset($_POST[$lKey]);
				}
			}
			
		}

		// Document not changed. Keep using the cached version of the contents.
		elseif (str_contains("304", $lReturnHeaders['status'])) {
			
			if ($this->loadFromCache()) {
				// We are done. No further processing.
				$this->processed = true;
				$this->cached = true;
			}
		}
	}
	
	
	/**
	 * Rewrite document for inline usage.
	 */ 
	function processDocument() {
		$this->contents = "<p>Error: Process document in inheritance class!</p>\n\n";
		$this->processed = true;	
	}
	
	
	/**
	 * Try to load the contents from cache.
	 *
	 * @return boolean  True if success.
	 */
	function loadFromCache() {

		$lTmpInfoFile = $this->cachedir . safe_filename($this->url);
		$this->contents = trim(implode("", @file($lTmpInfoFile)));
		return true;
	}
	

	/**
	 * Save contents and meta data(ETag/Content-Type) to cache. 
	 *
	 * @return boolean  True if success.
	 */ 
	function saveToCache() {

		// Cache not functional when no Etag available
		if (!$this->docache || $this->eTag == "") {
			return false;
		}

		// Save meta data and contents
		if ($this->saveMetaInfoToCache()
					&& $this->saveContentsToCache()) {
			return true;
		}
		return false;
	}
	
	/**
	 * Save contents to cache.
	 *
	 * @return boolean  True if success.
	 */
	function saveContentsToCache() {

		$lTmpFile = $this->cachedir . safe_filename($this->url);
		$fp = @fopen($lTmpFile, "w+");

		// invalid filepointer, probably no rights to write.
		if (!$fp) {
			// Tempdir not available?
			// Create directory and try again.
			@mkdir($this->cachedir);
			@chmod($this->cachedir, 0777);
			$fp = @fopen($lTmpFile, "w+");

			if (!$fp) {
				return false;
			}
		}

		// do an exclusive lock
		if (flock($fp, LOCK_EX)) {

			fwrite($fp, $this->contents);

			// release the lock
			flock($fp, LOCK_UN);

		} else {
			//echo "Couldn't lock the file !";
			return false;
		}

		// Close filepointer
		fclose($fp);
		return true;
	}


	/**
	 * Try to load Etag and Content-Type of cached version.
	 * 
	 * @return boolean  True if success.
	 */
	function loadCachedMetaInfo() {

		$this->eTag = "";

		// If GET parameter 'forcereload' is set, ignore the cached version.
		if (isset($_GET['forcereload'])) {
			return false;		
		}
		
		$lTmpInfoFile = $this->cachedir . safe_filename($this->url) . ".info";
		$lLines = @file($lTmpInfoFile);

		if ($lLines == false || !is_array($lLines)) {
			return false;
		}
		
		// Search for E-tag and Content-Type.
		for ($i=0; $i < sizeof($lLines); $i++) {
			if (str_startswith("Etag: ", $lLines[$i])) {
				$this->eTag = substr($lLines[$i], strlen("Etag: "));
			}
			if (str_startswith("Content-Type: ", $lLines[$i])) {
				$this->contenttype = substr($lLines[$i], strlen("Content-Type: "));
			}
		}
		
		return true;
	}

	
	/**
	 * Save E-tag and Content-Type to cache.
	 *
	 * @return boolean  True if success.
	 */ 
	function saveMetaInfoToCache() {

		$lTmpInfoFile = $this->cachedir . safe_filename($this->url) . ".info";
		$fp = @fopen($lTmpInfoFile, "w+");

		// Invalid filepointer, probably no rights to write.
		if (!$fp) {
			return false;
		}

		// Do an exclusive lock
		if (flock($fp, LOCK_EX)) {

			fwrite($fp, "Etag: ". $this->eTag . "\n");
			fwrite($fp, "Content-Type: " . $this->contenttype . "\n");

			// Release the lock
			flock($fp, LOCK_UN);

		} else {
			//echo "Couldn't lock the file !";
			return false;
		}

		// Close filepointer
		fclose($fp);
		return true;
	}
	
	
	/**
	 * Absolute path of URL with trailing slash.
	 * http://www.google.com/images/bla.png -> http://www.google.com/images/
	 * http://www.google.com/images/ -> http://www.google.com/images/
	 * http://www.google.com/images -> http://www.google.com/
	 *
	 * @return string  The path
	 */
	function getPath() {
		if (substr($this->url, -1, 1) == "/") {
			return $this->url;
		}
		// Chop filename and add trailing slash.
		return dirname($this->url) . "/";
	}
	
	
	/**
	 * Absolute root path of URL with trailing slash.
	 * http://www.google.com/images/bla.png -> http://www.google.com/
	 *
	 * @return string  The hostname part;
	 */
	function getRootPath() {
		return substr($this->url, 0, strpos(substr($this->url, 8), "/") + 9);
	}
	
	
	/**
	 * Get location and name of this included script for direct access.
	 *
	 * @return string  Full URL to this script.
	 */
	function getThisScriptName() {

		/*
		$lThisFileName = __FILE__;

		$lScriptFileName = $_SERVER['SCRIPT_FILENAME'];
		$lScriptName = $_SERVER['SCRIPT_NAME'];
		$lRealServerLocation = substr($lScriptFileName, 0, strlen($lScriptFileName) - strlen($lScriptName));
		return substr($lThisFileName, strlen($lRealServerLocation));
		*/
		return WB_URL . "/modules/inlinewrapper/WebPage.php";
	}
	
	
	/**
	 * Set if all HTTP GET varibales have to be forwarded.
	 *
	 * @param boolean $pValue  True if forward all HTTP GET variables.
	 */
	function setForwardget($pValue) {
		$this->forwardget = $pValue;
	}


	/**
	 * 	Delete cache content older than a month
	 */
	function cleanCache() {

		// Check cache dir for old content every 1000 hits or so
		if (rand(0,1000) == 500) {

			// First flush all cache.
			flush();

			// Never timeout
			set_time_limit(0);
			rm_cache_files($this->cachedir, 60 * 60 * 24 * 30);
		}
	}
}

?>