/*
 * Copyright (C) <2005>  <Steve Woodcock>
 * 
 * Created on Jul 31, 2005
 *
 */
package com.jofti.api;

import java.io.InputStream;

import com.jofti.config.IndexConfig;
import com.jofti.exception.JoftiException;


/**
 *
 * 
 * 
 * The manager is responsible for providing access to indexes. This class is not a singleton 
 * and is not intended to be used in that manner.
 * <p>
 * The usage for the IndexCache manager is either to retrieve an indexed cache that has been configured via a configuration file:
 *  e.g. <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.setConfigFile("configFile");<br>
 * manager.init();	<br>
 * Index index = manager.getIndexCache("name");<br>
 * <p>
 * or:
 * <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init(inputStream);	<br>
 * IndexCache index = (IndexCache)manager.getIndexCache("name");<br>
 * <p>
 * or by using one the addIndex methods:
 * <p>
 * 
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init();	<br>
 * IndexCache index = (IndexCache)manager.addIndexCache(indexConfig, "fileName");<br>
 * <p>
 * or
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init();	<br>
 * Index index = manager.addIndex(indexConfig, cacheImpl);<br>
 *<p>
 * The recommended usage for pre-existing Caches is to use the addIndex method and use a 
 * Listener adapter so all Index updates are managed using the listener callback methods. 
 * <p>
 * NameSpaced caches such as JBossCache are obtained using:<br>
 * <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init(configFile);	<br>
 * NameSpacedIndex index = manager.getNameSpacedIndex("name");<br>
 * 
  * @author  Steve Woodcock<br>
 * @version 1.10<p>
 */
public interface IndexManager
{
    /**
     * Sets a config file to use in the Manager. This file must be on the
     * classpath.<p>
     * @param configFile - the filename containing the config - relative to the classpath. 
     */
    public abstract void setConfigFile(String configFile);

    /**
     * 
     * Initialisation method that takes a config file name. This over-rides the
     * fileName (if any) set in the setConfigFile() method.This method (or one of the other init 
     * methods) must be called <b>BEFORE</b>  any other method is called on the manager.
     * <p>
     * @param configFileName - the filename containing the config - relative to the classpath.
     * @throws JoftiException - an exception detailing a failure to initialise the cache.
     */
    public abstract void init(String configFileName) throws JoftiException;

    /**
     * 
     * Initialisation method that takes no parameters. This configures the cache with the
     * fileName set in the setConfigFile() method. This method (or one of the other init 
     * methods) must be called <b>BEFORE</b>  any other method is called on the manager.
     * <p>
     * @throws JoftiException - an exception detailing a failure to initialise the cache.
     */
    public abstract void init() throws JoftiException;

    /**
     * 
     * Initialise method that takes an inputstream. This over-rides the fileName
     * (if any) set in the setConfigFile() method.This method (or one of the other init 
     * methods must be called <b>BEFORE</b> any other method is called on the manager.
     * <p>
     * @param configStream a stream containing the loaded file. 
     * @throws JoftiException - an exception detailing a failure to initialise the cache.
     */
    public abstract void init(InputStream configStream) throws JoftiException;

    /**
     * This method allows cache instances to be added to the manager
     * programatically rather than at start-up. The usage is for example:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * Index index =  manager.addIndex(config, cacheImpl,
     * xml-filename-with-classes-in-it);<br>
     * <p>
     * any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the 
     * @link IndexConfig class.<p>
     * 
     * If you are using a nameSpaced cache like JBossCache then the usage would
     * be:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * NameSpacedIndex index = (NameSpacedIndex)
     * manager.addIndex(config, cacheImpl, xml-filename-with-classes-in-it);<br>
     * 
     * Note: The cache implementations must be started correctly before they are
     * passed into this method. Added caches are assumed to have been started
     * and the manager will NOT attempt to initialise the actual cache
     * implementation.
     * 
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.<br>
     * @param cache -
     *            the cache implementation.<br>
     * @param classesFileName -
     *            the xml file containing the classes definitions for the cache.
     *            This file must be available on the classpath.<br>
     * @return The added cache.<br>
     * @throws JoftiException an exception detailing a failure to initialise the cache.
     */
    public abstract Index addIndex(IndexConfig config,
            Object cache, String classesFileName) throws JoftiException;

    /**
     * This method allows cache instances to be added to the manager
     * programatically rather than at start-up. <p>
     * 
     * 
     * Any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class.<p>
     * 
     * Note: This method is the equivalent of an index entry in the
     * configuration file. The manager will atttempt to construct and initialise
     * a new indexed cache based on the attributes in the IndexConfig class and
     * the class definition file. This method cannot be used with Listener type adapters and an 
     * attempt to do so will thow an exception.<p>
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.<br>
     * @param classesFileName -
     *            the xml file containing the classes definitions for the
     *            cache. This file must be available on the classpath.
     * @return The added cache.<br>
     * @throws JoftiException an exception detailing a failure to initialise the cache.
     */
    public abstract Index addIndexCache(IndexConfig config,
            String classesFileName) throws JoftiException;

    
    /**
     * This method allows cache instances to be added to the manager
     * programatically rather than at start-up.<p>
     * 

     * Any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class.<p>
     * 
     * 
     * Note: The cache implementations must be started correctly before they are
     * passed into this method. Added caches are assumed to have been started
     * and the manager will NOT attempt to initialise the actual cache
     * implementation.
     * 
     *      
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.<br>
     * @param cache -
     *            the cache implementation.<br>
     * @param stream -
     *            the inputstream containing the classes definitions for the
     *            cache loaded from the config xml file.<br>
     * @return The added Index.<br>
     * @throws JoftiException  an exception detailing a failure to initialise the cache.
     * */
    public Index addIndex(IndexConfig config, Object cache,
    			InputStream stream)  throws JoftiException;
    
    /**
     * This method allows cache instances to be added to the manager
     * programatically rather than at start-up.<p>
     * 
     *  
     * Any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class. This method will result in the
     * adapter used creating a new instance of its cache type.<p>
     * 
     * Note: This method is the equivalent of an index entry in the
     * configuration file. The manager will atttempt to construct and initialise
     * a new indexed cache based solely on the attributes in the IndexConfig
     * class.<P>
     * This method cannot be used with Listener type adapters and an 
     * attempt to do so will thow an exception.<p> 
     * Class definitions can be added using attribute classMappings in the
     * IndexConfig class. See this class for details on how to configure these.
     * <p>
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.<br>
     * @return The added Index.
     * @throws JoftiException   an exception detailing a failure to initialise the cache.
     */
    public abstract Index addIndexCache(IndexConfig config)
            throws JoftiException;

    /**
     * This method allows cache instances to be added to the manager
     * programatically rather than at start-up. The usage is for example:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * Index index =  manager.addIndex(config, cacheImpl,
     * xml-filename-with-classes-in-it);<br>
     * <p>
     * any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class.<p>
     * 
     * If you are using a nameSpaced cache like JBossCache then the usage would
     * be:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * NameSpacedIndex index = (NameSpacedIndex)
     * manager.addIndexedCache(config, cacheImpl);<p>
     * 
     * Note: The cache implementations must be started correctly before they are
     * passed into this method. Added caches are assumed to have been started
     * and the manager will NOT attempt to initialise the actual cache
     * implementation.<p>
     * 
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.<br>
     * @param cache -
     *            the cache implementation.<br>
     * @return The added cache.<br>
     * @throws JoftiException   an exception detailing a failure to initialise the cache.
     */
    public abstract Index addIndex(IndexConfig config,
            Object cache) throws JoftiException;

    /**
     * Retrieves an indexed cache from the manager. If the cache does not exist in the manager the 
     * method returns NULL, rather than throw an exception.<p>
     * 
     * Attempting to retrieve a name spaced indexed cache using this method will result in an 
     * exception.<p>
     * Use this method for Map, EHCache and OSCache types.<p>
     * @param indexName - the key name to retrive the indexed cache. This set in the config as the Name.<br>
     * @return - the cache or NULL if no cache can be found under that name.<br>
     * 
     * @throws JoftiException  an exception detailing a failure to retrieve the indexed cache.
     */
    public abstract Index getIndexCache(String indexName) throws JoftiException;

    
 
    /**
     * Retrieves a name spaced indexed cache from the manager. If the index does not exist in the manager the 
     * method returns NULL, rather than throw an exception.<p>
     * Use this method for JBossCache<br>
     * Attempting to retrieve a non-name spaced indexed cache using this method will result in an 
     * exception.<p>
     * @param indexName - the key name to retrive the indexed cache. This set in the config as the Name.<br>
     * @return - the cache or NULL if no cache can be found under that name.<br>
     * 
     * @throws JoftiException an exception detailing a failure to retrieve the indexed cache.
     */
    public abstract NameSpacedIndex getNameSpacedIndex(String indexName)
            throws JoftiException;

    
    /**
     * Used to only remove an indexed cache from the manager and return the cache instance. You should use this method
     when you want to remive the index but keep the cache. This is the opposite of the addIndex method.<p>
     Once this method has been called the index becomes unusable and behaviour if you retain a reference is indeterminate.<p>
     * 
     * @param cache - the cache to be removed.
     */
    
    public abstract Object removeIndex(Object cache);
    
    /**
     * Used to shutdown and remove an indexed cache from the manager. You should use this method
     * to remove an indexed cache from the manager and shutdown the cache instance- as some cache implementations explicitly require a 
     * shutdown phase to be run before they can be removed.<p>
     * 
     * @param cache - the cache to be destroyed.
     */
    public abstract void destroyIndex(Object cache);
    
    /**
     * Used to shutdown and remove all indexed caches from the manager. You should always use this method
     * to stop the manager - as some cache implementations explicitly require a 
     * shutdown phase to be run before they can be removed.<p>
     * 
     */
    public abstract void destroy();
    
  
    
    
}