/*
 *Copyright (C) <2005>  <Steve Woodcock>
 *
 * Created on 01 June 2005, 08:12
 */

package com.jofti.api;

import com.jofti.exception.JoftiException;


/**
 * The main interface used by clients of caches that are Name Spaced
 * indexes.<p>
 * 
 * The interface uses similar methods to the Map interface as most cache implementations 
 * can be easily adapted to these methods.<p>
 * 
 * As with the Map classes the metaphors are to add an object use the put method, to retrieve an 
 * object use get(nameSpace,key) and to remove use remove(nameSpace,key). 
 * <p>
 * The additional parameter on all these methods is a nameSpace object. The format of this object 
 * will depend on the cache implementation used. For JBossCache, for instance, the name Space is 
 * of type org.jboss.cache.Fqn.
 * <p>
 * The additional query method is for clients to query the cache by values other than the key. Please refer to 
 * the docs on IndexQuery.
 * <p>
 *  It is important to realise that although the interface takes an Object as both key and value, some caches will only 
 * allow certain types of Objects to be inserted. Check with the actual cache provider used to determine
 * these limitations.
 * <p>
 * The usage to obtain an indexed cache is to create a com.jofti.manager.IndexManager then use the 
 * getNameSpacedIndex method to obtain a reference to the indexed cache. You cannot retrieve a nameSpaced indexed cache using the
 * getIndex method.
 * <p>
 * 
 *  
 * @author  Steve Woodcock<br>
 * @version 1.4 <br>
 * 
 */
public interface NameSpacedIndex extends CacheAccessor, Index{
    
	
	 /**
     * Puts an object into the underlying cache instance and indexes the object according to
     * the class definitions in the index config file. Some caches may only accept certain types of 
     * objects.<p>
     * 
     * @param nameSpace value to use for nameSpace<br>
     * @param key value to use as key in cache<p>
     * @param value value to put into cache and to be indexed<p>
     * @throws JoftiException thrown as a wrapper exception that contains any cache specific exceptions.<p>
     */
	
    public void put(Object nameSpace, Object key, Object value) throws JoftiException;
        
    /**
     * Retrieves an object from the underlying cache instance. The behaviour of the caches have 
     * been normalised so that a not found object will always return null - rather than throw an exception (as some caches do) any cache error
     * tranformed into a log message only. This method only provides this normalised behaviour as it the 
     * caches are usually read-mostly in operation and most caches already do this.
     * <p>
     * @param nameSpace value to use for nameSpace <p>
     * @param key the key for the object in the name sapce cache<br>
     * @return the object mapped in the cache - or null if no object found.<p>
     */
    
    public Object get(Object nameSpace, Object key);
    
    
    /**
     * Deletes an object from the underlying cache. Attempting to remove a non-existant object will 
     * not generate an exception. This will also remove the key/object from the index.
     * <p>
     * @param nameSpace value to use for nameSpace <br>
     * @param key the key to remove from the cache<br>
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.<br>
     */
    public void remove(Object nameSpace, Object key) throws JoftiException;
    
    /**
     * Deletes all objects from the underlying cache within the namespace. Attempting to remove a non-existant object will 
     * not generate an exception. This will also remove the key/object entries for the namespace from the index.
     * <p>
     * @param nameSpace value to use for nameSpace<br>

     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
    public void removeNameSpace(Object nameSpace) throws JoftiException;
    
    /**
     * This will remove all the values in the cache and dump the current index.<p>

     * @throws JoftiException A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
    public void removeAll() throws JoftiException;
    
    
   
  
    
    
}
