/*
 * BTOperations.java
 *
 * Created on 26 May 2003, 08:53
 */

package com.jofti.btree;


import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.jofti.exception.JoftiException;
import com.jofti.locking.LockManager;
import com.jofti.util.OpenHashMap;

/**
 * Provides the group of low level operations that can be performed on the BTree. All
 * operations on the BTree are performed by this class ,rather than directly on the tree.<p>
 * <p>
 * The class is essentially an encapsulation of the functional update,search, contains and remove variants used 
 * by the TreeIndex. The class does not retain a handle to any particular {@link BTree} and so the same instance can be used 
 * for the transactional {@link BTree} instances found in some Tree instances.
 * </p>
 * 
 * 
 * @author Steve Woodcock<br>
 * @version 1.31
 */
public class BTOperations {

	private static NodeFactory factory = NodeFactory.getInstance();

	/** Creates a new instance of BTOperations */
	private BTOperations() {
	}

    /** approximate size of average results returned for map sizing*/
    private static int avRangeResults = 100;
    
	/**
	 * Inserts a value in the tree using a specific dimension. The key/value/dimension are
	 * grouped into a leaf node entry as a single entity. <p>
	 * 
	 * This method delegates the insert to the {@link BTree} insert method.
	 * <p>
	 * @param tree - the tree to insert the value into<br>
	 * @param key - the key to use as the unique id<br>
	 * @param object - the value to be inserted into the tree. This must be Comparable.<br>
	 * @param dimension - the dimension that this value is part of.<br>
	 * <br>
	 * @throws JoftiException a wrapping exception to trap errors in the tree.
	 */
	public static void insertValue(BTree tree, Comparable key,
			Comparable object, int dimension) throws JoftiException {

		LeafNodeEntry entry = factory.createLeafNodeEntry(key, new ValueObject(dimension, object));
		tree.insert(entry);
	}

	
	/**
	 * Inserts a key and the object attributes in the tree using a specific key dimension. The key/attributes/dimension are
	 * grouped into a leaf node entry. <p>
	 * 
	 * This method delegates the insert to the BTree insert method.
	 * <p>
	 * @param tree - the tree to insert the value into<br>
	 * @param key - the key to use as the unique id<br>
	 * @param object - the value to be inserted into the tree. This must be Comparable.<br>
	 * @param dimension - the dimension that this value is part of.<br>
	 * <br>
	 * @throws JoftiException a wrapping exception to trap errors in the tree.
	 */
	public static void insertKeyValue(BTree tree, Comparable key,
			Map attributes, int dimension) throws JoftiException {

		LeafNodeEntry entry = factory.createLeafNodeEntry(key, new KeyValueObject(dimension, key, attributes));
	
		tree.insert(entry);
	}

	/**
	 * Removes a particular uniqueId from the tree if it is indexed with that value and dimension. If the value 
	 * exists but the uniqueId is not stored with the value then no remove is performed. Otherwise the 
	 * the uniqueid is removed from the value's id set.
	 * <p>
	 * @param tree - the tree to operate on.<br>
	 * @param uniqueId - the uniqueId to remove<br>
	 * @param object - the value to look for in the tree<br>
	 * @param dimension - the dimension that the value is in.<br>
	 * @throws JoftiException
	 */
	public static void removeValue(BTree tree, Object uniqueId,
			Comparable object, int dimension) throws JoftiException {
		removeValueObject(tree, uniqueId, new ValueObject(dimension, object));
	}

	/**
	 * Removes a particular uniqueId from the tree if it is indexed with that 
	 * valueObject. If the value 
	 * exists but the uniqueId is not stored with the value then no remove is performed. Otherwise the 
	 * the uniqueid is removed from the value's id set.<p>
	 * 
	 * @param tree - the tree to operate on.<br>
	 * @param uniqueId - the uniqueId to remove<br>
	 * @param object - the value to look for in the tree<br>
	 * @throws JoftiException
	 */

	public static void removeValueObject(BTree tree, Object uniqueId,
			Comparable object) throws JoftiException {

		LeafNodeEntry entry = factory.createLeafNodeEntry(uniqueId,object);
		tree.removeEntry(entry);
	}

	/**
	 * Retrieves the matching uniqueIds for a particular value and dimension (if any).
	 * Failure to find a match returns an empty Map.<p>
	 * 
	 * @param tree - the tree to perform the search upon.<br>
	 * @param obj - the object to search for.<br>
	 * @param dimension - the dimension that the value is in.<br>
	 * @return the Map of results.<br>
	 * @throws JoftiException
	 * 
	 */
	public static Map match(BTree tree, Comparable obj, int dimension) throws JoftiException{
		return tree.matchDirect(new ValueObject(dimension, obj), null, null);	
	}
    
    /**
     * Retrieves the matching uniqueIds for a particular value and dimension (if any). The valueReturn object is used to specify what field alias we should 
     * be looking for against this ID.
     * Failure to find a match returns an empty Map.<p>
     * 
     * @param tree - the tree to perform the search upon.<br>
     * @param obj - the object to search for.<br>
     * @param dimension - the dimension that the value is in.<br>
     * @param valueReturn - The alias value of the object to return or null if we do not want to indicate a return restriction.<br>
     * @return the Map of results.<br>
     * @throws JoftiException
     * 
     */
	public static Map match(BTree tree, Comparable obj, int dimension, Object valueReturn) throws JoftiException{
		return tree.matchDirect(new ValueObject(dimension, obj), null, valueReturn);	
	}

    /**
     * Retrieves the matching uniqueIds for a particular array of values and dimension (if any).
     * Failure to find a match returns an empty Map.<p>
     * 
     * @param tree - the tree to perofrm the search upon.<br>
     * @param obj - the object to search for.<br>
     * @param dimension - the dimension that the value is in.<br>
     * @return the Map of results.<br>
     * @throws JoftiException
     * 
     */
    public static Map match(BTree tree, Comparable[] obj, int dimension) throws JoftiException{
        return match(tree, obj, dimension,null);
        
    }
    
    /**
     * Retrieves the matching uniqueIds for a particular array of values and dimension (if any).
     * The alias is used to pecify which alis we should use for the id. Failure to find a match returns an empty Map.<p>
     * 
     * @param tree - the tree to perform the search upon.<br>
     * @param obj - the object array containing the objects to search for.<br>
     * @param dimension - the dimension that the value is in.<br>
     * @param alias - The alias value of the object to return or null if we do not want to indicate a return restriction.<br>
     * @return the Map of results.<br>
     * @throws JoftiException
     * 
     */
    public static Map match(BTree tree, Comparable[] obj, int dimension, Object alias) throws JoftiException{
        
        OpenHashMap map = null;
        for (int i =obj.length-1;i>=0;i--){
            map = tree.matchDirect(new ValueObject(dimension, obj[i]), map,alias);
        }
        return map;  
    }
    
    /**
     * Returns the keys that are mapped against the comparable value.
     * @param tree
     * @param obj
     * @param dimension
     * @return
     * @throws JoftiException
     */
    public static Collection getKeyAttributes(BTree tree, Comparable obj,
			int dimension) throws JoftiException {
    	
        return tree.getAttributesDirect(new ValueObject(dimension, obj));
    }
    
	

	/**
	 * Checks if the tree contains a particular value in a dimension. This does not check if any 
	 * uniqueIds are stored against the value.<p>
	 * 
	 * @param tree - the tree to perform the search upon.<br>
	 * @param obj - the object to search for.<br>
	 * @param dimension - the dimension that the value is in.<br>
	 * @return the Map of results.<br>
	 * @throws JoftiException
	 */
	public static boolean contains(BTree tree, Comparable obj, int dimension)
			throws JoftiException {
		return tree.containsDirect(new ValueObject(dimension, obj));

	}

	
	
	private static INode getContainingNode(BTree tree, Comparable obj,
			int dimension) throws JoftiException {

		//first get the node we think the entry should be in
		ValueObject temp = new ValueObject(dimension, obj);
		INode result = (INode) tree.search(temp);
		if (result == null) {
			return null;
		}
		// now get the entry - is it exists from the node
		LeafNodeEntry val = ((IResultNode) result).getEntry(temp);
		if (val == null) {
			return null;
		} else {
			return ((ResultNode) result).getDelegate();
		}
	}

	/**
	 * Gets a collection of all value/dimension matches in the tree for a specific uniqueid.
	 * This method is used when we have a uniqueId and no object left in the cache (due to expiry etc) 
	 * and so the only alternative is a scan of the leaf nodes to find matches. This is very inefficient 
	 * if there are a lot of values in the tree but some cache implementations leave us with no alternative, as there are 
	 * no callbacks for some events.<p>
	 * @param tree - the Tree to search
	 * @param obj - the unique id to look for
	 * @param startDimension - the dimension where the key is by type.
	 * @return - a Collection of all the matching values that has the uniqueId
	 * @throws JoftiException
	 */
	public static Collection getAllValuesForKey(BTree tree, Comparable obj,
			int startDimension) throws JoftiException {
		// get the start point in the tree for the key dimension in the tree
		INode result = getContainingNode(tree, obj, startDimension);
		if (result != null) {
			// scan along the leaves if we find a key mapping
			Collection col = getAllValuesInTree(tree, result, obj,
					startDimension);
			return col;

		} else {
			// otherwise we should not have any entries ifthere is no key mapping
			return new ArrayList();
		}
	}

	/**
	 * Gets a collection of all value/dimension matches in the tree for a specific uniqueid.
	 * This method is used when we have a unique and no object left in the cache (due to expiry etc) 
	 * and so the only alternative is a scan of the leaf nodes to find matches. This is very inefficient 
	 * if there are a lot of values in the tree but some caches leave us with no alternative, as there are 
	 * no callbacks for some events.<p>
	 * @param tree - the Tree to search
	 * @param realNode - the initial node for the first value.
	 * @param obj - the unique id to look for
	 * @param startDimension - the dimension where the key is by type.
	 * @return - a Collection of all the matching values that has the uniqueId
	 * @throws JoftiException
	 */

	private static Collection getAllValuesInTree(BTree tree, INode realNode,
			Comparable obj, int startDimension) throws JoftiException {

		List resultList = new ArrayList();
		// gives us the containing node of this value

		if (realNode != null) {

			// get a read lock on this node
			try {
				boolean foundInDimension = false;
				boolean dimensionEnd = false;
				LockManager.acquireLock((Node) realNode, LockManager.READ_LOCK);
				boolean notEnd = true;

				while (notEnd) {
					Object[] entries = realNode.getEntries();
					for (int i = 0; i < entries.length; i++) {
						LeafNodeEntry entry = (LeafNodeEntry) entries[i];

						if (entry != BTree.MAX_LEAF_ENTRY) {
							if (entry.getUniqueIdSet().contains(obj)
									&& ((ValueObject) entry.getValue())
											.getDimension() == startDimension) {

								resultList.add(entry.getValue());
								foundInDimension = true;
								if (entries[entries.length - 1] != BTree.MAX_LEAF_ENTRY) {
									break;
								}
							} else if (((ValueObject) entry.getValue())
									.getDimension() > startDimension) {

								dimensionEnd = true;
								break;
							}

						} else {
							notEnd = false;

						}
					}
					if (notEnd) {
						INode nextNode = null;
						if (foundInDimension || dimensionEnd) {

							nextNode = getLowestNodeForDimension(tree,
									++startDimension);

							foundInDimension = false;
							dimensionEnd = false;
						} else {
							nextNode = realNode.getLinkNode().getNode();

						}

						LockManager.releaseLock((Node) realNode,
								LockManager.READ_LOCK);
						LockManager.acquireLock((Node) nextNode,
								LockManager.READ_LOCK);

						realNode = nextNode;

					}
				}

			} finally {
				LockManager.releaseLock((Node) realNode, LockManager.READ_LOCK);

			}
		}

		return (Collection) resultList;

	}

	/**
	 * Gets a collection of all uniqueIds in tree for a specific dimension.<p>
	 
	 * @param tree - the Tree to search
	 * @param dimension - the dimension where the key is by type.
	 * @return - a Collection of all the matching values
	 * @throws JoftiException
	 */
	public static Map getAllResultsForDimension(BTree tree, int dimension)
			throws JoftiException {

		return range(tree, ValueObject.MIN_COMAPARBLE_VALUE,
				ValueObject.MAX_COMAPARBLE_VALUE, dimension, true);

	}

	/**
	 * Gets the node containing the first entry for a dimension.<p>
	 
	 * @param tree - the Tree to search
	 * @param dimension - the dimension where the key is by type.
	 * @return - the starting node
	 * @throws JoftiException
	 */
	public static INode getLowestNodeForDimension(BTree tree, int dimension)
			throws JoftiException {
		ValueObject temp = new ValueObject(dimension,
				ValueObject.MIN_COMAPARBLE_VALUE);

		IResultNode result = (IResultNode) tree.search(temp);

		return ((ResultNode) result).getDelegate();
	}

	/**
	 * Gets the node containing the last entry for a dimension.<p>
	 
	 * @param tree - the Tree to search
	 * @param dimension - the dimension where the key is by type.
	 * @return - the starting node
	 * @throws JoftiException
	 */
	public static INode getHighestNodeForDimension(BTree tree, int dimension)
			throws JoftiException {
		ValueObject temp = new ValueObject(dimension,
				ValueObject.MAX_COMAPARBLE_VALUE);

		IResultNode result = (IResultNode) tree.search(temp);

		return ((ResultNode) result).getDelegate();
	}

	/**
	 * Gets all the uniqueIds that fall between the two objects for a dimension. If the startObj is NULL 
	 * it is assumed that the lower range starts from the first value in the dimension. Similarly, if the endObj is NULL
	 * it is assumed the upper bound is the maximum value in the dimension. <p>
	 
	 * @param tree - the Tree to search
	 * @param startObj - the starting value for the range search
	 * @param endObj - the end value for the range search
	 * @param dimension - the dimension where the key is by type.
	 * @param inclusive - sets whether the range search should be inclusive of the start and end objects.
	 * @return - A Map of the results.
	 * @throws JoftiException
	 */
     public static Map range(BTree tree, Comparable startObj, Comparable endObj,
                int dimension, boolean inclusive) throws JoftiException {

         return rangeArray(tree, startObj,endObj,dimension,dimension, inclusive, null);
     }
     
 	/**
 	 * Gets all the uniqueIds that fall between the two objects for a dimension. If the startObj is NULL 
 	 * it is assumed that the lower range starts from the first value in the dimension. Similarly, if the endObj is NULL
 	 * it is assumed the upper bound is the maximum value in the dimension. <p>
 	 
 	 * @param tree - the Tree to search
 	 * @param startObj - the starting value for the range search
 	 * @param endObj - the end value for the range search
 	 * @param dimension - the dimension where the key is by type.
 	 * @param inclusive - sets whether the range search should be inclusive of the start and end objects.
 	 * @param alias - passed back in the result map as the value to indicate what fields the key should filter on.
 	 * @return - a Map of the results.
 	 * @throws JoftiException
 	 */
     public static Map range(BTree tree, Comparable startObj, Comparable endObj,
            int dimension, boolean inclusive, Object alias) throws JoftiException {

     return rangeArray(tree, startObj,endObj,dimension,dimension, inclusive, alias);
 }
     
   
   
    
    private static Map rangeArray(BTree tree, Comparable startObj,
			Comparable endObj, int dimension, int endDimension,
			boolean inclusive, Object valueReturn) throws JoftiException {

		// we use an OpenHashMap here as it has lower memory and better speed
		// performance for larger numbers of results
		final OpenHashMap map = new OpenHashMap(avRangeResults * 2, 0.00f, 0.5f);

		if (startObj == null) {
			startObj = ValueObject.MIN_COMAPARBLE_VALUE;
		}
		if (endObj == null) {
			endObj = ValueObject.MAX_COMAPARBLE_VALUE;
		}

		ValueObject startRange = new ValueObject(dimension, startObj);
		ValueObject finishRange = new ValueObject(endDimension, endObj);

		// we do not need a lock on this node as it is a results node wrapper
		IResultNode startNode = (IResultNode) tree.search(startRange);

		boolean end = false;

		while (!end) {
			Object[] entries = startNode.getEntries();

			int tempLength = entries.length;

			// see if all are within range - we can short circuit some checks
			if (entries[tempLength - 1] != BTree.MAX_LEAF_ENTRY
					&& ((LeafNodeEntry) entries[0]).value.compareTo(startRange) > 0
					&& ((LeafNodeEntry) entries[tempLength - 1]).value
							.compareTo(finishRange) < 0) {
				for (int i = 0; i < tempLength; i++) {
					LeafNodeEntry entry = (LeafNodeEntry) entries[i];

					Object[] tempSet = entry.uniqueIdSet.toArray();
					final int length = tempSet.length;
					map.ensureCapacity(map.size() + length);

					// has to be iterator
					for (int j = length - 1; j >= 0; j--) {
						map.put(tempSet[j], valueReturn);
					}
				}
			} else {
				// we need to look at all values
				// change to int method of access
				for (int i = 0; i < tempLength; i++) {
					LeafNodeEntry entry = (LeafNodeEntry) entries[i];
					if (entry != BTree.MAX_LEAF_ENTRY) {

						if (inclusive && entry.value.compareTo(startRange) >= 0
								&& entry.value.compareTo(finishRange) <= 0) {

							Object[] tempSet = entry.uniqueIdSet.toArray();
							final int length = tempSet.length;
							map.ensureCapacity(map.size() + length);

							// has to be iterator
							for (int j = length - 1; j >= 0; j--) {
								map.put(tempSet[j], valueReturn);
							}

						} else if ((!inclusive)
								&& entry.value.compareTo(startRange) > 0
								&& entry.value.compareTo(finishRange) < 0) {

							Object[] tempSet = entry.uniqueIdSet.toArray();
							final int length = tempSet.length;

							map.ensureCapacity(map.size() + length);

							// has to be iterator
							for (int j = length - 1; j >= 0; j--) {
								map.put(tempSet[j], valueReturn);
							}

						} else if (entry.value.compareTo(finishRange) > 0) {
							// we are at the end

							end = true;
							break;

						}
					} else {
						end = true;
						break;
					}

				}
			}
			if (!end) {

				INode tempNode = ((ResultNode) startNode).getDelegate();

				// we temporarily acquire a lock here so we can get the result
				// node delegate
				LockManager.acquireLock((Node) tempNode, LockManager.READ_LOCK);

				// get the result node that is next node
				try {
					startNode = (IResultNode) startNode.getLinkNode().getNode();

				} finally {

					// do not lock couple here - and release the lock on the
					// node
					LockManager.releaseLock((Node) tempNode,
							LockManager.READ_LOCK);
				}

			}
		}

		// update the current average result size
		avRangeResults = (avRangeResults + map.size()) / 2;
		return map;

	}
    
    
    
    /**
 	 * Gets all the uniqueIds that are contained in the tree between two values irrespective of the dimension. If the startObj is NULL 
 	 * it is assumed that the lower range starts from the first value in the dimension. Similarly, if the endObj is NULL
 	 * it is assumed the upper bound is the maximum value in the dimension. <p>
 	 
 	 * @param tree - the Tree to search
 	 * @param startObj - the starting value for the range search
 	 * @param endObj - the end value for the range search
 	 * @param startdimension - the dimension of the startObj.
 	 * @param enddimension - the dimension of the end Obj.
 	 * @param inclusive - sets whether the range search should be inclusive of the start and end objects.
 	 * @return - A Map of the results.
 	 * @throws JoftiException
 	 */
    public static Map getSubTreeKeyValues(BTree tree, Comparable startObj, Comparable endObj,
            int dimension, int endDimension, boolean inclusive) throws JoftiException {
        Map map = new LinkedHashMap();

        if (startObj == null) {
            startObj = ValueObject.MIN_COMAPARBLE_VALUE;
        }
        if (endObj == null) {
            endObj = ValueObject.MAX_COMAPARBLE_VALUE;
        }

        ValueObject startRange = new ValueObject(dimension, startObj);
        ValueObject finishRange = new ValueObject(endDimension, endObj);

        IResultNode startNode = (IResultNode) tree.search(startRange);
 
    
            boolean end = false;

            while (!end) {
                Object[] entries = startNode.getEntries();
                // change to int method of access
                for (int i = 0; i < entries.length; i++) {
                    LeafNodeEntry entry = (LeafNodeEntry) entries[i];
                    if (entry != BTree.MAX_LEAF_ENTRY) {

                        if (inclusive && entry.value.compareTo(startRange) >= 0
                                && entry.value.compareTo(finishRange) <= 0) {

                            Object[] tempSet = entry.getUniqueIdSet().toArray();
                            // has to be iterator
                            for (int j=0;j<tempSet.length;j++) {
                            	if (entry.getValue() instanceof ValueObject){
	                            	Map tmpMap = getMapForDimension(map, new Integer(((ValueObject)entry.getValue()).dimension));
	                            	tmpMap.put(tempSet[j], entry.value);
                            	}
                            }

                        } else if ((!inclusive)
                                && entry.value.compareTo(startRange) > 0
                                && entry.value.compareTo(finishRange) < 0) {

                            Object[] tempSet = entry.getUniqueIdSet().toArray();
                            // has to be iterator
                            for (int j=0;j<tempSet.length;j++) {
                            	if (entry.getValue() instanceof ValueObject){
	                             	Map tmpMap = getMapForDimension(map, new Integer(((ValueObject)entry.getValue()).dimension));
	                                
	                                tmpMap.put(tempSet[j], entry.value);
                            	}
                            }

                        } else if (entry.value.compareTo(finishRange) > 0) {
                            //we are at the end

                            end = true;
                            break;

                        }
                    } else {
                        end = true;
                        break;
                    }

                }
                if (!end) {

                    INode tempNode = ((ResultNode)startNode).getDelegate();
                    LockManager.acquireLock((Node) tempNode, LockManager.READ_LOCK);
                    
                    try{
                    startNode = (IResultNode)startNode.getLinkNode().getNode();
                    
                    }finally{

                    //do not lock couple here
                    LockManager.releaseLock((Node) tempNode,
                            LockManager.READ_LOCK);
                    }

                }
            }
       
        return map;

    }
    
    private static Map getMapForDimension(Map resultMap, Integer dimension){
    	
     	Map tempMap = (Map)resultMap.get(dimension);
     	if (tempMap == null){
     		tempMap = new LinkedHashMap();
     		resultMap.put(dimension, tempMap);
     	}
     	return tempMap;
        
    }
    
    
   
	/**
	 * Gets all the uniqueIds that do not equal the object for a dimension. <p>
	 
	 * @param tree - the Tree to search
	 * @param obj - the value we want to exclude
	 * @param dimension - the dimension of the object.
	 * @return - the starting node
	 * @throws JoftiException
	 */
 public static Map notEqual(BTree tree, Comparable obj, int dimension, Object valueReturn)
			throws JoftiException {
        OpenHashMap map =  new OpenHashMap(avRangeResults*2,0.00f,0.5f);

		Comparable start = ValueObject.MIN_COMAPARBLE_VALUE;

		Comparable endObj = ValueObject.MAX_COMAPARBLE_VALUE;

		ValueObject startRange = new ValueObject(dimension, start);
		ValueObject finishRange = new ValueObject(dimension, endObj);

		ValueObject actualValue = new ValueObject(dimension, obj);

		IResultNode startNode = (IResultNode) tree.search(startRange);
		INode realNode = ((ResultNode) startNode).getDelegate();

		try {
			LockManager.acquireLock((Node) realNode, LockManager.READ_LOCK);

			boolean end = false;

			while (!end) {
				Object[] entries = realNode.getEntries();

				for (int i = 0; i < entries.length; i++) {
					LeafNodeEntry entry = (LeafNodeEntry) entries[i];
					if (entry != BTree.MAX_LEAF_ENTRY) {
						Comparable tempObj = entry.value;
						if (tempObj.compareTo(startRange) >= 0
								&& tempObj.compareTo(finishRange) <= 0) {

							if (tempObj.compareTo(actualValue) != 0) {
								

                                Object[] tempEntrySet = entry.getUniqueIdSet().toArray();
                                
                                map.ensureCapacity(map.size() + tempEntrySet.length);
                                Object retVal = valueReturn != null ?valueReturn : null ;
                                
                                
                                for (int j=0;j<tempEntrySet.length;j++) {
                                    map.put(tempEntrySet[j], retVal);
                                }
							}

						} else if (tempObj.compareTo(finishRange) > 0) {
							//we are at the end

							end = true;
							break;

						}

					} else {
						end = true;
						break;
					}

				}
				if (!end) {

					INode nextNode = realNode.getLinkNode().getNode();

					//do not lock couple here
					LockManager.releaseLock((Node) realNode,
							LockManager.READ_LOCK);
					LockManager.acquireLock((Node) nextNode,
							LockManager.READ_LOCK);

					realNode = nextNode;
				}
			}
		} finally {
			LockManager.releaseLock((Node) realNode, LockManager.READ_LOCK);
		}
		return map;

	}

}