/*
 * BTree.java
  *  Copyright (C) <2005>  <Steve Woodcock>
 *  

 * Created on 09 May 2003, 15:13
 */
package com.jofti.btree;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import edu.emory.mathcs.backport.java.util.concurrent.atomic.AtomicLong;


import com.jofti.core.IStoreManager;
import com.jofti.exception.JoftiException;
import com.jofti.locking.LockManager;
import com.jofti.oswego.concurrent.WriterPreferenceReadWriteLock;
import com.jofti.util.OpenHashMap;


/**
 * A B*Tree variation based on the paper by Lehman and Yao [Efficient concurrent operations on b-trees - 1981] on concurrent BTree design.<p> 
 * Following on from this paper the tree also adopts the recommendations of not coalesing nodes when half full. Rather nodes are only removed when empty.
 * This reduces the structural overheads significantly, while trading off for some redundant space. This space depends on the data order on deletion. But assuming it is reasonably 
 * random the space is a worthwhile tradeoff.<p>
 * 
 * @author Steve Woodcock<br>
 * @version 1.43<br>
 */
public class BTree {

	private static Log log = LogFactory.getLog(BTree.class);
	
	//not final as does get reset
	private static int MAX_NODE_SIZE = 64;
	
	private Node rootNode = null;

	private NodeFactory factory=null;
	
	public static LeafNodeEntry MAX_LEAF_ENTRY ;

	
	public static final ValueObject MIN_RIGHT_VALUE = new ValueObject(Integer.MIN_VALUE,ValueObject.MIN_COMAPARBLE_VALUE);
	
	// lock for the whole tree for structural modifications - possibly change to semaphore
    private final WriterPreferenceReadWriteLock treeEntryLock = new WriterPreferenceReadWriteLock();

    
    private long nodes =0;
    
    // count of number of entries
    private AtomicLong entries =new AtomicLong(0);
    
    // used to store LeafNodes to disk
    IStoreManager overflowManager =null;
    
	/**
	 * Constructs a new BTree
	 */
	public BTree()
	{
	}
    
    /**
     * Creates a BTree with a set maxNodeSize.
     * @param maxNodeSize
     */
    public BTree(int maxNodeSize)
    {
        MAX_NODE_SIZE = maxNodeSize;
    }
    
	/**
	 * @return the tree's current maxnode size.
	 */
	public static int getMaxNodeSize(){
	    return MAX_NODE_SIZE;
    }

	protected Node getRootNode()
	{
		return rootNode;
	}


	/**
	 * Used to initialise the B-Tree. This <b>must</b> be called in order to set a needed max
	 * right value. Without this subsequent calls will result in a failure.
	 * 
	 * @throws JoftiException
	 */
	public void init(Properties props) throws JoftiException
	{
        
		// see if we have an overflow manager configured
		factory = NodeFactory.getInstance(overflowManager);
		   
        MAX_LEAF_ENTRY = factory.createMaxLeafNodeEntry();
        
        // we need to be sure that the tree is not being restrutured
        LockManager.acquireRWLock(treeEntryLock, LockManager.WRITE_LOCK);
		rootNode = factory.createLeafNode();
		rootNode.setEntries(new Object[BTree.getMaxNodeSize()]);
		
		// max value for right value
		rootNode.setRightValue(MAX_LEAF_ENTRY.getValue());
		
		//		 populate tree with max value
		rootNode.insertEntry(MAX_LEAF_ENTRY);
        nodes++;
        LockManager.releaseRWLock(treeEntryLock, LockManager.WRITE_LOCK);

	}


	
	/**
     * 
     * Inserts a nodeEntry into the BTree. All values are stored in the leaves of the tree.<p>
     * <p>
     * The insertion acquires a readlock on the treeEntryLock.
     * </p>
     *<p>
     *The insertion will acquire readlocks as it descends the tree until it finds the correct node, upon which it will 
     *acquire a write lock. The descent is not lock coupled so if we arrive at the wrong leaf node we scan across until we find the 
     *correct node to insert. This behaviour prevents us having to lock the tree, or the path and is enabled by the right nod pointers in each node.
     *</p>
     *<p>
     *Having inserted the entry into a leaf node, the stack of nodes is then unwound to insert relevant pointers into the parent 
     *nodes if necessary. A similar locking strategy is applied on this ascent.
     *</p>
	 * @param entry a Node Entry.
	 * @throws JoftiException thrown on error inserting the entry into a node.
	 */
	public void insert(NodeEntry entry) throws JoftiException
	{
		Stack nodeStack = new Stack();
		
		// we need to be sure that the tree is not being restrutured
		LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);
		try {
			nodeStack.push(rootNode);
			insert(entry, nodeStack);
            entries.incrementAndGet();
		} finally{
			LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);
		}
	}

	/**
	 * <p>
	 * Removes all entries in the index and resets the nodes to 0.
	 * </p>
	 * <p>
	 * Acquires a write lock on the treeEntryLock.
	 * </p>
	 * <p>
	 * If an overflowManager is set in the tree this will also dump the disk storage.
	 * </p>
	 */
	public void removeAll(){
		
		try {
			LockManager.acquireRWLock(treeEntryLock, LockManager.WRITE_LOCK);
			
			//reset the manager
			if (overflowManager != null){
			    overflowManager.removeAll();
            }
			
			rootNode = factory.createLeafNode();
			rootNode.setEntries(new Object[BTree.getMaxNodeSize()]);
			rootNode.setRightValue(MAX_LEAF_ENTRY.getValue());
			rootNode.insertEntry(MAX_LEAF_ENTRY);
			nodes=1;
            entries =new AtomicLong(0);
			LockManager.releaseRWLock(treeEntryLock, LockManager.WRITE_LOCK);
		} catch (JoftiException e){
			log.fatal("Error removing all entries ",e);
			
		}

	}

	protected Node findNode(NodeEntry entry, Stack nodeStack) throws JoftiException
	{
		// get the first node
		Node currentNode = (Node) nodeStack.pop();

		//acquire a read lock on the node
		LockManager.acquireLock(currentNode, LockManager.READ_LOCK);

		// while not a leaf node scan through tree and record path
		while (!currentNode.isLeaf())
		{

			Object returnNode = scanNode(entry.getValue(), currentNode);
			// if not a link node then push prior node onto the stack

			if (!(returnNode instanceof NodeLink))
			{
				// if not the link node from scan node then it must be
				// a child node and we can put that on the stack
				nodeStack.push(currentNode);
				currentNode =  (Node)returnNode;
			} else
			{
				currentNode = (Node)((NodeLink) returnNode).getNode();
			}
			// repeat this loop until we arrive at a leaf
		}

		// release the read lock we held on the leaf
		LockManager.releaseLock(currentNode, LockManager.READ_LOCK);

		//		 we now acquire a write lock on the leaf node
		// we could be on the wrong leaf here by the time we acquire
		// this write lock

		LockManager.acquireLock(currentNode, LockManager.WRITE_LOCK);

		// we may well be in the wrong node a split may have occurred between
		// the read release and the

		// write lock gain so lets scan right with (write) lock coupling
		currentNode = (Node)moveRight(currentNode, entry);

		return currentNode;
	}


	protected void insert(NodeEntry entry, Stack nodeStack) throws JoftiException
	{

		Node currentNode = (Node)findNode(entry, nodeStack);
		// ok so now we can insert in this node and we have a write lock on this
		// node

		doInsert(currentNode, entry, nodeStack);

	}


	private void doInsert(Node currentNode, NodeEntry entry, Stack nodeStack)
			throws JoftiException
	{

		// we should now have the right node
		Object[] temp = currentNode.insertEntry(entry);

		// see if we need to split
		if (currentNode.isUnderFull())
		{
			// no need to split so we can return here
			LockManager.releaseLock(currentNode, LockManager.WRITE_LOCK);
		} else
		{
			// split the node
			Node newNode = currentNode.splitNode(temp);

		    // set the link node for the new node to be
			// the current node's link node
			newNode.setLinkNode(currentNode.getLinkNode());

			// create the new index node key
			IndexNodeEntry newEntry = constructKey(newNode);
			newEntry.setValue(newNode.getRightValue());

			// create a link node the new node
			NodeLink newLink = new NodeLink();
			newLink.setNode(newNode);
			// set up current node to point to new node - but no new node in
			// parent yet
			currentNode.setLinkNode(newLink);
			
			// current node is now sorted out and new node is populated
			// now we have to insert the new node in the parent
			// no one can read into current or new node as yet as write lock
			// still
			// in effect
			Node oldNode = currentNode;
			nodes++;
			// see if we are at the top of the stack
			if (nodeStack.isEmpty())
			{

				IndexNode newRoot = factory.createIndexNode();

				// these must be inserted this way round
				newRoot.insertEntry(newEntry);

				//				 create the new index node key
				IndexNodeEntry originalEntry = constructKey(oldNode);
				originalEntry.setValue(oldNode.getRightValue());

				newRoot.insertEntry(originalEntry);

				//set new right value
				newRoot.setRightValue(newEntry.getValue());

				rootNode = newRoot;
				LockManager.releaseLock(oldNode, LockManager.WRITE_LOCK);
			} else
			{
				currentNode = (Node) nodeStack.pop();
				// acquire a lock on the parent here
				LockManager.acquireLock(currentNode, LockManager.WRITE_LOCK);
				// look through parents to make sure we can insert in the right
				// place
				currentNode = (Node)moveRight(currentNode, newEntry);

				LockManager.releaseLock(oldNode, LockManager.WRITE_LOCK);

				// we have a lock on the current node so lets insert it
				doInsert(currentNode, newEntry, nodeStack);

			}

		}
	}


	private Node moveRight(Node currentNode, NodeEntry value)
			throws JoftiException
	{
		boolean rightnode = false;
		//scan along the level looking for a node whose right value
		// is greater then the value we are looking for.
		//we do this using lock coupling as we need to be
		// sure the nodes are not altered as we move right
		while (!rightnode)
		{
			// we have found the node we are looking for
			if (currentNode.getRightValue().compareTo(value.getValue()) >= 0)
			{
				rightnode = true;

			} else
			{
				// get the next node from the link node
				// we cannot go further right as there should always be
				// a max value set
				Node nextNode = (Node)currentNode.getLinkNode().getNode();

				LockManager.acquireLock(nextNode, LockManager.WRITE_LOCK);
				// we have locks on both current and next node here
				// if deleted we should set the next node link on the current
				// node
				// while we are scanning through it
				while (nextNode.isDeleted())
				{
					// we have to set the next link of current node to be the
					// next
					// nodes one to free up the node
					currentNode.setLinkNode(nextNode.getLinkNode());
					LockManager.releaseLock(nextNode, LockManager.WRITE_LOCK);
					nextNode = (Node)currentNode.getLinkNode().getNode();
					LockManager.acquireLock(nextNode, LockManager.WRITE_LOCK);
				}
				// now we can release the current Node
				LockManager.releaseLock(currentNode, LockManager.WRITE_LOCK);
				// set this node to be current node in readiness for testing
				// in while loop
				currentNode = nextNode;
			}
		}
		// ok so we have found the a node that has a bigger right value
		// lets return here
		return currentNode;
	}


	private Object scanNode(Comparable value, Node startNode)
			throws JoftiException
	{
		Node currentNode = startNode;

		// this is a split node as we expect to be in the right node
		// but the high value is too low so we need to return the pointer to the
		// next node instead
		if (currentNode.getRightValue().compareTo(value) < 0)
		{

			// this is not lock coupled as we are using read locks here
			// just to test one node at a time
			NodeLink nodeLink = currentNode.getLinkNode();

			LockManager.releaseLock(currentNode, LockManager.READ_LOCK);

			LockManager.acquireLock((Node)nodeLink.getNode(), LockManager.READ_LOCK);

			return nodeLink;
		} else
		{
			// try and get the containing node from this index node
			Node nextNode = ((IndexNode) currentNode).getContainingNode(value);

			LockManager.releaseLock(currentNode, LockManager.READ_LOCK);

			LockManager.acquireLock(nextNode, LockManager.READ_LOCK);

			return nextNode;

		}
	}


	/**
	 * <p>
	 * Used to find the node (if any) containing the value. 
	 * </p>
	 * <p>
	 * acquires read lock on the treeEntryLock
	 * </p>
	 * @param value - value to find
	 * @return
	 * @throws JoftiException
	 */
	public INode search(Comparable value) throws JoftiException
	{
		LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);
		try {
				return realSearch(value, rootNode);
		}finally{
			LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);
		}
	}

	
	
	/**
	 * <p>
	 * Used to test if the tree contains a value.
	 * </p>
	 * <p>
	 * acquires a read lock on the treeEntryLock
	 * </p>
	 * @param value to test
	 * @return
	 * @throws JoftiException
	 */
	
	
	public boolean containsDirect(Comparable value) throws JoftiException{
        
        LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);
        
        try{
            Node currentNode = rootNode;
            LockManager.acquireLock(currentNode, LockManager.READ_LOCK);

            // while not a leaf node scan through tree and record path
            while (!(currentNode.isLeaf()))
            {
                // will either return a child node or a link node
                
                Object returnNode = scanNode(value, currentNode);
               
                // if a link node then we try and scan the node pointed to by the
                // link
                if (returnNode instanceof NodeLink)
                {
                    currentNode = (Node)((NodeLink) returnNode).getNode();
                } else
                {
                    currentNode = (Node) returnNode;
                }

            }
            // repeat this loop until we arrive at a leaf
            Node tempNode = scanLeafNode(currentNode, value);
            boolean contains =false;
            
            // see if we actually have the value in our node
            if (tempNode != null && (((Leaf)tempNode).getEntry(value) != null)){
                contains =true;;
            }
            
          
            
            // release the node lock
            LockManager.releaseLock(tempNode, LockManager.READ_LOCK);
            
            // return the val
            return contains;
                   
        }finally{
            
                LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);
            
            }
        
    }


  
	protected Collection getAttributesDirect(Comparable value)
			throws JoftiException {
		Collection col = new ArrayList();

		LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);

		try {
			Node currentNode = rootNode;
			LockManager.acquireLock(currentNode, LockManager.READ_LOCK);

			// while not a leaf node scan through tree and record path
			while (!(currentNode.isLeaf())) {
				// will either return a child node or a link node

				Object returnNode = scanNode(value, currentNode);

				// if a link node then we try and scan the node pointed to by
				// the
				// link
				if (returnNode instanceof NodeLink) {
					currentNode = (Node) ((NodeLink) returnNode).getNode();
				} else {
					currentNode = (Node) returnNode;
				}

			}
			// repeat this loop until we arrive at a leaf
			Node tempNode = scanLeafNode(currentNode, value);

			if (tempNode == null) {
				return col;
			} else {
				try {
					// if no matching value in node it should be in then just
					// return an
					// empty map
					LeafNodeEntry val = ((Leaf) tempNode).getEntry(value);

					if (val == null || val.getUniqueIdSet() == null
							|| val.getUniqueIdSet().isEmpty()) {

						return col;
					} else {
						// put all the matching keys into the map
						// get the attributes
						KeyValueObject attribWrapper = null;
						try {
							attribWrapper = (KeyValueObject) val.getValue();
						} catch (ClassCastException t) {
							throw new JoftiException(
									"key dimension must only contain KeyValueObjects");
						}

						Set tempSet = attribWrapper.getAttributes().entrySet();
						Iterator other = tempSet.iterator();

						for (int j = tempSet.size(); j > 0; j--) {
							Map.Entry entry = (Map.Entry) other.next();
							Object tempVal = entry.getValue();
							if (tempVal.getClass().isArray()) {
								int size = Array.getLength(tempVal);
								for (int i = 0; i < size; i++) {
									Object inner = Array.get(tempVal, i);
									col.add(new ValueObject(((Integer) entry
											.getKey()).intValue(),
											(Comparable) inner));
								}
							} else {
								col.add(new ValueObject(((Integer) entry
										.getKey()).intValue(),
										(Comparable) entry.getValue()));
							}
						}

					}
				} finally {
					// release the node lock
					LockManager.releaseLock(tempNode, LockManager.READ_LOCK);
				}
			}

		} finally {

			LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);

		}
		return col;

	}


/**
 * Optimized matching method that uses direct matching in the tree instead of intermediate ResulNode wrapping the results.
 * 
 * @param value
 * @param map
 * @param valueReturn
 * @return
 * @throws JoftiException
 */
protected OpenHashMap matchDirect(Comparable value, OpenHashMap map, final Object valueReturn) throws JoftiException{
    
    LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);
    
    try{
        Node currentNode = rootNode;
        LockManager.acquireLock(currentNode, LockManager.READ_LOCK);

        // while not a leaf node scan through tree and record path
        while (!(currentNode instanceof Leaf))
        {
            // will either return a child node or a link node
            
            Object returnNode = scanNode(value, currentNode);
           
            // if a link node then we try and scan the node pointed to by the
            // link
            if (returnNode instanceof NodeLink)
            {
                currentNode = (Node)((NodeLink) returnNode).getNode();
            } else
            {
                currentNode = (Node) returnNode;
            }

        }
        // repeat this loop until we arrive at a leaf
        Node tempNode = scanLeafNode(currentNode, value);
        
        // we have to return an OpenHashMap here
        if (tempNode == null) {
            if (map == null){
                return new OpenHashMap(1);
            }else{
                return map;
            }
		}
        // get the results
        
       try {
        LeafNodeEntry val = ((Leaf)tempNode).getEntry(value);
           
        
        if (val !=null){
        	 
            // this is cached call for subsequent access to the set
            Object[] tempSet = val.uniqueIdSet.toArray();
            
            int setLength = tempSet.length;
             // make the map initially big enough to hold the length - so no resize is needed
            // this is 2x initial size - see openHashMap for docs
             if (map == null){
                 map = new OpenHashMap(setLength*2,0.00f,0.5f);
                 // add all elements from set
                 
                 for (int i=setLength-1;i>=0;i--) {
                     
                     map.put(tempSet[i],valueReturn);
                 }
             } else{
             	// make sure the map can enlarge to need no resize
                 map.ensureCapacity(map.size() + setLength);
                 
                 // add in value if it does not already contain it
                 for (int i=setLength-1;i>=0;i--)  {
                 	Object temp = tempSet[i];
                     if (! map.containsKey(temp)){
                         map.put(temp,valueReturn);
                     }
                 }
             } 
            
        }else{
        	// return a minimum sized map
            if (map == null){
                map = new OpenHashMap(1);
            }
        }
        
        // release lock
       } finally {
        LockManager.releaseLock(tempNode, LockManager.READ_LOCK);
       }
        return map;
        

        
    }finally{
        
            LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);
        
        }
    
}

	
	private INode realSearch(Comparable value, Node currentNode) throws JoftiException
	{
		// first acquire a read lock on the root node
		LockManager.acquireLock(currentNode, LockManager.READ_LOCK);

		// while not a leaf node scan through tree and record path
		while (!(currentNode instanceof Leaf))
		{
			// will either return a child node or a link node
            
			Object returnNode = scanNode(value, currentNode);
           
			// if a link node then we try and scan the node pointed to by the
			// link
			if (returnNode instanceof NodeLink)
			{
				currentNode = (Node)((NodeLink) returnNode).getNode();
			} else
			{
				currentNode = (Node) returnNode;
			}

		}
		// repeat this loop until we arrive at a leaf
		Node tempNode = scanLeafNode(currentNode, value);
		INode resultNode = new ResultNode(tempNode);
		
		// we should have a read lock here and a leaf value with a containing
		// range that is closest to our desired value
		LockManager.releaseLock(tempNode, LockManager.READ_LOCK);
		return resultNode;
	}


	private Node scanLeafNode(Node currentNode, Comparable value)
			throws JoftiException
	{
		boolean rightnode = false;
		while (!rightnode)
		{
			// this is true if any of the entries are >= the value
			// we are looking for
			if (currentNode.contains(value))
			{
				rightnode = true;

			} else
			{
				// otherwise scan along the nodes until we find a node that 
				// may contain our value - no lock coupling here
				Node nextNode = (Node)currentNode.getLinkNode().getNode();

				LockManager.releaseLock((Node)currentNode, LockManager.READ_LOCK);

				LockManager.acquireLock((Node)nextNode, LockManager.READ_LOCK);

				currentNode = nextNode;
			}
		}
		// ok so we have found the node we are looking for
		return currentNode;
	}


	private IndexNodeEntry constructKey(Node childNode)
	{
		IndexNodeEntry nodeKey = factory
				.createIndexNodeEntry();
		nodeKey.setValue(childNode.getRightValue());
		nodeKey.setChildNode(childNode);
		return nodeKey;
	}


	/**
	 * <p>
	 * Removes an entry from the tree that matches the {@link NodeEntry}.
	 * </p>
	 * <p>
	 * acquires a readlock on the treeEntryLock.
	 * </p>
	 * @param entry the entry to remove.
	 * @throws JoftiException
	 */
	public void removeEntry(NodeEntry entry) throws JoftiException
	{
		Stack nodeStack = new Stack();

		LockManager.acquireRWLock(treeEntryLock, LockManager.READ_LOCK);
		try {
			nodeStack.push(rootNode);
			removeEntry(entry, nodeStack);
            entries.decrementAndGet();
		}finally{
		
			LockManager.releaseRWLock(treeEntryLock, LockManager.READ_LOCK);
		
		}
			if (rootNode.getEntryNumber() == 1){
				collapseRoot();
			}
		}


	private void collapseRoot() throws JoftiException{
		LockManager.acquireRWLock(treeEntryLock, LockManager.WRITE_LOCK);
		
		// we need to make sure that no other threads are in the 
		// tree in order to do this collapse
		try {
			
			while (rootNode instanceof IndexNode && rootNode.getEntryNumber() ==1){
				rootNode = ((IndexNodeEntry)rootNode.getEntries()[0]).getChildNode();
			}
		}finally{
			
			LockManager.releaseRWLock(treeEntryLock, LockManager.WRITE_LOCK);
		
		}
	}
	private void removeEntry(NodeEntry entry, Stack nodeStack)
			throws JoftiException
	{
		Node currentNode = findNode(entry, nodeStack);
		// ok so now we can delete in this node and we have a write lock on this
		// node

		doRemove(currentNode, entry, nodeStack);
	}


	private void doRemove(Node currentNode, NodeEntry entry, Stack nodeStack)
			throws JoftiException
	{

		//we should now have the right node
		// lets delete the entry

		Comparable oldRightValue = currentNode.getRightValue();
		boolean deleted = currentNode.deleteEntry(entry);

		// if we did not delete a value or we are root or the right value of the
		// node
		// was not changed we can just release the lock and return
		if (nodeStack.isEmpty() || !deleted
				|| (currentNode.getRightValue().compareTo(oldRightValue) == 0))
		{
			
			LockManager.releaseLock((Node)currentNode, LockManager.WRITE_LOCK);

		} else
		{
			// construct an index entry so we can find the correct parent node
			// for the value we have just deleted
			IndexNodeEntry tempEntry = null;
			
			if (entry instanceof IndexNodeEntry){
				tempEntry = (IndexNodeEntry)entry;
			}else{
				tempEntry = factory
						.createIndexNodeEntry();
				tempEntry.setValue(oldRightValue);
			}

			// we have deleted all the values from the node
			if (currentNode.isEmpty())
			{
				
				// set a min right value so the scans will always go past it
				currentNode.setRightValue(MIN_RIGHT_VALUE);
				nodes --;
//				 we need to mark node for deletion
				currentNode.setDeleted(true);
				// and somehow reset right link from previous node
				// this we do on insert and delete from nodes
				// when the next node is marked as deleted - the lazy 
				// removal can result in some nodes hanging around a bit
				// but we dot really care about that as it is 
				// not really feasible to do it any other way

				currentNode = obtainParentNode(currentNode, nodeStack,
						tempEntry);

				// we need to remove the same value from the parent node
				doRemove(currentNode, tempEntry, nodeStack);
			} else
			{
				// must have deleted the right value of the node
				// without deleting the node
				// so we must step back up one level and reset the
				// value on the parent possibly to the root

				// get the right parent
				currentNode = obtainParentNode(currentNode, nodeStack,
						tempEntry);
				// update the right values as far as we need to
				doUpdateIndexValue(currentNode, tempEntry, nodeStack);

			}
		}

	}


	
	private Node obtainParentNode(Node currentNode, Stack nodeStack,
			IndexNodeEntry tempEntry) throws JoftiException
	{
		Node childNode = (Node)currentNode;

		// pop what was the parent on the way down off the stack
		currentNode = (Node) nodeStack.pop();
		
		// acquire a lock on the parent here
		LockManager.acquireLock(currentNode, LockManager.WRITE_LOCK);
		//scan the parent and if needed right siblings to find the correct
		// parent (could have moved in between the delete)

		currentNode = (Node)moveRight(currentNode, tempEntry);
		
		// we have the correct parent so release the child here
		LockManager.releaseLock((Node)childNode, LockManager.WRITE_LOCK);
		return currentNode;
	}


	private void doUpdateIndexValue(Node currentNode, NodeEntry entry,
			Stack nodeStack) throws JoftiException
	{

		Comparable oldRightValue = currentNode.getRightValue();

		boolean updated = ((IndexNode) currentNode).updateEntry(entry);
		if (nodeStack.isEmpty() || !updated
				|| (currentNode.getRightValue().compareTo(oldRightValue) == 0))
		{
			// we just need to return here
			LockManager.releaseLock((Node)currentNode, LockManager.WRITE_LOCK);

		} else
		{
			currentNode = obtainParentNode(currentNode, nodeStack,
					(IndexNodeEntry) entry);

			doUpdateIndexValue(currentNode, entry, nodeStack);
		}

	}


	/* 
	 * returns a String representation of the number of node and entries in the tree.
     * */
	public String toString()
	{
		StringBuffer buf = new StringBuffer();
		buf.append("number of nodes " + nodes + " number of entries "+ entries);
		return buf.toString();
	}
    
    /* 
     *This method needs to be called with caution as it will print out the entire node structure for the tree.
     * If you include this in log statements it will cause the performance to degenerate significantly.
     */
    public String treeToString()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(rootNode);
        return buf.toString();
    }
    
    /**
     * @return the number entries in the tree
     */
    public long getEntries(){
        return entries.get();
    }

    /**
     * Returns the StoreManager if one is configured or null.
     * @return
     */
    public IStoreManager getOverflowManager() {
		return overflowManager;
	}
    
	/**
	 * Sets an overflowManager in the tree to enable paging nodes to and from disk.
	 * @param overflowManager
	 */
	public void setOverflowManager(IStoreManager overflowManager) {
		this.overflowManager = overflowManager;
	}
	
}