/*
 * IndexNode.java
  *  Copyright (C) <2005>  <Steve Woodcock>
 *  

 * Created on 09 May 2003, 15:13
 */
package com.jofti.btree;



/**
 * <p>
 * The internal nodes in the BTree. These nodes do not contain any data - they are simply pointers to lower level 
 * index nodes or leaf nodes.</p>
 * <p>
 * Each node follows the pattern of an indexnode which is an array of sorted pointers to a lower level index or leaf node. Each entry 
 * in the array contains the largest value in the subtree under the pointer. The Node itself in its right value contains the largest 
 * entry in its array.
 * </p>
 *
 * @author  Steve Woodcock (steve@jofti.com)<br>
 * @version 1.12<br>
 * 
 */
class IndexNode extends Node {
    
    private IndexNodeEntry parentKey = null;
    
    protected Object[] keyList = new Object[BTree.getMaxNodeSize()+1];
    
       

    private Comparable rightValue = null;
    // b+ linked list metaphor to traverse sequentially
    
    private NodeLink nextNode = null;
    
    private boolean deleted =false;
    
	/**
	 * @return  if the node has been deleted.
	 */
	public boolean isDeleted()
	{
		return deleted;
	}
    
	/**
	 * @param deleted Sets the node to deleted if true.
	 */
	public void setDeleted(boolean deleted)
	{
		this.deleted = deleted;
	}
	
    /** Creates a new instance of IndexNode */
    protected IndexNode() {
    }
    
    
    
    /**
     * Returns the node entry where the entry should be. This is achieved by comparing the right values 
     * of the node entries in the node.<p>
     * 
     * 
     * @param entry the value to find.
     * @return The node in this index node that the value should be in (or in a subtreee from that node).
     */
    public Node getContainingNode(Comparable entry) {

			if (entryNumber ==0)
			{
				return null;
			}
			
			
			// look through list and see if we have a match
				
			return indexedBinaryRetrieve(keyList,entry);

		}

    protected  Node indexedBinaryRetrieve(Object[] list1, Object obj)
    {
        int i = 0;
        IndexNodeEntry entry =null;
        int size =entryNumber;
        for(int j =  size- 1; i <= j;)
        {
            int k = i + j >> 1;
            entry = (IndexNodeEntry)list1[k];
            int l = entry.value.compareTo(obj);
            if(l < 0){
                i = k + 1;
            } else if(l > 0){
                j = k - 1;
            }else{
                return entry.getChildNode();
            }
        }
        
        i= -(i + 1);
        
        if (-i == entryNumber){
        		return entry.getChildNode();
        }  else{
        	return  ((IndexNodeEntry)list1[-i-1]).getChildNode();
        }
        
    }

 
   
    protected  int indexedBinaryLocate(Object[] list1, Object obj)
    {

  		  
  		  int low = 0;
  			int high = entryNumber-1;
  			
  			IndexNodeEntry entry =null;
  			while (low <= high) {
  			    int mid = (low + high) >> 1;
  			    
  			    entry = (IndexNodeEntry)list1[mid];
  	            int cmp = entry.compareTo(obj);

  			    if (cmp < 0)
  				low = mid + 1;
  			    else if (cmp > 0)
  				high = mid - 1;
  			    else
  				return mid; // key found
  			}
  			return -(low + 1);  // key not found

  	    }



    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#insertEntry(com.jofti.btree.NodeEntry)
     */
    public Object[] insertEntry(NodeEntry key) {
    	
    	//lets reset the next node if it is deleted
        resetNextNode();
    	
    	IndexNodeEntry entry = (IndexNodeEntry) key;
        
    	// if there are no entries then set it in
        if (entryNumber == 0) {
            keyList[0]=entry;
            entry.setContainingNode(this);
            entryNumber++;
            rightValue=entry.getValue();
            return keyList;
           
        } else {
            //loop through entries to find right place
            // we do not use a binary search here as we want to do different things
            for (int i=0;i<entryNumber;i++) {
                IndexNodeEntry listEntry = (IndexNodeEntry)keyList[i];
                // the current node is bigger so insert here
                
                
                // is it bigger than a value in the list
                int comparison = listEntry.getValue().compareTo(entry.getValue());
                if ( comparison>0) {
                	
                	System.arraycopy(keyList, i, keyList, i + 1,
                			entryNumber - i);
                	keyList[i] = entry;

                    entry.setContainingNode(this);
                    entryNumber++;
					return keyList;
					// is it equal - which means we will need to rest the conflicting value to make sure
					// there is no problem - should be as the result of a split
                } else if (comparison ==0 ){
                	// this is a split node that has a key value that is equal
                	listEntry.updateValue();
                	if (i == entryNumber -1){
                		keyList[entryNumber]=entry;
                		rightValue=entry.getValue();
                		
                	}else{
                		int inner =i+1;
                		System.arraycopy(keyList, inner, keyList, inner+1,
                    			entryNumber - inner);
                    	keyList[inner] = entry;
                	}
                	 entry.setContainingNode(this);
                     entryNumber++;
                     return keyList;
                }
            }
       
            keyList[entryNumber]=entry;
            rightValue=entry.getValue();
            entry.setContainingNode(this);
            entryNumber++;
            return keyList;
        }
        
        
        
    }
    
    
    private void resetNextNode(){
    	if (nextNode != null && nextNode.getNode() != null 
    			&& nextNode.getNode().isDeleted())
    	{
    		nextNode = nextNode.getNode().getLinkNode();
    	}
    }
    
    /**
     * updates the key right value for the node entry passed in.<p> 
     * 
     * @param key - the key for the entry to update.<br>
     * @return - true of the right value for the entire node was updated - false if the node entry was 
     * not the right most entry.
     */
    public boolean updateEntry(NodeEntry key) {
    	
    	resetNextNode();
        IndexNodeEntry entry = (IndexNodeEntry) key;
        
        if (entryNumber == 0) {
           
            return false;
           
        } else {
            int length = entryNumber;
            for (int i=0;i<length;i++) {
                IndexNodeEntry listEntry = (IndexNodeEntry)keyList[i];
                if (listEntry.getValue().equals(entry.getValue()) ) {
                	// this is a split node that has a key value that is equal
                	listEntry.updateValue();
                	if (i == length -1){
                		rightValue=listEntry.getValue();
                	}
                	return true;
                }
            }
            return false;
        }
        
        
        
    }
    
    

    /**
     * Sets the keylist for the node entries.<p>
     * 
     * @param keys - the new entry list for the node.
     */
    public void setKeyList(Object[] temp){
    	


        for(int i=0;i<entryNumber;i++){
        		IndexNodeEntry entry = (IndexNodeEntry) temp[i];
	        	entry.setContainingNode(this);
	        	keyList[i]=entry;
        }

    }
    
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#isEmpty()
     */
    public boolean isEmpty() {
        if (entryNumber == 0) {
            return true;
        }
        return false;
    }
    
  
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#getRightValue()
     */
    public Comparable getRightValue() {
    	 return rightValue;
    }
    
   
    
    IndexNodeEntry getParentKey() {
        return parentKey;
    }
    
    
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("  IndexNode{");
       
        buf.append(" rightValue:" + getRightValue() +"}");
        return buf.toString();
    }
    
    public boolean isUnderFull() {
        if (entryNumber < BTree.getMaxNodeSize() ) {
            return true;
        }
        return false;
    }
    
    /** Setter for property parentNode.
     * @param parentNode New value of property parentNode.
     *
     *
     */
    void setParentKey(IndexNodeEntry parentKey) {
        this.parentKey = parentKey;
    }
    
 
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#deleteEntry(com.jofti.btree.NodeEntry)
     */
    public boolean deleteEntry(NodeEntry entry) {

    		resetNextNode();
    		
    		int location = indexedBinaryLocate(keyList,entry);
           
            if (location >=0) {
            	// we need to remove the entry
            	
            	int numMoved = entryNumber - location - 1;
            	if (numMoved > 0)
            	    System.arraycopy(keyList, location+1, keyList, location,
            			     numMoved);
            	keyList[--entryNumber] = null; // Let gc do its work
            	
            	
            
	            if (entryNumber ==0){
	            	rightValue = BTree.MIN_RIGHT_VALUE;
	            }else{
	            	((IndexNodeEntry)keyList[entryNumber-1]).updateValue();
	            	setRightValue(((IndexNodeEntry)keyList[entryNumber-1]).getValue());
	            }
	            return true;
            }
            return false;
            
    }
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#splitNode()
     */
    public Node splitNode(Object[] entries)
	{
		// first insert the entry

    	Object[] entriesList = split(entries,entryNumber);
		
    	
    	
		Node newNode = NodeFactory.getInstance().createIndexNode();

		Comparable currentRight = rightValue;

		//set up new node
		EntrySplitWrapper newEntries = ((EntrySplitWrapper)entriesList[1]);
		
		newNode.entryNumber = newEntries.size; 
		newNode.setEntries( (Object[]) newEntries.entries );
		
		
		newNode.setRightValue(currentRight);
		newNode.setLinkNode(getLinkNode());
		
//			replace values in current node
		
		EntrySplitWrapper replacements = (EntrySplitWrapper)entriesList[0];
		
		
		keyList = (Object[])replacements.entries;
		entryNumber = replacements.size;
		setRightValue(((NodeEntry) keyList[entryNumber - 1]).getValue());
		
		
	
		
		return newNode;

	}
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#getEntryNumber()
     */
    public int getEntryNumber() {
        return entryNumber;
    }
    
 
    
    /* (non-Javadoc)
     * @see com.jofti.btree.INode#getEntries()
     */
    public Object[] getEntries() {
    	
        return keyList;
    }
    


	/* (non-Javadoc)
	 * @see com.jofti.btree.Node#setRightValue(java.lang.Comparable)
	 */
    public void setRightValue(Comparable value)
	{
		this.rightValue = value;
		
	}

	/* (non-Javadoc)
	 * @see com.jofti.btree.Node#split()
	 */
    

	/* (non-Javadoc)
	 * @see com.jofti.btree.Node#setEntries(java.util.List)
	 */
    public void setEntries(Object[] entries)
	{
    	setKeyList(entries);
		
	}

	/* (non-Javadoc)
	 * @see com.jofti.btree.Node#getLinkNode()
	 */
    public NodeLink getLinkNode()
	{
		
		return nextNode;
	}

	/* (non-Javadoc)
	 * @see com.jofti.btree.Node#setLinkNode(com.jofti.btree.Node)
	 */
    public void setLinkNode(NodeLink node)
	{
		this.nextNode = node;
		
	}

	/** 
	 * This method checks to see if the value should be contained in the leaf nodes that are 
	 * children of this indexNode (or children of its children). It does not check if the actual 
	 * value is in the sub tree- rather it checks if its values are larger than this value.<p>
	 * 
	 * @param value
	 * 
	 * @return true if the node has a larger value in it - false otherwise.
	 */
    public boolean contains(Comparable value)
	{
		if (entryNumber ==0 || value ==null)
		{
			return false;
		}

		if (value instanceof MinComparableValue){
			return true;
		}
         return rightValue.compareTo(value) >=0;
	}
	
   
}
