package com.jofti.btree;

import java.io.Serializable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.core.IStoreKey;
import com.jofti.core.IStoreManager;
import com.jofti.exception.JoftiException;
import com.jofti.store.Page;
import com.jofti.store.StoreWrapper;

/**
 * <p>
 * A version of a {@link LeafNode} that pages its entries to and from disk. The entries are managed in an {@link IPage} 
 * which has operations performed upon it. Storage and retrieval is performed by an {@link IStoreManager}. This object uses the IPage as
 * its transfer object.
 * </p>
 * <p>
 * The Node does not keep a reference to the IPage inbetween operations and so a retrieval from the {@link IStoreManager} is required 
 * each time. The Manager is free to cache these objects in order to make this operation more performant. 
 * </p>
 * @author steve Woodcock
 * @version 1.36
 */
public class PagedLeafNode extends AbstractLeafNode implements Leaf,
        Serializable
{

    /**
     * 
     */
    private static final long serialVersionUID = -5295978203948185278L;

    Page                    pageData = null;

    private static Log      log      = LogFactory.getLog(PagedLeafNode.class);

    transient IStoreManager manager  = null;

    IStoreKey               key      = null;

    private boolean         dirty    = false;

    public PagedLeafNode(IStoreManager manager, IStoreKey key)
    {

        this.manager = manager;
        this.key = key;

    }

    /* (non-Javadoc)
     * @see com.jofti.btree.INode#setEntries(java.lang.Object[])
     */
    public void setEntries(Object[] temp)
    {

        // not used in paged node

    }

    /**
     * Sets this node's page entries to be the IPage passed in.
     * @param temp
     */
    public void setPageEntries(IPage temp)
    {

        try {
            dirty = true;

            key.setNumber(entryNumber);
            setPage(temp);

        } catch (Throwable e) {

            throw new RuntimeException(e);

        }

    }

    /**
     * Removes all the entries in the node and propogate this to the underlying {@link IStoreManager}.
     */
    protected void removeEntries()
    {

        try {

            manager.remove(key, null);
            dirty = false;
        } catch (JoftiException e) {

            log.error("Unable to remove for node", e);

        }

    }

    protected void setPage(IPage page) throws JoftiException
    {
        if (dirty) {

            key.setNumber(entryNumber);
            manager.store(key, page);
            dirty = false;
            key.setNewKey(false);
        }
    }

    protected void removePage(IPage page)
    {

        key.setNumber(entryNumber);

        try {
            manager.remove(key, page);
        } catch (Throwable t) {
            System.out.println("unable to remove " + page);
        }
        dirty = false;
        key.setNewKey(false);
    }

    protected IPage getPage()
    {

        StoreWrapper wrapper = null;
        try {
            wrapper = manager.retrieve(key);

        } catch (Throwable e) {
            while (e.getCause() != null) {
                System.err.println(e);
                e = e.getCause();
            }

            throw new RuntimeException(e);
        }
        key = wrapper.key;

        return wrapper.page;

    }

    public LeafNodeEntry getEntry(Comparable value)
    {
        if (entryNumber == 0) {
            return null;
        }

        // look through list and see if we have a match
        IPage page = getPage();
        LeafNodeEntry entry = indexedBinaryRetrieve(page, value);
        manager.releasePage(key, page);
        return entry;

    }

    protected Object[] realGetEntries()
    {

        if (dirty) {
            // we have a read before we have re-written - this is a problem
            // and should not happen with the locking
            log.error("Dirty node retrieval - Unable to get entries for node");
            return null;
        }
        Object[] entries = null;

        try {

            StoreWrapper wrapper = manager.retrieve(key);
            key = wrapper.key;
            // entries = wrapper.entries;
            if (entries == null) {
                throw new JoftiException("null returned for entries");
            }
            if (entryNumber != 0 && entries[entryNumber - 1] == null) {
                int tempCount = 0;
                for (int i = entryNumber - 1; i >= 0; i--) {
                    if (entries[i] != null) {
                        tempCount = i;
                        break;
                    }
                }
                throw new JoftiException("expected " + entryNumber + " got "
                        + tempCount + " for " + key);
            }

        } catch (JoftiException e) {

            log.error("Unable to get entries for node", e);

        }

        return entries;

    }

   

    public boolean equals(Object obj)
    {
        try {
            PagedLeafNode temp = (PagedLeafNode) obj;
            return key.equals(temp.key);
        } catch (Exception e) {
            // intentional
        }
        return false;
    }

    public int hashCode()
    {
        return key.hashCode();
    }

    protected LeafNodeEntry indexedBinaryRetrieve(IPage page, Object obj)
    {
        int i = 0;
        int size = entryNumber;
        for (int j = size - 1; i <= j;) {
            int k = i + j >> 1;
            LeafNodeEntry obj1 = page.getEntry(k);


            int l = obj1.getValue().compareTo(obj);
            if (l < 0)
                i = k + 1;
            else if (l > 0)
                j = k - 1;
            else
                return obj1;
        }

        return null;
    }

    protected int indexedBinaryLocate(IPage page, Object obj)
    {

        int low = 0;
        int high = entryNumber - 1;

        LeafNodeEntry entry = null;
        while (low <= high) {
            int mid = (low + high) >> 1;

            entry = page.getEntry(mid);

            int cmp = entry.compareTo(obj);

            if (cmp < 0)
                low = mid + 1;
            else if (cmp > 0)
                high = mid - 1;
            else
                return mid; // key found
        }
        return -(low + 1); // key not found

    }

    public boolean deleteEntry(NodeEntry entry)
    {
        resetNextNode();
        IPage page = getPage();
        if (entry == null || entry.getValue() == null) {
            return false;
        }
        if (entryNumber == 0) {
            return false;
        } else {
            int result = indexedBinaryLocate(page, entry);
            if (result >= 0) {
                LeafNodeEntry listEntry = page.getEntry(result);

                listEntry
                        .removeAllIds(((LeafNodeEntry) entry).getUniqueIdSet());

                if (listEntry.getUniqueIdSize() == 0) {
 

                    page.removeEntry(result);
                    // Let gc do its work
                    entryNumber--;
                    // update the right value if we need to
                    if (entryNumber == 0) {
                        rightValue = BTree.MIN_RIGHT_VALUE;
                        removePage(page);

                    } else {

                        LeafNodeEntry tempEntry = page
                                .getEntry(entryNumber - 1);
 
                        rightValue = tempEntry.getValue();
                        dirty = true;
                        try {
                            setPage(page);
                        } catch (JoftiException e) {
                            throw new RuntimeException(e);
                        }
                    }

                    return true;
                } else {
                    try {

                        page.updateEntry(result, listEntry);
                        dirty = true;
                        setPage(page);
                    } catch (JoftiException e) {
                        throw new RuntimeException("Unable to remove entry ", e);
                    }
                }
            }
        }

        return false;

    }

    public Object[] insertEntry(NodeEntry entry) throws JoftiException
    {

        LeafNodeEntry leafEntry = (LeafNodeEntry) entry;
        IPage page = getPage();

        if (entry == null || entry.getValue() == null) {
            throw new JoftiException("Null node entry cannot be inserted");
        }

        resetNextNode();
        if (entryNumber == 0) {
            entryNumber++;
            rightValue = entry.getValue();
            dirty = true;

            page.setEntry(0, leafEntry);

            setPage(page);

            return null;
        } else if (rightValue.compareTo(entry.getValue()) >= 0) {

            int result = indexedBinaryLocate(page, entry);

            if (result < 0) {
                // we need to insert it
                int index = (-result) - 1;

                page.setEntry(index, leafEntry);
                entryNumber++;
                dirty = true;
                setPage(page);
                return null;
            } else {
                LeafNodeEntry listEntry = page.getEntry(result);

                listEntry.addUniqueIdSet(((LeafNodeEntry) entry)
                        .getUniqueIdSet());

                page.updateEntry(result, listEntry);

                dirty = true;
                setPage(page);
                return null;
            }

        }
        throw new JoftiException("unable to insert " + entry.getValue()
                + "as is bigger than current right value");

    }

    public Object[] getEntries()
    {
        Object[] objArr = null;
        try {
            IPage page = getPage();
            objArr = new Object[entryNumber];
            for (int i = 0; i < entryNumber; i++) {
                LeafNodeEntry entry = page.getEntry(i);

                objArr[i] = entry;

            }
        } catch (Throwable t) {
            log.fatal("Error getting entries from node " + key + t);

        }
        return objArr;

    }

    public boolean contains(Comparable value)
    {
        if (rightValue == null) {
            return false;
        }

        return rightValue.compareTo(value) >= 0;

    }

    public Node splitNode(Object[] entries) throws JoftiException
    {
        // first insert the entry

        IPage page = getPage();

        EntrySplitWrapper[] entriesList = manager.split(page, entryNumber);
        //			
        Node newNode = NodeFactory.getInstance(manager).createLeafNode();
        //
        Comparable currentRight = rightValue;
        //
        //			
        EntrySplitWrapper newEntries = entriesList[1];
        //            
        newNode.entryNumber = newEntries.size;

        newNode.setRightValue(currentRight);
        ((PagedLeafNode) newNode).setPageEntries((IPage) newEntries.entries);
        newNode.setLinkNode(getLinkNode());

        //			
        ////			replace values in current node
        //
        EntrySplitWrapper replacements = (EntrySplitWrapper) entriesList[0];
        //			
        //			
        //			
        entryNumber = replacements.size;
        IPage replacementPage = (IPage) replacements.entries;


        setRightValue(replacementPage.getEntry(entryNumber - 1).getValue());

        setPageEntries(replacementPage);
        return newNode;
    }

    public boolean isLeaf()
    {
        return true;
    }

    public synchronized IStoreKey getKey()
    {
        return key;
    }


    public synchronized void setKey(IStoreKey key)
    {
        this.key = key;
    }


}
