/*
 * Created on 05-May-2005
 *
 */
package com.jofti.cache;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.IndexQuery;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ClassFieldMethods;
import com.jofti.parser.ParserManager;
import com.jofti.util.CompositeComparator;
import com.jofti.util.ObjectProcedureAdapter;
import com.jofti.util.OpenHashMap;
import com.jofti.util.ValueTreeMap;

/**

 *
 * The base class for the adapters. The locking is used to allow a reasonable amount of concurrency in the 
 * cache. There are 31 locks that are obtained as a mod 31 of the Object's hashcode. The premise is that a lock can be obtained on methods that modify or access the cache. The objective is to prevent 
 * concurrent modification for identical keys . However, it is also desirable to allow concurrent usage of the 
 * cache and the index as both should be multi-threaded. Although that is reliant on the implementation.<p>
 
 * @author Steve Woodcock<br>
 *
 */
public abstract class BaseAdaptor implements IBaseAdaptor {

	// array of locks
	protected Object[] cacheLocks = new Object[31];

	int updates =0;
	int queries =0;
	// is check mutable enabled?
	protected boolean checkMutable = false;

	protected String MUTABLE_VALUES = "mutable-values";
	
    protected InternalIndex index;

	// lock that provides two groups of exclusive readers
	private final QueryUpdateLock queryUpdateLock = new QueryUpdateLock();

	private static Log log = LogFactory.getLog(BaseAdaptor.class);

	// initialise the object locks
	{
		for (int i = 0; i < 31; i++) {
			cacheLocks[i] = new Object();
		}
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#getCacheLock(java.lang.Object)
	 */
	public Object getCacheLock(Object key) {
		int temp = (key.hashCode() % 31);
		return cacheLocks[temp < 0 ? -temp : temp];
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see com.jofti.api.IndexCache#query(com.jofti.api.IndexQuery)
	 */
	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#query(com.jofti.api.IndexQuery)
	 */
	public Map query(IndexQuery query) throws JoftiException {
		
		Map temp = null;
		
		query = processQuery(query, index.getParserManager());
		
		
		acquireQueryLock();
		try {
		
		temp=index.query(query);
		} finally {
			releaseQueryLock();
		}
		return getCacheValues(temp, (IParsedQuery)query,index.getIntrospector());
        
		

	}
	public IndexQuery processQuery(IndexQuery query, ParserManager manager) throws JoftiException {
		QueryType type = ((QueryId)query).getQueryType();
	   	 if ( type!= QueryType.PARSED_QUERY) {
	           query = (IndexQuery) manager.parseQuery( query);
	   	 }
      
           if (((INameSpaceAware) query).getNameSpace() != null) {
               throw new JoftiException(
                       "namespace not allowed in non name-space cache query");
           }
		return query;
	}
	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#getCacheValues(java.util.Map, com.jofti.core.IParsedQuery, com.jofti.introspect.ClassIntrospector)
	 */
	public Map getCacheValues(Map col, final IParsedQuery query,
			final ClassIntrospector introspector) throws JoftiException {

		
		final Map returnClasses = query.getResultFieldsMap();
		final CompositeComparator comp = query.getOrderingComparator();
		final int maxResults = query.getMaxResults();
		final int startEntry = query.getFirstResult();
		
		Map interim = null;
		if (comp ==null || comp.getSize() ==0){
			interim =new HashMap(col.size() + 2, 1.00f);
		}else{
			interim = new ValueTreeMap(comp);
		}
		
		// return map
		final Map temp = interim;
		
		

		// the results should always be an OpenHashMap
		OpenHashMap originalMap = (OpenHashMap) col;

		
		// populate the return Map
		originalMap.forEachPair(new ObjectProcedureAdapter() {
			
			public boolean apply(Object key, Object value) {
				
				//this will be an alias or null or it is not in the return set and we can ignore it
				if (value == null || returnClasses ==null || returnClasses.containsKey(value)) {
					// see if we have the class in the return map

					// strip the key of wrapper if it is noncomparable
					key = stripKey(key);

					Object result = getCacheValue(key);

					if (result == null) {
						warnOutOfDate(key);
					} else {
						// get the methods to be operated on the value
						if (checkMutable) {
							result = checkMutable(key, result);
							// mutability check failed
							if (result == null) {
								if (log.isDebugEnabled()) {
									log
											.debug("Object under key:"
													+ key
													+ " has changed in cache since it was indexed");
								}
								return true;
							}
						}
						
						
						if (returnClasses != null) {
							// we need to get the value from result if we are using fields
							ClassFieldMethods fieldSet = (ClassFieldMethods) returnClasses
									.get(value);

							Map tempMap = fieldSet != null ? fieldSet
									.getFieldMap() : null;

							if (tempMap != null && tempMap.size() > 0) {
								// get the fields and apply the values to them
				
									Map resultMap = new LinkedHashMap();
				
									resultMap = introspector.getResultValuesFromObject(resultMap,result,tempMap);
									
									// no values means no mapping in results
									if (resultMap.size()>0){
										Object[] res = new Object[resultMap.size()];
										Iterator it = resultMap.entrySet().iterator();
										for (int i = 0;i<resultMap.size();i++){
											Map.Entry entry = (Map.Entry)it.next();
											res[i]= entry.getValue();
										}
										temp.put(key,res);
									}
									
							} else {
								// there are no fields just return the value
								temp.put(key, result);
							}
						} else {
							temp.put(key, result);
						}
					}

				}
				return true;
			}
		});

		if (startEntry <0){
			throw new IllegalArgumentException("startResult cannot be less than 0:"+startEntry); 
		}
		if (maxResults <0){
			throw new IllegalArgumentException("maxResults cannot be less than 0:"+maxResults); 
		}
		// now limit result size
		if (maxResults >0 || startEntry >0){
			return limitResults(temp, startEntry, maxResults);
		}
				
		return temp;
	}
	
	public void warnOutOfDate(Object key){
		if (log.isWarnEnabled()) {
			log
					.warn("Index and cache have become out of date for key "
							+ key);
		}
	}

	public Map limitResults(Map temp, int startResult, int maxResults){
		
		if (maxResults ==0){
			maxResults = temp.size();
		}
		
		if (temp.size() <= (maxResults - startResult)){
			return temp;
		}
		
		if (temp instanceof ValueTreeMap) {
			return ((ValueTreeMap)temp).getSubMap(startResult, maxResults);
			
		}else{

		Iterator it = temp.entrySet().iterator();
		
		
		int count =0;
			while (it.hasNext()){
				it.next();
				if (count  < startResult || count >= maxResults +startResult){
					it.remove();
				}
				count++;
			}
			return temp;
		}
		
		
		
	}
	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#acquireUpdateLock()
	 */
	public void acquireUpdateLock() throws JoftiException {
		try {

			queryUpdateLock.updateLock().acquire();
		} catch (Exception e) {
			throw new JoftiException(e);
		}

	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#releaseUpdateLock()
	 */
	public void releaseUpdateLock() {

		queryUpdateLock.updateLock().release();
	
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#acquireQueryLock()
	 */
	public void acquireQueryLock() throws JoftiException {
		try {
			queryUpdateLock.queryLock().acquire();
			} catch (Exception e) {
			throw new JoftiException(e);
		}
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#releaseQueryLock()
	 */
	public void releaseQueryLock() {

		queryUpdateLock.queryLock().release();
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#decorateKey(java.lang.Object)
	 */
	public Object decorateKey(Object key) {
		if (key != null && !(key instanceof Comparable)) {
			key = new NonComparableKeyWrapper(key);
		}
		return key;
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#stripKey(java.lang.Object)
	 */
	public Object stripKey(Object key) {
		if (key instanceof NonComparableKeyWrapper) {
			key = ((NonComparableKeyWrapper) key).getKey();
		}
		return key;
	}

	/**
	 * Returns the value from the Cache for the key. Each adpter is responsible for the specific implementation of this, as 
	 * the Cache retrieval mechanism will probably be different for each Cache.
	 * @param key
	 * @return
	 */
	protected abstract Object getCacheValue(Object key);

	/**
	 * Checks the multability of a Cache value retrieved from the Cache instance in the adapter. Each 
	 * adapter is responsible for providing the implementation.
	 * @param key
	 * @param result
	 * @return The value from the Cache or Null if the value has changed after it was indexed
	 */
	protected Object checkMutable(Object key, Object result) {
		try {
			if (log.isDebugEnabled()) {
				log.debug("Checking mutability ");
			}
			// first parse the object - again
			Map cacheObjectValues = index.getIntrospector().getAttributeValues(
					result);

			Map indexObjectValues = index.getAttributesByKey(decorateKey(key));
			if (cacheObjectValues.equals(indexObjectValues)) {
				return result;
			} else {
				if (log.isDebugEnabled()) {
					log.debug("Object under Key " + key
							+ " -  attributes changed without re-insert");
				}
			}
			// then get the values for the key

		} catch (JoftiException e) {
			log.warn("Error checking mutability", e);
		}

		return null;
	}
	

	/* (non-Javadoc)
	 * @see com.jofti.cache.IBaseAdaptor#getIndex()
	 */
	public abstract InternalIndex getIndex();
}