package com.jofti.cache;

import java.util.Map;

import com.jofti.api.IndexQuery;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ParserManager;

public interface IBaseAdaptor {

	/**
	 * Gets a cache lock based on the mod of the hashcode of the key
	 * @param key
	 * @return - a lock Object
	 */
	public abstract Object getCacheLock(Object key);

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.jofti.api.IndexCache#query(com.jofti.api.IndexQuery)
	 */
	public abstract Map query(IndexQuery query) throws JoftiException;

	/**
	 * Takes a map containing the key values returned from the Index for the Query. The keys 
	 * are the primary keys in the Cache.<p>
	 * This is the default implementation that loops through the returned keys and extracts the appropriate value 
	 * from the Cache. This method is also responsible for checking if the value has changed since it was last 
	 * indexed or if the value is no longer in the Cache excluding it from the returned results.<p>
	 * 
	 * @param col A Map of result keys
	 * @return A Map of key/value results.
	 * @throws JoftiException Thrown if the retrieval from the cache throws an exception
	 */
	public abstract Map getCacheValues(Map col, final IParsedQuery query,
			final ClassIntrospector introspector) throws JoftiException;

	/**
	 * Attempts to acquire an update lock for the cache. Multiple updates can run 
	 * in parallel as the Index itself is threadsafe and the assumption is that the Cache is also. 
	 * <p>
	 * However, queries and updates cannot run at the same time to prevent situations where if multiple
	 * nodes in the index are updated by one update then a query which runs at the same time may well misreport 
	 * a match that should be returned if they were run in sequence.</p>
	 *  
	 * @throws JoftiException
	 */
	public abstract void acquireUpdateLock() throws JoftiException;

	/**
	 * Releases an update lock.
	 */
	public abstract void releaseUpdateLock();

	/**
	 * Attempts to acquire a query lock for the cache. Multiple queries can run 
	 * in parallel as the Index itself is threadsafe and the assumption is that the Cache is also. 
	 * <p>
	 * However, queries and updates cannot run at the same time to prevent situations where if multiple
	 * nodes in the index are updated by one update then a query which runs at the same time may well misreport 
	 * a match that should be returned if they were run in sequence.</p>
	 *  
	 * @throws JoftiException
	 */
	public abstract void acquireQueryLock() throws JoftiException;

	/**
	 * Releases a query lock.
	 */
	public abstract void releaseQueryLock();

	/**
	 * A utility method that decorates a Cache key with a comparable wrapper if the key itself is not Comparable. This wrapper 
	 * is not propogated into the Cache, but is used to wrap the key when placed in the Index.
	 * 
	 * @param key
	 * @return Either an instance of a NonComparableKeyWrapper if the key is not Comparable or the original key object.
	 */
	public abstract Object decorateKey(Object key);

	/**
	 * A utility method that strips a decorated Cache key of its comparable wrapper if it has been previously decorated by the decorate method. 
	 * 
	 * @param key
	 * @return The object wrapped in a  NonComparableKeyWrapper if wrapped or the original object if not.
	 */
	public abstract Object stripKey(Object key);

	/**
	 * Returns the Index instance from the adapter.
	 * 
	 * @return the internal index implementation
	 */
	public abstract InternalIndex getIndex();
	
	/**
	 * Normalises the query format for convenience queries or processes one the unparsed text query formats 
	 * into its parsed form.
	 * 
	 * @param query an unparsed query
	 * @param manager the Parser Manager that stores the mappings for the query
	 * 
	 * @return the parsed query
	 */
	public IndexQuery processQuery(IndexQuery query, ParserManager manager) throws JoftiException;
	
	
	/**
	 * Takes a fully populated result set and applies any of the limiting steps defined in the 
	 * maxResults or startResults. 0 for both values is considered a no-op. Negative values throw a runtime exception. 
	 * a Start Value larger than the result original size will return no results, a maxResults large than the result size 
	 * is reset to temp.size().
	 * 
	 * @param temp the original result set
	 * @param startResult an int defining at which record the record set should begin
	 * @param maxResults an int defining the maximum number of records to return.
	 * 
	 * @return the result set
	 */
	public Map limitResults(Map temp, int startResult, int maxResults);
		

}