/*
 * Created on 19-Feb-2005
 *

 */
package com.jofti.cache.adapter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import com.jofti.api.IndexQuery;
import com.jofti.btree.BTree;
import com.jofti.core.GenericIndexFactory;
import com.jofti.core.InternalIndex;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.tree.NameSpacedTreeIndex;
import com.jofti.tree.TreeIndex;



/**
 *
 *
 * Used by the JBossCacheAdapter to provide Transaction isolation for the 
 * changes to the index. In effect each transaction acquires its own mini-index which stores the 
 * changes to the larger main index. These changes are then applied on a commit (after the cache changes have committed).
 * 
 *  @author Steve Woodcock (steve@jofti.com)<p>
 */

public class ChangeRecorder  {

	public final static Object DUMMY_VALUE = new Object();
	
	protected InternalIndex index =null;
	private Set removedKeys = new HashSet();
	private Map updatedMap = new HashMap();
	
	private ClassIntrospector parser = null;
	
	//stores values that were originally in the map but have been removed in the transaction
	private Map removedValues = new HashMap();


	public ChangeRecorder(ClassIntrospector parser){
		this.parser = parser;
	}
	
	
	/**
	 * Initialises a new NameSpacedtreeIndex so local changes can be stored.
	 * @param props - properties to strat index with.
	 * @throws JoftiException
	 */
	public void init(Properties props, String indexType) throws JoftiException{
		index = GenericIndexFactory.getInstance().createIndex(indexType, parser, props, "localIndex");
		
		
	}
	
	/**
	 * @return A collection of all the local tree values in the recorder
	 * @throws JoftiException
	 */
	public Collection getAllTreeValues() throws JoftiException {
		
		return ((NameSpacedTreeIndex)index).getAllKeyEntriesFromTree();
		
	}
	
	/**
	 * Removes an object from the local index. This method also adds the removed key and
	 * values to the removedValues map so when committed a full list of removed objects can be
	 * identified.
	 * @param key
	 * @param value
	 * @param parser
	 * @throws JoftiException
	 */
	public synchronized void remove(Object key,Object value, ClassIntrospector parser) throws JoftiException{
		
		List temp = (List)removedValues.get(key);
		if (temp == null){
			temp = new ArrayList();
			
		}
		temp.add(value);
		removedValues.put(key, temp);
		index.remove((Comparable)key,value);
		
			


	}
	
	/**
     * Removes an object mapping from the change recorder.
     * <p>
	 * @param key
	 * @param parser
	 * @throws JoftiException
	 */
	public synchronized void remove(Object key,ClassIntrospector parser )throws JoftiException {
		

		removedKeys.add(key);
		index.removeByKey((Comparable)key);
	


	}
	
	/**
     * Adds a value to the ChangeRecorder.<p>
	 * @param key
	 * @param value
	 * @param parser
	 * @throws JoftiException
	 */
	public synchronized void add(Object key, Object value,ClassIntrospector parser) throws JoftiException{
		
	
		index.insert((Comparable)key, value);
		
		
	}

	/**
     * Checks the local index to see if it contains the key.<p>
	 * @param key
	 * @param parser
	 * @return a boolean indicating if the change recorder contains the key
	 * @throws JoftiException
	 */
	public boolean contains(Object key, ClassIntrospector parser) throws JoftiException{
		
		return index.contains((Comparable)key);
		
	}
	
	/**
     * Notifies the Change Recorder that an attribute has been updated in the local index.<p>
	 * @param key
	 * @param oldValue
	 * @param value
	 * @param parser
	 * @throws JoftiException
	 */
	public synchronized void update(Object key, Object oldValue,Object value, ClassIntrospector parser) throws JoftiException{
				
		// remove if we already have set it 
		index.removeByKey((Comparable)key);

		// add to the updated list
		List temp = (List)updatedMap.get(key);
		if (temp == null){
			temp = new ArrayList();
			
		}
		temp.add(oldValue);
		updatedMap.put(key, temp);
		
		// insert the new value
		index.insert((Comparable)key,value);
	
	}
		

	Map getRemovedMap(){
		return removedValues;
	}
		
	
	Map getUpdatedMap(){
		return updatedMap;
	}
	

	/**
	 * @return Returns the removedKeys.
	 */
	public Set getRemovedKeys() {
		return removedKeys;
	}
	
	/**
      * Executes a query against the local index.
	 * @param query
	 * @return A Map containing the results of the query
	 * @throws JoftiException
	 */
	public Map query(IndexQuery query) throws JoftiException{
	 	return index.query(query);
	 }
}
