package com.jofti.cache.adapter;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;



import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.CacheAccessor;
import com.jofti.api.IndexQuery;
import com.jofti.cache.CacheListenerAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.adapter.listener.CoherenceEventListener;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.util.CompositeComparator;
import com.tangosol.net.NamedCache;
import com.tangosol.util.MapEvent;

/**

 * 
 * The adapter implementation specific to Coherence.</p>
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of Coherence.</p>
 * 
 * The Listener adapter is for use as a listener to Coherence and does not provide the get, set, remove methods 
 * that the wrapper adapter provides. The addition, removal and getting of values from the Cache must be 
 * done on the Cache implementation directly. This is the preferred way of using an already existing Cache.</p>
 * 
 * The start up of the adapter will also try and index any elements already in
 * the cache.<P>
 * 
 * @author Steve Woodcock
 *         <p>
 * @version 1.0
 * @since 1.1
 */
public class CoherenceListenerAdapter extends BaseAdaptor implements CacheAccessor, CacheListenerAdapter
{


    private static Log                  log = LogFactory
                                                    .getLog(CoherenceListenerAdapter.class);


    private String                      name;

    public InternalIndex               index;

    private CoherenceEventListener eventListener = null;
    
    
    private NamedCache cache =null;


    public CoherenceListenerAdapter()
    {
    }

    public CoherenceListenerAdapter(Object cache)
    {
        this.cache = (NamedCache) cache;

    }
    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException
    {
        if (properties != null) {
            String key = null;
            for (Iterator it = properties.keySet().iterator(); it.hasNext();) {
                key = (String) it.next();
                if (MUTABLE_VALUES.equalsIgnoreCase(key)) {
                    checkMutable = Boolean.valueOf(
                            properties.getProperty(key)).booleanValue();
                    if (log.isInfoEnabled()) {
                        log.info("Mutability checking is set to "
                                + checkMutable);
                    }
                }
                if ("file".equalsIgnoreCase(key)) {
                    log.warn("Listener adapters cannot be used to start up a Cache - check docs for Adapter Type");
                    throw new JoftiException("Listener adapters cannot be used to start up a Cache -  check docs for Adapter Type");
                }
            }
        }

     
        if (cache == null) {
            throw new JoftiException("Cache cannot be null for listener adapter");
        } else {
        	    log.info("Index started using cache "+ cache.getCacheName());
        }
        eventListener = new CoherenceEventListener(this);
    
    }

    
 
   

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public synchronized void destroy() throws JoftiException
    {
        try {
            cache.removeMapListener(eventListener);
           if (index != null){
                index.removeAll();
           }
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#getName()
     */
    public String getName()
    {
        return name;
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    public String toString()
    {
        return "CoherenceCache(" + getName() + ')';
    }

  


    /*
     * (non-Javadoc)
     * 
     * @see com.jofticache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
     */
    public void setInternalIndex(InternalIndex index)
    {
        this.index = index;

    }

    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#query(com.jofti.api.IndexQuery)
     */
    public Map query(IndexQuery query) throws JoftiException
    {
    	
    	Map temp = null;
    	
    	query = processQuery(query,index.getParserManager());
       
    	acquireQueryLock();
		try {
        
        temp=index.query(query);
    	} finally {
			releaseQueryLock();
		}
    	return getCacheValues(temp, (IParsedQuery)query,index.getIntrospector());

    }

	

    /* (non-Javadoc)
     * @see com.jofti.cache.CacheLocking#checkMutable(java.lang.Object, java.lang.Object)
     */
    protected Object checkMutable(Object key, Object result)
    {
        try {
            // first parse the object - again
            Map cacheObjectValues = index.getIntrospector().getAttributeValues(
                    result);

            Map indexObjectValues = index.getAttributesByKey(decorateKey( key));
            
            if (cacheObjectValues.equals(indexObjectValues)) {
                return result;
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Object under Key " + key
                            + " -  attributes changed without re-insert");
                }
            }
            // then get the values for the key

        } catch (JoftiException e) {
            log.warn("Error checking mutability", e);
        }

        return null;
    }

	/**
	 * @param cache
	 */
	public void setCacheImpl(Object cache){
		CoherenceListenerAdapter.this.cache = (NamedCache)cache;
		if (this.cache != null){
			this.cache.addMapListener(eventListener);
		}
		
	}
	
  	protected Object getCacheValue(Object key) {
		
		return cache.get(key);
		
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.LifeCycleAdapter#start()
	 */
	public void start() throws JoftiException {
		if (cache != null){
			cache.addMapListener(eventListener);

	            loadInitialValues(cache);
	        
	          
	        
		}
		
		
	}

	  private void loadInitialValues(NamedCache cache)
			throws  JoftiException {
		Set entries = cache.entrySet();
		 int size = entries.size();
	        Iterator it = entries.iterator();
	    for (int i=0;i<size;i++) {
	        Map.Entry entry = (Map.Entry)it.next();

			eventListener.entryInserted(new MapEvent(cache,1,entry.getKey(),entry.getValue(),entry.getValue()));

		}
	}


	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		
		return index;
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.CacheAccessor#getCacheImpl()
	 */
	public Object getCacheImpl() {
		
		return cache;
	}

	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
	
}