package com.jofti.cache.adapter;

import java.io.IOException;
import java.io.Serializable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.IndexQuery;
import com.jofti.cache.CacheAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.adapter.listener.EhEventListener;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.util.CompositeComparator;

/**

 * 
 * The adapter implementation specific to EHCache.</p>
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of EHCache. This adapter is for versions 
 * of EHCache 1.2 and upwards. For earlier versions you <b>must</b> use the EhCachePre1_2Adapter.</p>
 * 
 * 
 * The start up of the adapter will also try and index any elements already in
 * the cache.<P>
 * 
 * @author Steve Woodcock
 *         <p>
 * @version 1.0
 * @since 1.1
 */
public class EhCacheAdapter extends BaseAdaptor implements CacheAdapter
{

    private net.sf.ehcache.CacheManager manager;

    private static Log                  log = LogFactory
                                                    .getLog(EhCacheAdapter.class);

    private net.sf.ehcache.Cache        cache;

    private String                      name;

    
    private EhEventListener eventListener =null;

    public EhCacheAdapter()
    {
    }

    public EhCacheAdapter(Object cache)
    {
        this.cache = (net.sf.ehcache.Cache) cache;

    }

    public void setCacheImpl(Object cache)
    {
        this.cache = (net.sf.ehcache.Cache) cache;

    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#get(java.lang.Object)
     */
    public Object get(Object key)
    {
    
        try {
            if (key != null) {
                // do the cast early - seems to help the performance slightly -
                // otherwise
                // we have multiple casts in this method
                Comparable tempKey = (Comparable) key;
                // get a lock for the key that we are trying to add
                // we do not care about other parts of the cache or the index
                synchronized (getCacheLock(key)) {

                    // get the element from the cache if it exists
                    Element element = cache.get((Serializable) tempKey);
                    // if it is null either it has expired or it was never there
                    if (element == null) {
                        return null;
                    } else {
                        // we should not need to do this as there should be no
                        // way an entry can just appear without going through
                        // this classe

                        return element.getValue();
                    }
                }

            } else {
                return null;
            }
        } catch (net.sf.ehcache.CacheException e) {
            log.warn("Unable to retrieve value from cache", e);
 
        } 
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#put(java.lang.Object, java.lang.Object)
     */
    public void put(Object key, Object value) throws JoftiException
    {
    	
    	try{
            synchronized (getCacheLock(key)) {

                // we have to remove a potentially expired result

                Element element = new Element((Serializable) key,
                        (Serializable) value);
                cache.put(element);

            }
        } catch (IllegalArgumentException e) {
            throw new JoftiException(e);
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } 

    }

    /**
     * Removes the element which matches the key.
     * <p>
     * If no element matches, nothing is removed and no Exception is thrown.
     * 
     * @param key
     *            the key of the element to remove
     * @throws CacheException
     */
    public void remove(Object key) throws JoftiException
    {
    

        try {

            synchronized (getCacheLock(key)) {
                // remove if we need to based on the key - as it may be expired
               

                cache.remove((Serializable) key);
            }

        } catch (ClassCastException e) {
            throw new JoftiException(e);
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } 
    }

    /**
     * Remove all elements in the cache, but leave the cache in a useable state.
     * 
     * @throws CacheException
     */
    public synchronized void removeAll() throws JoftiException
    {
        try {

            cache.removeAll();
            index.removeAll();

        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } catch (Exception e) {
            throw new JoftiException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException
    {
        try {
            String cacheConfigFile = null;
            if (properties != null) {
                String key = null;
                for (Iterator it = properties.keySet().iterator(); it.hasNext();) {
                    key = (String) it.next();
                    if (MUTABLE_VALUES.equalsIgnoreCase(key)) {
                        checkMutable = Boolean.valueOf(
                                properties.getProperty(key)).booleanValue();
                        if (log.isInfoEnabled()) {
                            log.info("Mutability checking is set to "
                                    + checkMutable);
                        }
                    }
                    if ("file".equalsIgnoreCase(key)) {
                        cacheConfigFile = properties.getProperty(key);
                    }
                }
            }
            log.debug("looking up CacheManager for EHCache");
	            if (manager == null) {
	            	log.debug("no CacheManager found obtaining one via static CacheManager methods");	
	                if (cacheConfigFile != null) {
	                	log.debug("obtaining CacheManager using config file "+ cacheConfigFile);
	                    manager = net.sf.ehcache.CacheManager
	                            .create(cacheConfigFile);
	                } else {
	                   	log.debug("obtaining CacheManager using no config file ");
	                    manager = CacheManager.create();
	                }
	            }
	            log.debug("finished looking up CacheManager for EHCache");
         
            if (cache == null) {
            	log.debug("EHCache impl is null - initialising cache");
                initCache();
            } else {
                // try and add it to the cache
            	log.debug("EHCache impl is not null - checking if exists in CacheManager");
                if (!(manager.cacheExists(cache.getName()))) {
                	log
                    .info("Cache "
                            + cache.getName()
                            + "does  not exist in CacheManager - adding new cache");
                    manager.addCache(cache);
                } else {
                    log
                            .info("Cache "
                                    + cache.getName()
                                    + "already exists in CacheManager - not adding new cache");
                }
            }
            eventListener = new EhEventListener(this);
         } catch (net.sf.ehcache.CacheException e) {
            throw new JoftiException(e);
        }

    }

    private void initCache() throws JoftiException
    {
        if (cache == null) {
        	if (manager == null){
        		throw new JoftiException("No Cache Manager specified unable to initialise cache "+name);
        	}
            cache = manager.getCache(name);
            if (cache == null) {
                log.warn("Unable to find cache named '" + name
                        + "' in EHCacheManager");
                String[] names = manager.getCacheNames();
                log.warn("Available cache names are:");
                for (int i = 0; i < names.length; i++) {
                    log.warn("IndexCache:" + names[i]);
                }
                throw new JoftiException("Unable to find cache named '" + name
                        + "' in EHCacheManager");
            }
        }
  
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#start()
     */
    public synchronized void start() throws JoftiException
    {
    	if (cache != null){
			((Cache)cache).getCacheEventNotificationService().registerListener(eventListener);
			
		}
        try {
            loadInitialValues(cache);
        } catch (CacheException ce) {
            throw new JoftiException(ce);
        }
    }

    private void loadInitialValues(net.sf.ehcache.Cache cache)
            throws CacheException, JoftiException
    {
    	if (cache == null){
    		log.info("No initial values to load as Cache is null");
    		return;
    	}
        List keys = cache.getKeys();
        int size = keys.size();
        Iterator it = keys.iterator();
        for (int i=0;i<size;i++) {
            Object key = it.next();
            Element element = cache.get((Serializable) key);
            if (key instanceof Comparable) {
                Comparable tempKey = (Comparable) key;
                if (element != null && element.getValue() != null) {
                    // then we must remove from index
                    index.insert(tempKey, element.getValue());
                }
            } else {
                log.info("Ignoring value at key:" + key
                        + " as key is not Comparable");
            }
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public synchronized void destroy() throws JoftiException
    {
        try {
            if (manager != null) {
                manager.removeCache(name);
                manager.shutdown();
                index.removeAll();
            }
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#getName()
     */
    public String getName()
    {
        return name;
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    public String toString()
    {
        return "EHCache(" + getName() + ')';
    }


    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#getCacheImpl()
     */
    public Object getCacheImpl()
    {
        return cache;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofticache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
     */
    public void setInternalIndex(InternalIndex index)
    {
        this.index = index;

    }

  


  /* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getCacheValue(java.lang.Object)
	 */
	protected Object getCacheValue(Object key) {
		try{
			Element res=  cache.get((Serializable)key);
			if (res != null){
				return res.getValue();
			}else{
				log.warn("No Cache value found for indexed key "+key);
			}
		} catch (Throwable e){
			log.warn(e);
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		return index;
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#addQuery(java.lang.String, com.jofti.api.IndexQuery)
	 */
	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
}