package com.jofti.cache.adapter;

import java.io.IOException;
import java.io.Serializable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.sf.ehcache.CacheException;
import net.sf.ehcache.Element;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.IndexQuery;
import com.jofti.cache.CacheAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.util.CompositeComparator;

/**

 * 
 * The adapter implementation specific to EHCache.
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of EHCache (pre-1.2 versions).
 * 
 * The main drawback with ealier versions of EHCache is that it is not possible to find out when an
 * entry has been expired in the cache so there is a possibility that if you do
 * not access the cache entries the index will end up with extra keys that are
 * not present in the cache.
 * 
 * While not a huge problem, over time this could be an issue in an ongoing
 * situation. It is strongly recommended that you upgrade to version 1.2 and use the Listener or 
 * wrapper adapter.</p>
 * 
 * The start up of the adapter will also try and index any elements already in
 * the cache.<P>
 * 
 * @author Steve Woodcock
 *         <p>
 * @version 1.0
 *          <p>
 */
public class EhCachePre1_2Adapter extends BaseAdaptor implements CacheAdapter
{

    private net.sf.ehcache.CacheManager manager;

    private static Log                  log = LogFactory
                                                    .getLog(EhCachePre1_2Adapter.class);

    private net.sf.ehcache.Cache        cache;

    private String                      name;


    public EhCachePre1_2Adapter()
    {
    }

    public EhCachePre1_2Adapter(Object cache)
    {
        this.cache = (net.sf.ehcache.Cache) cache;

    }

    public void setCacheImpl(Object cache)
    {
        this.cache = (net.sf.ehcache.Cache) cache;

    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#get(java.lang.Object)
     */
    public Object get(Object key)
    {
    	key = decorateKey(key);
        try {
            if (key != null) {
                // do the cast early - seems to help the performance slightly -
                // otherwise
                // we have multiple casts in this method
                Comparable tempKey = (Comparable) key;
                // get a lock for the key that we are trying to add
                // we do not care about other parts of the cache or the index
                synchronized (getCacheLock(key)) {

                    // get the element from the cache if it exists
                    Element element = cache.get((Serializable) tempKey);
                    // if it is null either it has expired or it was never there
                    if (element == null) {

                        // we always do this as we can't tell if it has expired
                        // unless we have the full
                        // object - which we can't get because it has expired!
                        // this is crap - but there is little other option
                        // have raised a change request in ehcache
                        if (log.isDebugEnabled()) {
                            log
                                    .debug("unexpected null object retrieved for key "
                                            + key + " removing");
                        }
                        // check if it exists first (might always have been
                        // null) as otherwise we have to remove anyway

                        if (index.contains(tempKey)) {
                            index.removeByKey(tempKey);
                        }

                        return null;
                    } else {
                        // we should not need to do this as there should be no
                        // way an entry can just appear without going through
                        // this classe

                        return element.getValue();
                    }
                }

            } else {
                return null;
            }
        } catch (net.sf.ehcache.CacheException e) {
            log.warn("Unable to retrieve value from cache", e);
 
        } catch (JoftiException e) {
            log.warn("Unable to retrieve value from cache", e);
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#put(java.lang.Object, java.lang.Object)
     */
    public void put(Object key, Object value) throws JoftiException
    {
    	key = decorateKey(key);
    	acquireUpdateLock();
        try {
            Comparable tempKey = (Comparable) key;

            synchronized (getCacheLock(key)) {

                // we have to remove a potentially expired result
                if (index.contains(tempKey)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Object does exist so remove " + tempKey);
                    }

                    index.removeByKey(tempKey);
                }

                Element element = new Element((Serializable) tempKey,
                        (Serializable) value);
                cache.put(element);

                index.insert(tempKey, value);

            }
        } catch (IllegalArgumentException e) {
            throw new JoftiException(e);
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } finally {
			releaseUpdateLock();
		}

    }

    /**
     * Removes the element which matches the key.
     * <p>
     * If no element matches, nothing is removed and no Exception is thrown.
     * 
     * @param key
     *            the key of the element to remove
     * @throws CacheException
     */
    public void remove(Object key) throws JoftiException
    {
    	key = decorateKey(key);
    	
    	acquireUpdateLock();
        try {

            synchronized (getCacheLock(key)) {
                Comparable tempKey = (Comparable) key;
                // remove if we need to based on the key - as it may be expired
                // this synchronous as we have to remove these entries before
                // we can return
                if (index.contains(tempKey)) {
                    index.removeByKey(tempKey);
                }

                cache.remove((Serializable) key);
            }

        } catch (ClassCastException e) {
            throw new JoftiException(e);
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } finally {
			releaseUpdateLock();
		}
    }

    /**
     * Remove all elements in the cache, but leave the cache in a useable state.
     * 
     * @throws CacheException
     */
    public synchronized void removeAll() throws JoftiException
    {
        try {

            cache.removeAll();
            index.removeAll();

        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        } catch (Exception e) {
            throw new JoftiException(e);
        }
    }

    public synchronized void init(Properties properties) throws JoftiException
    {
        try {
            String cacheConfigFile = null;
            if (properties != null) {
                String key = null;
                for (Iterator it = properties.keySet().iterator(); it.hasNext();) {
                    key = (String) it.next();
                    if (MUTABLE_VALUES.equalsIgnoreCase(key)) {
                        checkMutable = Boolean.valueOf(
                                properties.getProperty(key)).booleanValue();
                        if (log.isInfoEnabled()) {
                            log.info("Mutability checking is set to "
                                    + checkMutable);
                        }
                    }
                    if ("file".equalsIgnoreCase(key)) {
                        cacheConfigFile = properties.getProperty(key);
                    }
                }
            }

            if (manager == null) {
                if (cacheConfigFile != null) {
                    manager = net.sf.ehcache.CacheManager
                            .create(cacheConfigFile);
                } else {
                    manager = net.sf.ehcache.CacheManager.create();
                }
            }
            if (cache == null) {
                initCache();
            } else {
                // try and add it to the cache
                if (!(manager.cacheExists(cache.getName()))) {
                    manager.addCache(cache);
                } else {
                    log
                            .warn("IndexCache "
                                    + cache.getName()
                                    + "already exists in manager - not adding new cache");
                }
            }

        } catch (net.sf.ehcache.CacheException e) {
            throw new JoftiException(e);
        }

    }

    private void initCache() throws JoftiException
    {
        if (cache == null) {
            cache = manager.getCache(name);
            if (cache == null) {
                log.warn("Unable to find cache named '" + name
                        + "' in EHCacheManager");
                String[] names = manager.getCacheNames();
                log.warn("Available cache names are:");
                for (int i = 0; i < names.length; i++) {
                    log.warn("IndexCache:" + names[i]);
                }
                throw new JoftiException("Unable to find cache named '" + name
                        + "' in EHCacheManager");
            }
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#start()
     */
    public synchronized void start() throws JoftiException
    {
        try {
            loadInitialValues(cache);
        } catch (CacheException ce) {
            throw new JoftiException(ce);
        }
    }

    private void loadInitialValues(net.sf.ehcache.Cache cache)
            throws CacheException, JoftiException
    {
        List keys = cache.getKeys();
        for (Iterator it = keys.iterator(); it.hasNext();) {
            Object key = it.next();
            Element element = cache.get((Serializable) key);
            if (key instanceof Comparable) {
                Comparable tempKey = (Comparable) key;
                if (element != null && element.getValue() != null) {
                    // then we must remove from index
                    index.insert(tempKey, element.getValue());
                }
            } else {
                log.info("Ignoring value at key:" + key
                        + " as key is not Comparable");
            }
        }
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public synchronized void destroy() throws JoftiException
    {
        try {
            if (manager != null) {
                manager.removeCache(name);
                manager.shutdown();
                index.removeAll();
            }
        } catch (IllegalStateException e) {
            throw new JoftiException(e);
        }
    }

    public String getName()
    {
        return name;
    }

    public void setName(String name)
    {
        this.name = name;
    }

    public String toString()
    {
        return "EHCache(" + getName() + ')';
    }

    public Object getCache()
    {
        return cache;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#getCacheImpl()
     */
    public Object getCacheImpl()
    {
        return cache;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofticache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
     */
    public void setInternalIndex(InternalIndex index)
    {
        this.index = index;

    }

   
	protected Object getCacheValue(Object key) {
		return get(key);
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		return index;
	}
	
	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
}