package com.jofti.cache.adapter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;

import net.sf.ehcache.CacheException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.GlobalTransaction;
import org.jboss.cache.Node;
import org.jboss.cache.PropertyConfigurator;
import org.jboss.cache.TreeCache;
import org.jboss.cache.TreeCacheListener;
import org.jboss.cache.lock.IsolationLevel;
import org.jgroups.View;



import com.jofti.api.IndexCache;
import com.jofti.api.IndexQuery;
import com.jofti.api.NameSpaceKey;
import com.jofti.cache.CacheAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.NameSpacedCacheAdapter;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.ITransactionAware;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.core.TransactionLevel;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ClassFieldMethods;
import com.jofti.tree.NameSpacedTreeIndex;
import com.jofti.util.CompositeComparator;
import com.jofti.util.ObjectProcedureAdapter;
import com.jofti.util.OpenHashMap;
import com.jofti.util.ValueTreeMap;
import com.tangosol.dev.compiler.Manager;

import edu.emory.mathcs.backport.java.util.concurrent.ConcurrentHashMap;

/**
 * 
 * 
 * The adapter is responsible for interaction between the index and JBossCache.
 * <p>
 * 
 * The main difference between JBossCache and the other caches is its use of
 * transactions. The adapter is responsible for maintaining the transactional
 * semantics of the index.
 * <p>
 * 
 * @author Steve Woodcock (steve@jofti.com)<br>
 * @version 1.0<br>
 */
public class JBossCacheAdapter extends BaseAdaptor implements CacheAdapter,
        NameSpacedCacheAdapter,  ITransactionAware
{

    TreeCache                            cache          = null;

    javax.transaction.TransactionManager txMgr          = null;

    private static Log                   log            = LogFactory
                                                                .getLog(JBossCacheAdapter.class);

    private String                       name;


    TransactionLevel                     level          = null;

    private int                          expiryTime     = 3600;                                  // seconds

    private boolean                      requiresStarting;

    // used for non-existent transactions
    private Indexer                      defaultIndexer = new Indexer();

    private Map                          transactionMap = new ConcurrentHashMap();

    private Map                          xaMap          = new ConcurrentHashMap();

    /**
     * @return Returns the expiryTime.
     */
    public int getExpiryTime()
    {
        return expiryTime;
    }

    /**
     * @param expiryTime
     *            The expiryTime to set.
     */
    public void setExpiryTime(int expiryTime)
    {
        this.expiryTime = expiryTime;
    }

    public JBossCacheAdapter()
    {
        requiresStarting = true;
    }

    public JBossCacheAdapter(Object cache)
    {
        this.cache = (TreeCache) cache;
    }

    public void setCacheImpl(Object cache)
    {
        this.cache = (TreeCache) cache;
        try {
            initCache(this.cache);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    
    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#get(java.lang.Object)
     */
    public Object get(Object key)
    {

        // test the key
        if (key != null) {
            if (key instanceof NameSpaceKey) {
                NameSpaceKey temp = null;
                try {
                    temp = (NameSpaceKey) key;
                } catch (ClassCastException cce) {
                    log.warn("key value must be a NameSpaceKey ", cce);
                    return null;
                }
                // do the get on the tree

                try {
                    synchronized (getCacheLock(key)) {
                        Object val = cache.get((Fqn) temp.getNameSpace(), temp
                                .getKey());
                        return val;
                    }
                } catch (org.jboss.cache.CacheException ce) {
                    log.warn("Unable to retrieve value from cache ", ce);
                }

            } else {
                log
                    .warn("Unable to retrieve value from cache Key type must be "
                                + NameSpaceKey.class);
            }

        }
        // we return null here if we got an exception because it is consistent
        // with most other cache implementations
        return null;
    }

    private Object getFromCache(Object key) throws JoftiException
    {

        // test the key
            if (key instanceof NameSpaceKey) {
                NameSpaceKey temp = null;
                try {
                    temp = (NameSpaceKey) key;
                } catch (ClassCastException cce) {
                    log.warn("key value must be a NameSpaceKey ", cce);
                    return null;
                }
                // do the get on the tree

                try {
                        Object val = cache.get((Fqn) temp.getNameSpace(), temp
                                .getKey());
                        return val;
                } catch (Throwable ce) {
                    throw new JoftiException(ce);
                }

            } else {
                log
                    .warn("Unable to retrieve value from cache Key type must be "
                                + NameSpaceKey.class);
            }

        // we return null here if we got an exception because it is consistent
        // with most other cache implementations
        return null;
    }
    
    
    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#put(java.lang.Object, java.lang.Object)
     */
    public void put(Object key, Object value) throws JoftiException
    {

        NameSpaceKey temp = null;

        try {
            temp = (NameSpaceKey) key;
        } catch (ClassCastException cce) {
            throw new JoftiException("key value must be a NameSpaceKey ", cce);
        }

        // get the transaction for the thread
        Transaction tx = getTransaction();

        // get the indexer to be used for the transaction
        JBossIndexer indexer = getIndexer(tx);

        try {
			acquireUpdateLock();
        }catch (Exception e){
        	log.error("unable to acquire update lock ",e);
        	throw new JoftiException(e);
        }
        try {
            synchronized (getCacheLock(key)) {
                // first see if it already exists
                Object result = cache.get((Fqn) temp.getNameSpace(), temp
                        .getKey());
                if (log.isDebugEnabled()){
                	log.debug( "inserting " + temp.getNameSpace() + " " + temp
                        .getKey());
                }
                cache.put((Fqn) temp.getNameSpace(), temp.getKey(), value);

                if (result != null) {
                    indexer.update(temp, result, value);
                } else {
                    indexer.add(temp, value);
                }
            }
            registerIndexer(indexer, tx);

        } catch (org.jboss.cache.CacheException ce) {
            throw new JoftiException(ce);
        } catch (Exception e) {
            throw new JoftiException(e);
        }finally {
			releaseUpdateLock();
		}
       

    }

    // get a transaction from the transaction manager
    private Transaction getTransaction() throws JoftiException
    {
        Transaction tx = null;
        try {
            if (txMgr != null) {
                tx = txMgr.getTransaction();
            }
        } catch (SystemException se) {
            throw new JoftiException(se);
        }
        return tx;
    }

    /**
     * Removes the element which matches the namespace.
     * <p>
     * If no element matches, nothing is removed and no Exception is thrown.
     * 
     * @param namespace  the namespace to remove
     * @throws CacheException
     */
    public void removeNameSpace(Object nameSpace) throws JoftiException
    {
    	try{
    		
    	acquireUpdateLock();
        try {
            synchronized (getCacheLock(nameSpace)) {
                // first see if it already exists
                Node result = null;
                Fqn temp = null;
                if (nameSpace instanceof String) {
                    temp = Fqn.fromString((String) nameSpace);

                } else if (nameSpace instanceof Fqn) {
                    temp = (Fqn) nameSpace;
                } else {
                    throw new JoftiException(
                            "namespace object must be a String or an Fqn object "
                                    + nameSpace.getClass());
                }
                // we have the node here
                result = cache.get(temp);
            
                // lets loop through and get a full list of child nodes that
                // are going to be removed
                if (result != null){
                	
	                Map childMap = result.getChildren();
	                List nodesToRemove = parseChildren(childMap, new ArrayList(),
	                        temp.toString() + "/");
	                nodesToRemove.add(temp.toString());
	                if (log.isDebugEnabled()) {
	                    log.debug(nodesToRemove);
	                }
	
	                
	                for (Iterator it = nodesToRemove.iterator(); it.hasNext();) {
	                    cache.remove((String) it.next());
	                }
	                
	              
                }
            }
        } catch (org.jboss.cache.CacheException ce) {
            throw new JoftiException(ce);
        } catch (Exception e) {
            throw new JoftiException(e);
        }finally {
			releaseUpdateLock();
		}
    	} catch (Exception e){
    		log.error("unable to acquire update lock",e);
    	}
    }

    private List parseChildren(Map childMap, List tempList, String prefix)
    {
        if (childMap != null) {
            for (Iterator it = childMap.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                Map map = ((Node) entry.getValue()).getChildren();
                parseChildren(map, tempList, prefix + "/" + entry.getKey()
                        + "/");
                tempList.add(prefix + entry.getKey());
            }
        }
        return tempList;
    }

    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#remove(java.lang.Object)
     */
    public void remove(Object key) throws JoftiException
    {

        NameSpaceKey temp = null;

        try {
            temp = (NameSpaceKey) key;
        } catch (ClassCastException cce) {
            throw new JoftiException("key value must be a JBossKeyWrapper ",
                    cce);
        }
        Transaction tx = getTransaction();
        JBossIndexer indexer = getIndexer(tx);

        try{
        	acquireUpdateLock();
        try {

            synchronized (getCacheLock(key)) {

                Object result = cache.remove((Fqn) temp.getNameSpace(), temp
                        .getKey());

                if (result != null) {
                    indexer.remove(temp, result);
                } else {
                    if (index.contains(temp)) {
                        indexer.remove(temp);
                    }
                }
            }
            registerIndexer(indexer, tx);
        } catch (org.jboss.cache.CacheException ce) {
            throw new JoftiException(ce);
        } catch (Exception e) {
            throw new JoftiException(e);
        }finally {
			releaseUpdateLock();
		}
    	} catch (Exception e){
    		log.error("unable to acquire update lock",e);
    	}

    }

    /**
     * Remove all elements in the cache, but leave the cache in a useable state.
     * 
     * @throws CacheException
     */
    public void removeAll() throws JoftiException
    {
        try {
            cache.remove("/");
  
            index.removeAll();

        } catch (Exception ce) {
            throw new JoftiException(ce);
        }

    }

    private synchronized JBossIndexer getIndexer(Transaction tx)
            throws JoftiException
    {

        JBossIndexer indexer = null;

        if (tx == null || level == TransactionLevel.NONE
                || level == TransactionLevel.READ_UNCOMMITTED) {
            // we have no transaction
            indexer = defaultIndexer;
        } else {
            // get an existing indexer if we have one or create a new one

            indexer = (JBossIndexer) transactionMap.get(tx);

            if (indexer == null) {
                indexer = new TransactionIndexer(index.getIntrospector());
                ((TransactionIndexer) indexer).init();
                transactionMap.put(tx, indexer);
            }
        }
        return indexer;
    }

    private synchronized void registerIndexer(JBossIndexer indexer,
            Transaction tx) throws SystemException, RollbackException,
            JoftiException
    {
        if (indexer instanceof TransactionIndexer
                && (!xaMap.containsKey(indexer))) {
            try {
                tx.registerSynchronization((Synchronization) indexer);
                transactionMap.put(tx, indexer);
                xaMap.put(indexer, tx);
                if (log.isDebugEnabled()) {
                    log.debug("registering transaction " + tx + " in cache "
                            + cache.getLocalAddress());
                }
            } catch (Exception e) {
                throw new JoftiException(e);
            }
        }

    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException
    {
        try {
            String cacheConfigFile = null;
            if (properties != null) {
                String key = null;
                for (Iterator it = properties.keySet().iterator(); it.hasNext();) {
                    key = (String) it.next();
                    if (MUTABLE_VALUES.equalsIgnoreCase(key)) {
                        checkMutable = Boolean.valueOf(
                                properties.getProperty(key)).booleanValue();
                        if (log.isInfoEnabled()) {
                            log.info("Mutability checking is set to "
                                    + checkMutable);
                        }
                    }
                    if ("file".equalsIgnoreCase(key)) {
                        cacheConfigFile = properties.getProperty(key);
                    }
                }
            }
            if (cache == null) {
                cache = new TreeCache();
            }
            if (requiresStarting) {
                PropertyConfigurator config = new PropertyConfigurator();
                if (cacheConfigFile == null){
                    throw new JoftiException("Config file cannot be null in config - check you have set the config file property correctly");
                }else{
                    config.configure(cache, cacheConfigFile);
                }
            }

            initCache(cache);

        } catch (Exception e) {
            throw new JoftiException(e);
        }

    }

    private void initCache(TreeCache cache) throws JoftiException
    {
        cache.addTreeCacheListener(new EventListener());
        txMgr = cache.getTransactionManager();
        level = getTransactionLevel();
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public void destroy() throws JoftiException
    {
        cache.destroy();
    }

    public String getName()
    {
        return name;
    }

    public void setName(String name)
    {
        this.name = name;
    }

    public String toString()
    {
        return "JBossCache(" + getName() + ')';
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#getCacheImpl()
     */
    public Object getCacheImpl()
    {
        return cache;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.cache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
     */
    public void setInternalIndex(InternalIndex index)
    {
        if (!(index instanceof NameSpacedTreeIndex)){
            throw new RuntimeException("JBossCache tree index must be set to "+ NameSpacedTreeIndex.class);
        }
        this.index = index;

    }

    class TransactionIndexer implements Synchronization, JBossIndexer
    {


        protected ChangeRecorder recorder   = null;

        TransactionIndexer(ClassIntrospector parser)
        {
            recorder = new ChangeRecorder(parser);
        }

        /*
         * (non-Javadoc)
         * 
         * @see javax.transaction.Synchronization#beforeCompletion()
         */

        public void init() throws JoftiException
        {
            recorder.init(new Properties(),index.getClass().getName());
        }

        public void beforeCompletion()
        {
            // we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see javax.transaction.Synchronization#afterCompletion(int)
         */
        public void afterCompletion(int status)
        {
            // first remove
            // first remove ourselves
            Object obj = xaMap.get(this);
            xaMap.remove(this);
            transactionMap.remove(obj);
            switch (status)
                {
                case Status.STATUS_COMMITTED:
                	try {
                		
                	acquireUpdateLock();
                    try {
                        applyAll();
                    } catch (Throwable e) {
                        log.error("failed committing changes to index", e);
                    }finally{
                    	releaseUpdateLock();
                    }
                	} catch (JoftiException e){
                		log.error("unable to acquire update lock",e);
                	}
                    break;

                case Status.STATUS_MARKED_ROLLBACK: // this one is probably not
                // needed
                case Status.STATUS_ROLLEDBACK:
                default:
                    ;
                }

        }

        public void update(NameSpaceKey keyWrapper, Object oldValue,
                Object newValue) throws JoftiException
        {

            recorder.update(keyWrapper, oldValue, newValue, index
                    .getIntrospector());

        }

        public void add(NameSpaceKey keyWrapper, Object value)
                throws JoftiException
        {
            recorder.add(keyWrapper, value, index.getIntrospector());
        }

        public void remove(NameSpaceKey keyWrapper, Object value)
                throws JoftiException
        {
            recorder.remove(keyWrapper, value, index.getIntrospector());
        }

        
        public void remove(NameSpaceKey keyWrapper) throws JoftiException
        {
            recorder.remove(keyWrapper, index.getIntrospector());

        }

        private void applyAll() throws JoftiException
        {
            // first do removes
            if (log.isDebugEnabled()) {
                log.debug("Applying all index updates after transaction commit on cache "
                                + cache.getLocalAddress());
            }
            Map removes = recorder.getRemovedMap();
            for (Iterator it = removes.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                Object key = entry.getKey();
                List values = (List) entry.getValue();
                
                synchronized (getCacheLock(key)) {
                    for (Iterator valIt = values.iterator(); valIt.hasNext();) {
                        Object value = valIt.next();

                        index.removeByKey((Comparable) key);
                        if (log.isDebugEnabled()) {
                            log.info("index removed  " + key + " value "
                                    + value);
                        }
                    }
                }
            }
            // now remove key only
            for (Iterator it = recorder.getRemovedKeys().iterator(); it
                    .hasNext();) {
                Object key = it.next();

                synchronized (getCacheLock(key)) {

                    index.removeByKey((Comparable) key);
                    if (log.isDebugEnabled()) {
                        log.debug("index removed  entries for " + key);
                    }
                }
            }

            // now remove the updates
            Map updates = recorder.getUpdatedMap();
            for (Iterator it = updates.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                Object key = entry.getKey();
                List values = (List) entry.getValue();
                synchronized (getCacheLock(key)) {
                    for (Iterator valIt = values.iterator(); valIt.hasNext();) {
                        Object value = valIt.next();
                        index.removeByKey((Comparable) key);
                        if (log.isDebugEnabled()) {
                            log.debug("index removed for update " + key
                                    + " value " + value);
                        }
                    }
                }
            }

            // now add all the adds
            Collection col = recorder.getAllTreeValues();
            for (Iterator it = col.iterator(); it.hasNext();) {
                NameSpaceKey wrapper = (NameSpaceKey) it.next();

                synchronized (getCacheLock(wrapper)) {
                    Object val = null;

                    try {
                        val = cache._get((Fqn) wrapper.getNameSpace(), wrapper
                                .getKey(), false);
                        // val =
                        // cache.peek(wrapper.fqn).getData().get(wrapper.key);
                    } catch (Exception e) {
                        log.warn("Unable to insert entry " + wrapper
                                + " into index ", e);
                    }
                    if (val != null) {
                        if (log.isDebugEnabled()) {
                            log.debug("index removed for update " + wrapper
                                    + " value " + val);
                        }
                        index.insert(wrapper, val);
                    } else {
                        log.warn("Entry lookup for " + wrapper
                                + " in transaction tree is null in cache ");
                    }
                }
            }
            if (log.isDebugEnabled()) {
                log
                        .debug("Finished Applying all index updates after transaction commit "
                                + cache.getLocalAddress());
            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.adapter.JBossIndexer#contains(com.jofti.cache.adapter.JBossKeyWrapper)
         */
        public boolean contains(NameSpaceKey keyWrapper) throws JoftiException
        {
            
            return index.contains(keyWrapper);
        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.adapter.JBossIndexer#query(com.jofti.api.IndexQuery,
         *      com.jofti.api.CacheAccessor)
         */
        public Map query(IndexQuery query, IndexCache parent) throws JoftiException
        {

            // repeatable read - always return previous results plus any new
            // ones

            // serializable - block if any are blocked
        	acquireQueryLock();
        	
        	
        	OpenHashMap localResults =null;
        	final ChangeRecorder tempRecorder = recorder;
        	OpenHashMap processedQueryMap =null;
        	
        	 Object nameSpace = ((INameSpaceAware) query)
             .getNameSpace();
             
             if (((QueryId)query).getQueryType() != QueryType.PARSED_QUERY) {
                 query = index.getParserManager().parseQuery(query);
                  
          	}
             
        	try {
            // query the main index
        	final OpenHashMap queryMap = (OpenHashMap)index.query(query);

            // remove all stuff in our transaction

            queryMap.forEachKey(new ObjectProcedureAdapter(){
            	public boolean apply(Object key){

            		 if (tempRecorder.getRemovedKeys().contains(key)
                            || tempRecorder.getRemovedMap().containsKey(key)
                            || tempRecorder.getUpdatedMap().containsKey(key)){
            		 	queryMap.removeNoReHash(key);
            		 }
            		 return true;
            	}
            });
           processedQueryMap = queryMap;

            // query the local index
            localResults = (OpenHashMap)recorder.query(query);
        	} finally {
        		releaseQueryLock();
        	}
            // now let us merge the results
           Map res = mergeOpenMaps(parent, processedQueryMap, localResults );
         
          
           
           return getCacheValues(parent,res,nameSpace,(IParsedQuery)query,index.getIntrospector());

        }



        protected Map mergeOpenMaps(IndexCache parent, OpenHashMap map1, OpenHashMap map2){
        	OpenHashMap smaller =null;
        	OpenHashMap larger =null;
        	if (map1.size() < map2.size()){
        		smaller = map1;
        		larger = map2;
        	}else{
        		smaller =map2;
        		larger = map1;
        	}
        	
        	larger.ensureCapacity(larger.size() + smaller.size() *2);
        	
        	final OpenHashMap tempLarger = larger;
        	smaller.forEachPair(new ObjectProcedureAdapter(){
        		public boolean apply(Object key, Object value){
        			if (!tempLarger.containsKey(key)){
        				tempLarger.put(key, value);
        			}
        			return false;
        		}
        	});
        	return larger;
        }
       
    }

  
    /**
     * @param parent
     * @param map2
     * @param previousResults
     * @param resultMap
     * @throws JoftiException
     */

    class Indexer implements JBossIndexer
    {

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.JBossIndexer#update(com.jofti.cache.JBossKeyWrapper,
         *      java.lang.Object, java.lang.Object)
         */
        public void update(NameSpaceKey keyWrapper, Object oldValue,
                Object newValue) throws JoftiException
        {

           index.removeByKey(keyWrapper);

            index.insert(keyWrapper, newValue);
     
        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.JBossIndexer#add(com.jofti.cache.JBossKeyWrapper,
         *      java.lang.Object)
         */
        public void add(NameSpaceKey keyWrapper, Object value)
                throws JoftiException
        {
            index.insert(keyWrapper, value);

        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.JBossIndexer#remove(com.jofti.cache.JBossKeyWrapper,
         *      java.lang.Object)
         */
        public void remove(NameSpaceKey keyWrapper, Object value)
                throws JoftiException
        {
            index.removeByKey(keyWrapper);

        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.JBossIndexer#remove(com.jofti.cache.JBossKeyWrapper)
         */
        public void remove(NameSpaceKey keyWrapper) throws JoftiException
        {
            index.removeByKey(keyWrapper);

        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.adapter.JBossIndexer#contains(com.jofti.cache.adapter.JBossKeyWrapper)
         */
        public boolean contains(NameSpaceKey keyWrapper) throws JoftiException
        {

            return index.contains(keyWrapper);
        }

        /*
         * (non-Javadoc)
         * 
         * @see com.jofti.cache.adapter.JBossIndexer#query(com.jofti.api.IndexQuery)
         */
        public Map query(IndexQuery query, IndexCache parent) throws JoftiException
        {

        	Map temp =null;
        	
        	Object nameSpace = ((INameSpaceAware) query)
            .getNameSpace();
        	
        	if (((QueryId)query).getQueryType() != QueryType.PARSED_QUERY) {
               query = index.getParserManager().parseQuery(query);
                
        	}
        
        	acquireQueryLock();
        	
        	try {
            	temp = index.query(query);
        	} finally {
        		releaseQueryLock();
        	}

            return getCacheValues(parent, temp, nameSpace,(IParsedQuery)query,index.getIntrospector());
        }

    }

    private boolean isValidForNameSpace(Object nameSpace, Object keyNameSpace)
    {

        return (nameSpace.equals(keyNameSpace) || ((Fqn) keyNameSpace)
                .isChildOf((Fqn) nameSpace));
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.cache.CacheAdapter#start()
     */
    public void start() throws JoftiException
    {

        try {
            if (requiresStarting) {
                if (cache == null){
                    log.warn("JBossCache instance is NULL in adaptor - ensure Cache has been configured correctly");
                }
                cache.start();
            }
            txMgr = cache.getTransactionManager();
            loadInitialValues(cache);
        } catch (Exception e) {

            throw new JoftiException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.NameSpacedIndex#put(java.lang.Object,
     *      java.lang.Object, java.lang.Object)
     */
    public void put(Object nameSpace, Object key, Object value)
            throws JoftiException
    {
        if (nameSpace instanceof Fqn) {
            put(new NameSpaceKeyWrapper((Fqn) nameSpace, key), value);
        } else if (nameSpace instanceof String) {
            put(
                    new NameSpaceKeyWrapper(Fqn.fromString((String) nameSpace),
                            key), value);
        } else {
            throw new JoftiException("Unable to insert value " + value
                    + " namespace must be either a " + Fqn.class
                    + "or a String ");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.NameSpacedIndex#get(java.lang.Object,
     *      java.lang.Object)
     */
    public Object get(Object nameSpace, Object key)
    {

        if (nameSpace instanceof Fqn) {
            return get(new NameSpaceKeyWrapper((Fqn) nameSpace, key));
        } else if (nameSpace instanceof String) {
            return get(new NameSpaceKeyWrapper(Fqn
                    .fromString((String) nameSpace), key));
        } else {
            return null;
        }
    }

    

    /* (non-Javadoc)
     * @see com.jofti.api.NameSpacedIndex#remove(java.lang.Object, java.lang.Object)
     */
    public void remove(Object nameSpace, Object key) throws JoftiException
    {
        if (nameSpace instanceof Fqn) {
            remove(new NameSpaceKeyWrapper((Fqn) nameSpace, key));
        } else if (nameSpace instanceof String) {
            remove(new NameSpaceKeyWrapper(Fqn.fromString((String) nameSpace),
                    key));
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.api.IndexCache#query(com.jofti.api.IndexQuery)
     */
    public Map query(IndexQuery query) throws JoftiException
    {

    	IndexQuery origQuery = query;
        if (((QueryId)query).getQueryType() == QueryType.UNPARSED_QUERY) {
            query = (IndexQuery) index.getParserManager().parseQuery( query);
        }
        // decorate the query with the fqn
        if (query instanceof INameSpaceAware) {
            INameSpaceAware orig = (INameSpaceAware) query;
            // reset the namespace object to make sure it is an fqn
            Fqn tempFqn = null;
            
            Object tempNameSpace = orig.getNameSpace();
            if (tempNameSpace ==null){ 
            	throw new JoftiException("NameSpace is required for JBossCache query: "+ origQuery);
            }else if (orig.getNameSpace() instanceof String) {
                tempFqn = Fqn.fromString((String) orig.getNameSpace());
                orig.setNameSpace(tempFqn);
            } else if (!(orig.getNameSpace() instanceof Fqn)) {
                throw new JoftiException(
                        "name space object must be a string or Fqn for JBoss IndexCache");
            }

        }

        Transaction tx = getTransaction();

        JBossIndexer indexer = getIndexer(tx);

        try {
            registerIndexer(indexer, tx);
        } catch (Exception e) {
            throw new JoftiException(e);
        }

        
        return indexer.query(query, this);


    }

	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
	
    protected Map getCacheValues(final IndexCache cache, Map col,final Object nameSpace,final IParsedQuery query, ClassIntrospector introspector 
    		)
            throws JoftiException
    {
    	final Map returnClasses = query.getResultFieldsMap();
		final CompositeComparator comp = query.getOrderingComparator();
		final int maxResults = query.getMaxResults();
		final int startEntry = query.getFirstResult();
	
    	Map interim = null;
		if (comp ==null || comp.getSize() ==0){
			interim =new HashMap(col.size() + 2, 1.00f);
		}else{
			interim = new ValueTreeMap(comp);
		}
		
		// return map
		final Map temp = interim;
        
        final Object[] errors = new Object[1];
      //  Map returnClasses = parsedQuery.getResultFieldsMap();
        OpenHashMap originalMap = (OpenHashMap)col;
        
        boolean noError = originalMap.forEachPair(new ObjectProcedureAdapter(){
           public boolean apply(Object key,Object value){
           	
           	if (value == null || returnClasses.containsKey(value)){
           		
           	
           		Object result =null;
           		
           		//get value if valid namespace or subnamespace for query namespace
           		if  (isValidForNameSpace(nameSpace, ((NameSpaceKey) key)
                        .getNameSpace()))
           		{
	           		try {
	           			result = getCacheValue(cache, key, nameSpace);
	           		
		           	}catch (JoftiException e){
		           	 log.warn("unable to get cache value for key " + key + " in nameSpace "+ nameSpace,e);
		           	 errors[0] = new JoftiException("unable to get cache value for key " + key + " in nameSpace "+ nameSpace,e);
		           	 return false;
		           	}
	           	
		           	if (result == null){
	            	
	            		if(log.isWarnEnabled()){
	            			log.warn("Index and cache have become out of date for key "+key);
	            		}
	            	
	           		return true;
		            }else{
		           
		            	if (checkMutable) {
			                result = checkMutable(key, result);
			                // mutability check failed
			                if (result == null){
			                	if (log.isDebugEnabled()){
			                		log.debug("Object under key:"+key +" has changed in cache since it was indexed");
			                	}
			                	return true;
			                }
			            }
		            	
		            	if (returnClasses != null){
		            	 	// we need to get the value from result if we are using fields
		            	 	ClassFieldMethods fieldSet = (ClassFieldMethods)returnClasses.get(value);
			           		
		            	 	Map tempMap = fieldSet != null ? fieldSet.getFieldMap():null;
		            	 	
			           		if (tempMap != null && tempMap.size()>0 ){
			           			// get the fields and apply the values to them
			           			log.warn("field is set for "+ value);
			           		}else{
			           			// there are no fields just return the value
			           			temp.put(key, result);
			           		}
	            	 	}else{
	            	 		temp.put(key, result);
	            	 	}
			            
		           	}
           		}
           	}
            return true;
           }
        });
        if (noError){
//        	 now limit result size
    		if (maxResults >0 || startEntry >0){
    			return limitResults(temp, startEntry, maxResults);
    		}else if (startEntry <0 || maxResults <0){
    			if (startEntry <0){
    				throw new IllegalArgumentException("startResult cannot be less than 0:"+startEntry); 
    			}
    			if (maxResults <0){
    				throw new IllegalArgumentException("maxResults cannot be less than 0:"+maxResults); 
    			}
    		}
        	return temp;
        }else{
        	throw (JoftiException)errors[0];
        }
      
     
    }

  
    protected Object checkMutable(Object key, Object result)
    {
        try {
            // first parse the object - again
            Map cacheObjectValues = index.getIntrospector().getAttributeValues(
                    result);

            Map indexObjectValues = index.getAttributesByKey((Comparable) key);
            if (cacheObjectValues.equals(indexObjectValues)) {
                return result;
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Object under Key " + key
                            + " -  attributes changed without re-insert");
                }
            }

        } catch (JoftiException e) {
            log.warn("Error checking mutability", e);
        }

        return null;
    }

    private void loadInitialValues(TreeCache temp) throws JoftiException
    {
        Throwable t =null;
        // get a list of all the children in the tree
        Collection fqns = new ArrayList();

        Transaction tx = null;

        try {
            if (cache.getIsolationLevelClass() != IsolationLevel.NONE) {
                tx = getTransaction();
                if (tx == null) {
                    if (cache.getTransactionManager() == null){
                        log.warn("No TransactionManager founf in TreeCache for isolation level "+ cache.getIsolationLevel());
                        throw new JoftiException("You must have a transaction manager configured in JBossCache to use Isolation level "+ cache.getIsolationLevel());
                    }
                    cache.getTransactionManager().begin();
                    tx = getTransaction();
                }
            }
            Fqn start = cache.getRoot().getFqn();
            fqns.add(start);
            try {
                fqns = getChildrenForFqn(temp, start, fqns);
            } catch (org.jboss.cache.CacheException e) {
                throw new JoftiException(e);
            }
            
            // for each of thse then get the key and value out
            for (Iterator it = fqns.iterator(); it.hasNext();) {
                Fqn fqn = (Fqn) it.next();
                Node node = cache.get(fqn);

                if (node != null && node.getDataKeys() != null
                        && node.getDataKeys().size() > 0) {
                    Set dataKeys = node.getDataKeys();
                    for (Iterator mapIt = dataKeys.iterator(); mapIt
                            .hasNext();) {
                        Object key = mapIt.next();
                        Object value = node.get(key);
                        index.insert(new NameSpaceKeyWrapper(fqn, key), value);
                    }
                }
            }
        } catch (Exception e) {
            // dump the index
            t = e;
            index.removeAll();
            if (cache.getIsolationLevelClass() != IsolationLevel.NONE) {
                try {
                    log
                            .warn("Unable to complete index of initial values -  rolling back "
                                    + e);
                    if (tx != null){
                        tx.rollback();
                    }else{
                        log.warn("Expected to rollback Transaction but Transaction is null");
                    }
                } catch (IllegalStateException e1) {
                    throw new JoftiException(e1);
                } catch (SystemException e1) {
                    throw new JoftiException(e1);
                }catch (Throwable t1){
                    throw new JoftiException("Unable to rollback tx as is "+ tx);
                }
            }

        } finally {
            if (cache.getIsolationLevelClass() != IsolationLevel.NONE) {
                try {
                    if (tx != null && !isRollback(tx.getStatus())){
                     
                        tx.commit();
                    }else{
                        log.info("Transaction not committed as marked as rolledback or null");
                    }
                    log.info("initial data transaction loaded");
                } catch (SecurityException e) {

                    throw new JoftiException(e);
                } catch (RollbackException e) {

                    throw new JoftiException(e);
                } catch (HeuristicMixedException e) {

                    throw new JoftiException(e);
                } catch (HeuristicRollbackException e) {

                    throw new JoftiException(e);
                } catch (SystemException e) {

                    throw new JoftiException(e);
                }
            }
            if (t != null){
                if (t instanceof JoftiException){
                    throw (JoftiException) t;
                }else{
                    throw new JoftiException(t);
                }
            }
        }

    }

    private Collection getChildrenForFqn(TreeCache temp, Fqn fqn,
            Collection names) throws org.jboss.cache.CacheException
    {

        Set set = temp.getChildrenNames(fqn);
        if (set != null && !set.isEmpty()) {
        	 int size = set.size();
             Iterator it = set.iterator();
             for (int i=0;i<size;i++) {
                Fqn tempFqn = Fqn.fromString((String) it.next());

                getChildrenForFqn(temp, tempFqn, names);
                names.add(tempFqn);
            }
            return names;

        } else {
            return names;
        }

    }

    private Object getCacheValue(IndexCache parent, Object key,
            Object nameSpace) throws JoftiException
    {
        Object result = null;

        if (key != null
                && isValidForNameSpace(nameSpace, ((NameSpaceKey) key)
                        .getNameSpace())) {

            // will acquire readlock or will throw an exception after some time
        	// if no readlock can be acquired
           
    			result = ((JBossCacheAdapter)parent).getFromCache(key);
    		
        }
        return result;
    }

    class EventListener implements TreeCacheListener
    {

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeCreated(org.jboss.cache.Fqn)
         */
        public void nodeCreated(Fqn arg0)
        {
            // check with loading otherwise we do not care about node
            // creation yet

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeRemoved(org.jboss.cache.Fqn)
         */
        public void nodeRemoved(Fqn arg0)
        {
            if (log.isDebugEnabled()) {
                log.debug("node removed triggered for " + arg0);
            }
            try {
                Transaction tx = getTransaction();

               final  JBossIndexer indexer = getIndexer(tx);
                registerIndexer(indexer, tx);

                OpenHashMap searchKeys = (OpenHashMap)index.getEntries(new NameSpaceWrapper(arg0));

                // this is an openHashMap
                searchKeys.forEachKey(new ObjectProcedureAdapter(){
                	/* (non-Javadoc)
					 * @see com.jofti.util.ObjectProcedureAdapter#apply(java.lang.Object)
					 */
					public boolean apply(Object element) {
						try {
							indexer.remove((NameSpaceKeyWrapper)element);
						}catch (Exception e){
							// we should log this
							log.warn("unable to remove key "+ element);
						}
						return true;
					}
                });
                
              

            } catch (Exception e) {
                log.warn("unable to remove entries in node removal " + arg0);
            }

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeLoaded(org.jboss.cache.Fqn)
         */
        public void nodeLoaded(Fqn arg0)
        {
            // we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeEvicted(org.jboss.cache.Fqn)
         */
        public void nodeEvicted(Fqn arg0)
        {
            //  we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeModified(org.jboss.cache.Fqn)
         */
        public void nodeModified(Fqn arg0)
        {

            // this can be either a put or a remove - all this work because
            // JBOSS APIs
            // give no information to go on
            try {
                //see if transaction is local

                Transaction tx = getTransaction();

                GlobalTransaction gtx = cache.getTransactionTable().get(tx);

                if (gtx != null && gtx.getAddress() != null
                        && (!gtx.getAddress().equals(cache.getLocalAddress()))) {
                    if (log.isDebugEnabled()) {
                        log.debug("gtx " + gtx.getAddress() + " cache address "
                                + cache.getLocalAddress());
                    }
                    //is remote - now find the indexed key list
                   final  List indexKeys = new ArrayList();
                    

                    OpenHashMap searchKeys = (OpenHashMap)index.getEntries(new NameSpaceWrapper(arg0));

                    // this is an openHashMap
                    searchKeys.forEachKey(new ObjectProcedureAdapter(){
                    	/* (non-Javadoc)
    					 * @see com.jofti.util.ObjectProcedureAdapter#apply(java.lang.Object)
    					 */
    					public boolean apply(Object element) {
    						try {
    							NameSpaceKey entry = (NameSpaceKey) element;
    	                        indexKeys.add(entry.getKey());
    						}catch (Exception e){
    							// we should log this
    							log.warn("unable to remove key "+ element);
    						}
    						return true;
    					}
                    });
                  
                    // now update the actions
                    JBossIndexer indexer = getIndexer(tx);
                    registerIndexer(indexer, tx);
                    // see if key list bigger or smaller than node set

                    Set cacheKeys = cache._get(arg0).getDataKeys();

                    if (indexKeys.size() == cacheKeys.size()) {
                        // must be an update so all key/values will need
                        // reindexing
                        // worst cost activity
                        if (log.isDebugEnabled()) {
                            log.debug("Node key updated for " + arg0);
                        }
                        for (Iterator it = indexKeys.iterator(); it.hasNext();) {
                            Object tempKey = it.next();
                            if (log.isDebugEnabled()) {
                                log.debug("Node key removing " + tempKey);
                            }

                            NameSpaceKey tempWrapper = new NameSpaceKeyWrapper(
                                    arg0, tempKey);
                            indexer.remove(tempWrapper);
                            Object temp = cache.peek(arg0, tempKey);
                            if (log.isDebugEnabled()) {
                                log.debug("Node key adding " + tempKey);
                            }
                            indexer.add(tempWrapper, temp);
                        }

                    } else if (indexKeys.size() < cacheKeys.size()) {
                        //must be an add
                        // copy set keys just to make sure
                        if (log.isDebugEnabled()) {
                            log.debug("Node key added for " + arg0
                                    + " indexKeys:" + indexKeys + " cacheKeys"
                                    + cacheKeys);
                        }
                        Set tempSet = new HashSet(cacheKeys);
                        tempSet.removeAll(indexKeys);
                        // should be left with new key
                        for (Iterator it = tempSet.iterator(); it.hasNext();) {
                            Object tempKey = it.next();
                            Object temp = cache.peek(arg0, tempKey);
                            if (log.isDebugEnabled()) {
                                log.debug("Node key adding " + tempKey);
                            }
                            indexer.add(new NameSpaceKeyWrapper(arg0, tempKey),
                                    temp);
                        }
                    } else {
                        // otherwise must be a removal
                        if (log.isDebugEnabled()) {
                            log.debug("Node key removed for " + arg0);
                        }
                        indexKeys.removeAll(cacheKeys);
                        // should be left with new key
                        for (Iterator it = indexKeys.iterator(); it.hasNext();) {
                            Object tempKey = it.next();
                            if (log.isDebugEnabled()) {
                                log.debug("Node key removed " + tempKey);
                            }
                            indexer.remove(new NameSpaceKeyWrapper(arg0,
                                    tempKey));
                        }
                    }
                }

            } catch (Exception e) {
                log.warn("node modification failed to update index with node "
                        + arg0, e);
            }

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#nodeVisited(org.jboss.cache.Fqn)
         */
        public void nodeVisited(Fqn arg0)
        {
            // we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#cacheStarted(org.jboss.cache.TreeCache)
         */
        public void cacheStarted(TreeCache arg0)
        {
            // we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#cacheStopped(org.jboss.cache.TreeCache)
         */
        public void cacheStopped(TreeCache arg0)
        {
            //	we do not care

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.jboss.cache.TreeCacheListener#viewChange(org.jgroups.View)
         */
        public void viewChange(View arg0)
        {
            //	we do not care

        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see com.jofti.core.ITransactionAware#getTransactionLevel()
     */
    public TransactionLevel getTransactionLevel() throws JoftiException
    {
        if (cache.getIsolationLevelClass() == IsolationLevel.NONE) {
            return TransactionLevel.NONE;
        } else if (cache.getIsolationLevelClass() == IsolationLevel.READ_COMMITTED) {
            return TransactionLevel.READ_COMMITTED;
        } else if (cache.getIsolationLevelClass() == IsolationLevel.READ_UNCOMMITTED) {
            return TransactionLevel.READ_UNCOMMITTED;
        } else if (cache.getIsolationLevelClass() == IsolationLevel.REPEATABLE_READ) {
            return TransactionLevel.REPEATABLE_READ;
        } else {
            return TransactionLevel.SERIALIZABLE;
        }
    }

   

	protected Object getCacheValue(Object key) {
		return get(key);
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		
		return null;
	}
    
    
    boolean isRollback(int status) {
        return status==Status.STATUS_MARKED_ROLLBACK ||
            status==Status.STATUS_ROLLING_BACK ||
            status==Status.STATUS_ROLLEDBACK;
    }

}
