package com.jofti.cache.adapter;



import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;



import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



import com.jofti.api.IndexQuery;
import com.jofti.cache.CacheAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.util.CompositeComparator;

import edu.emory.mathcs.backport.java.util.concurrent.ConcurrentHashMap;


/**

 * 
 * The adapter implementation specific to the Map interface.
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of the Map.
 * 
 * Although it is possible to use this adapter with any Map implementation, it 
 * does not deal with the LinkedHashMap where it has been configured to run as 
 * an LRU cache. As values will dispear from the Map without callback to the index. This also applies to other Map 
 * type structures that silently expire elements.<p>
 * 
 * Care must be taken when using Maps with similar expiry functionality as it <b> will</b> lead to memory leaks.
 * 
 * 
 * The start up of the adapter will also try and index any elements already in
 * the Map.<p>
 * 
 * @author Steve Woodcock
 *         <p>
 * @version 1.0
 *          <p>
 */

public class MapAdapter extends BaseAdaptor implements CacheAdapter {

	 private static Log log =  LogFactory.getLog(MapAdapter.class);
	
    private Map map = new ConcurrentHashMap();

    private String name;
    
    


    
    public MapAdapter() {
    }

    public MapAdapter(Object cache) {
    	
    	this.map = (Map)cache;
    }
    
    public void setCacheImpl(Object cache){
    	this.map = (Map)cache;
    	
    }
    
    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#get(java.lang.Object)
     */
    public Object get(Object key) {

		Object result = null;
		if (key != null) {


				result = map.get(key);

		}
		return result;

	}

	

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.jofti.api.IndexCache#put(java.lang.Object, java.lang.Object)
	 */
	public void put(Object key, Object value) throws JoftiException{
		
		Comparable newKey = (Comparable)decorateKey(key);
		
		acquireUpdateLock();
		try{
			synchronized(getCacheLock(key))
			{			
				if (index.contains(newKey)){
	        		    index.removeByKey(newKey);
	        	 }
							
		    	// put the value back in
		        map.put(key,value);
		        index.insert(newKey, value);
		           
	      
			}
		} finally {
			releaseUpdateLock();
		}
    }

    /**
     * Removes the element which matches the key.
     * <p>
     * If no element matches, nothing is removed and no Exception is thrown.
     * @param key the key of the element to remove
     * @throws CacheException
     */
    public void remove(Object key) throws JoftiException {
    	Comparable newKey = (Comparable)decorateKey(key);

		acquireUpdateLock();
		try {
			synchronized (getCacheLock(key)) {
				map.remove(key);

				if (index.contains(newKey)) {
					index.removeByKey(newKey);
				}

			}

		} catch (Exception e) {
			throw new JoftiException(e);
		} finally {
			releaseUpdateLock();
		}

	}

    /**
	 * Remove all elements in the cache, but leave the cache in a useable state.
	 * 
	 * @throws CacheException
	 */
    public void removeAll() throws JoftiException{
    	acquireUpdateLock();
        try {
        	if (map != null){
        		map.clear();
        	}
        	if (index != null){
        		index.removeAll();
        	}
        } 
        catch (Exception e) {
            throw new JoftiException(e);
        } finally{
        	releaseUpdateLock();
        }
       
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException {
		
    	
		if (properties != null){
			String key = null;
			for (Iterator it = properties.keySet().iterator();it.hasNext();){
				key = (String)it.next();
				if (MUTABLE_VALUES.equalsIgnoreCase(key)){
					checkMutable = Boolean.valueOf(properties.getProperty(key)).booleanValue();
					if (log.isInfoEnabled()){
						log.info("Mutability checking is set to " + checkMutable);
					}
				}
			
			}
		}
	}
    
    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public void destroy() throws JoftiException {
    	
    	map = new ConcurrentHashMap();
    	
    	if (index != null){
    		index.removeAll();
    	}
    }

   

	/* (non-Javadoc)
	 * @see com.jofti.cache.LifeCycleAdapter#getName()
	 */
	public String getName() {
		return name;
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.LifeCycleAdapter#setName(java.lang.String)
	 */
	public void setName(String name) {
		this.name = name;
	}

	public String toString() {
		return "HashMapCache(" + getName() + ')';
	}
	

	/* (non-Javadoc)
	 * @see com.jofti.api.IndexCache#getCacheImpl()
	 */
	public Object getCacheImpl() {

		return map;
	}



	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
	 */
	public void setInternalIndex(InternalIndex index) {
		this.index = index;
		
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#start()
	 */
	public void start() throws JoftiException {

    		loadInitialValues(map);
    	
		
	}
    
	private void loadInitialValues(Map temp) throws JoftiException {
		if (map ==null){
			log.info("No initial values to index as map is null");
			return;
		}
		Set entries = temp.entrySet();
		 int size = entries.size();
	        Iterator it = entries.iterator();
	        for (int i=0;i<size;i++) {

			Map.Entry entry = (Map.Entry) it.next();

			if (entry.getKey() instanceof Comparable) {
				if (entry.getValue() != null) {
					// then we must remove from index
					index.insert((Comparable) entry.getKey(), entry.getValue());
				}
			} else {
				log.warn("unable to index value at " + entry.getKey()
						+ "as key is not Comparable");
			}
		}
	}
	
	

	
	
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getCacheValue(java.lang.Object)
	 */
	protected Object getCacheValue(Object key) {
        synchronized (getCacheLock(key)) {
            return map.get(key);
        }
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		return index;
	}
	
	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}

}