package com.jofti.cache.adapter;

import java.util.Iterator;
import java.util.Properties;

import net.sf.ehcache.CacheException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.IndexQuery;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.CacheAdapter;
import com.jofti.cache.adapter.listener.OSEventListener;
import com.jofti.core.InternalIndex;
import com.jofti.exception.JoftiException;
import com.jofti.util.PropertyLoader;
import com.opensymphony.oscache.base.NeedsRefreshException;
import com.opensymphony.oscache.base.events.CacheEntryEventListener;
import com.opensymphony.oscache.general.GeneralCacheAdministrator;

/**

 * 
 * The adapter implementation specific to OSCache.
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of OSCache.
 * 
 * For OSCache this means that the behaviour where a NeedsRefreshExeption is 
 * generated on a get is no longer visible to the caller. Instead, the entry is flushed 
 * as it has expired. <p>
 * 
 * OSCache has a full set of callbacks for the object in its cache so updates to the 
 * index are driven by these callbacks.<p>
 * 
 * On start up this adpater will <b>not</b> try and index existing elements in the cache as OSCache provides no way to obtain a set of exsting 
 * keys.<p>
 * 
  * @author Steve Woodcock
 *         <p>
 * @version 1.0
 *          <p>
 */


public class OSCacheAdapter extends BaseAdaptor implements CacheAdapter {

	private GeneralCacheAdministrator admin = null;
	
	
	private static Log log =  LogFactory.getLog(OSCacheAdapter.class);
	
    private String name;
    

    
    private int expiryTime =3600; //seconds
    
    private OSEventListener eventListener =null;
	/**
	 * @return Returns the expiryTime.
	 */
	public int getExpiryTime() {
		return expiryTime;
	}
	/**
	 * @param expiryTime The expiryTime to set.
	 */
	public void setExpiryTime(int expiryTime) {
		this.expiryTime = expiryTime;
	}
    public OSCacheAdapter() {
    }

    public OSCacheAdapter(Object cache) {
    	if (! (cache instanceof GeneralCacheAdministrator) ){
    		throw new RuntimeException("OSCacheAdapter requires GeneralCacheAdministrator ");
    	}
     	this.admin = (GeneralCacheAdministrator)cache;
    }
    
    public void setCacheImpl(Object cache){
    	this.admin = (GeneralCacheAdministrator)cache;
    	
    }
    
    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#get(java.lang.Object)
     */
    public Object get(Object key) {
		if (key != null && (!(key instanceof String))) {
			log.warn("key value for OSCache must by of type String " + key);
			return null;
		}
		Object obj = null;
		synchronized (getCacheLock(key)) {
			try {
				obj = admin.getFromCache((String) key, expiryTime);
			} catch (NeedsRefreshException nre) {
				// we do this because of the bizarre way OSCache throws
				// exceptions for gets
				admin.cancelUpdate((String) key);
				// it has expired so lets flush the entry
				admin.flushEntry((String) key);

			}
		}
    	// we return null here if we got an exception because it is consistent
    	// with most other cache implementations
       return obj;
    }

	

	/* (non-Javadoc)
	 * @see com.jofti.api.IndexCache#put(java.lang.Object, java.lang.Object)
	 */
	public void put(Object key, Object value) throws JoftiException {

		if (key != null && (!(key instanceof String))) {
			throw new JoftiException(
					"key value for OSCache must by of type String " + key);
		}

		// get a lock for the key that we are trying to add
		// we do not care about other parts of the cache or the index
		synchronized (getCacheLock(key)) {

			// rely on call back listener to populate index entry
			admin.putInCache((String) key, value);

		}

	}

    /**
	 * Removes the element which matches the key.
	 * <p>
	 * If no element matches, nothing is removed and no Exception is thrown.
	 * 
	 * @param key
	 *            the key of the element to remove
	 * @throws CacheException
	 */
    public void remove(Object key) throws JoftiException{
    	
	    	// remove this entry and rely on call back listener to remove index entry
	    	if (key != null && (!(key instanceof String))){
	    		throw new JoftiException("key value for OSCache must by of type String "+ key);
	    	}
	       admin.flushEntry((String)key);
	   
    }

    /**
     * Remove all elements in the cache, but leave the cache
     * in a useable state.
     * @throws CacheException
     */
    public void removeAll() throws JoftiException{
    	//rely on call back listener to remove index entry
       admin.flushAll();
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException {
    	try {
    		
	    	
	    	
	    		String cacheConfigFile = null;
				if (properties != null){
					String key =null;
					for (Iterator it = properties.keySet().iterator();it.hasNext();){
						key = (String)it.next();
						if (MUTABLE_VALUES.equalsIgnoreCase(key)){
							checkMutable = Boolean.valueOf(properties.getProperty(key)).booleanValue();
							if (log.isInfoEnabled()){
								log.info("Mutability checking is set to " + checkMutable);
							}
						}
						if ("file".equalsIgnoreCase(key)){
							cacheConfigFile = properties.getProperty(key);
						}				
					}
				}
				log.debug("looking up GeneralCacheAdministrator");
				if (admin == null){
				
			    	if (cacheConfigFile == null){
			    		log.debug("not found - creating default GeneralCacheAdministrator");
						admin = new GeneralCacheAdministrator();
					}else{
						// we need to construct a properties object
						log.debug("not found - creating default GeneralCacheAdministrator");
						Properties props  =null;
						try {
							props= PropertyLoader.loadProperties(cacheConfigFile);
						} catch (Throwable t){
							log.warn("Unable to load config file " + cacheConfigFile + " from classpath");
						}
						if (props == null){
							throw new JoftiException("Unable to load config file " + cacheConfigFile + " from classpath"); 
						}
						admin = new GeneralCacheAdministrator(props);
					}
				}else{
					log.debug("using supplied GeneralCacheAdministrator");
				}
				if (properties != null){
			    	String expiryParam = properties.getProperty("expiryTime");
			    	if(expiryParam != null){
			    		try {
			    			expiryTime = Integer.parseInt(expiryParam);
			    		}catch (NumberFormatException nfe){
			    			log.error("Unable to set expiry time - using default of " + expiryTime,nfe);
			    		}
			    	}
				}
	    	
	    	
	        eventListener = new OSEventListener(this);
	        
	    	admin.getCache().addCacheEventListener(eventListener,CacheEntryEventListener.class);

    	} catch (Exception e){
    		throw new JoftiException(e);
    	}
		
		
	}
    
    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public void destroy() throws JoftiException {
    	  if (admin != null) {
    	  	 admin.destroy();
           
        }
    	  if (index != null){
    	  	index.removeAll();
    	  }
      
    }

   

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String toString() {
		return "OSCache(" + getName() + ')';
	}
	
	

	/* (non-Javadoc)
	 * @see com.jofti.api.IndexCache#getCacheImpl()
	 */
	public Object getCacheImpl() {
		return admin;
	}

	

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
	 */
	public void setInternalIndex(InternalIndex index) {
		this.index = index;
		
	}
	
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#start()
	 */
	public void start() throws JoftiException {

    //		do nothing here
    	
		
	}
	
	

	protected Object getCacheValue(Object key) {
		return get(key);
	}
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		return index;
	}
	
	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
}