package com.jofti.cache.adapter;

import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.api.CacheAccessor;
import com.jofti.api.IndexQuery;
import com.jofti.cache.CacheListenerAdapter;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.adapter.listener.OSEventListener;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.exception.JoftiException;
import com.jofti.util.CompositeComparator;
import com.opensymphony.oscache.base.Cache;
import com.opensymphony.oscache.base.NeedsRefreshException;
import com.opensymphony.oscache.base.events.CacheEntryEventListener;

/**

 * 
 * The adapter implementation specific to OSCache that acts as a Listener to the Cache.
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of OSCache.
 * 
 * For OSCache this means that the behaviour where a NeedsRefreshExeption is 
 * generated on a get is no longer visible to the caller. Instead, the entry is flushed 
 * as it has expired. <p>
 * 
 * Similarly, a put will also overwrite the previous value - making the first expired if it needs refreshing.<p>
 * 
 * OSCache has a full set of callbacks for the object in its cache so updates to the 
 * index are driven by these callbacks.<p>
 * 
 * On start up this adpater will <b>not</b> try and index existing elements int he cache as OSCache provides no way to obtain a set og exsting 
 * keys.<p>
 * 
  * @author Steve Woodcock
 *         <p>
 * @version 1.0
 *          <p>
 */


public class OSCacheListenerAdapter extends BaseAdaptor implements CacheAccessor, CacheListenerAdapter {

	private Cache cache = null;
	
	
	private static Log log =  LogFactory.getLog(OSCacheListenerAdapter.class);
	
    private String name;
    

    
    private int expiryTime =3600; //seconds
    
    private OSEventListener eventListener =null;
	/**
	 * @return Returns the expiryTime.
	 */
	public int getExpiryTime() {
		return expiryTime;
	}
	/**
	*/
    public OSCacheListenerAdapter() {
    }

    public OSCacheListenerAdapter(Object cache) {
    	if (!(cache instanceof Cache)){
    		throw new RuntimeException("Cache is not instance of "+ Cache.class.getName());
    	}
     	this.cache = (Cache)cache;
    }
    
  
 

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException {
    	try {
    		
				if (properties != null){
					String key =null;
					for (Iterator it = properties.keySet().iterator();it.hasNext();){
						key = (String)it.next();
						if (MUTABLE_VALUES.equalsIgnoreCase(key)){
							checkMutable = Boolean.valueOf(properties.getProperty(key)).booleanValue();
							if (log.isInfoEnabled()){
								log.info("Mutability checking is set to " + checkMutable);
							}
						}
									
					}
					  if ("file".equalsIgnoreCase(key)) {
	                    log.warn("Listener adapters cannot be used to start up a Cache - please use addCache method");
	                    throw new JoftiException("Listener adapters cannot be used to start up a Cache - please use addCache method");
	                }
				}
				 if (cache == null) {
		            throw new JoftiException("Cache cannot be null for listener adapter");
		        } else {
		        	log.info("Index started using cache "+ cache);
		        }
				 // fix for missing expiry time config 
				 if (properties != null){
				    	String expiryParam = properties.getProperty("expiryTime");
				    	if(expiryParam != null){
				    		try {
				    			expiryTime = Integer.parseInt(expiryParam);
				    		}catch (NumberFormatException nfe){
				    			log.error("Unable to set expiry time - using default of " + expiryTime,nfe);
				    		}
				    	}
					}
	        eventListener = new OSEventListener(this);
	        
	    
    	} catch (Exception e){
    		throw new JoftiException(e);
    	}
		
		
	}
    
    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public void destroy() throws JoftiException {
    	
    	  index.removeAll();
      
    }

   

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String toString() {
		return "OSCache(" + getName() + ')';
	}
	
	

	/* (non-Javadoc)
	 * @see com.jofti.api.IndexCache#getCacheImpl()
	 */
	public Object getCacheImpl() {
		return cache;
	}

	
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
	 */
	public void setInternalIndex(InternalIndex index) {
		this.index = index;
		
	}
	
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheAdapter#start()
	 */
	public void start() throws JoftiException {

		if (cache != null){
			cache.addCacheEventListener(eventListener,CacheEntryEventListener.class);			
		}
    	
		
	}
	


	

	protected Object getCacheValue(Object key) {
		Object obj=null;
		try {
    		obj =cache.getFromCache((String)key, expiryTime);
    	}catch (NeedsRefreshException nre){
    		//we do this because of the bizarre way OSCache throws exceptions for gets
    		cache.cancelUpdate((String)key);
    		// it has expired so lets flush the entry
    		cache.flushEntry((String)key);
           
    	}
		return obj;
	}
	
	public void warnOutOfDate(Object key){
		if (log.isWarnEnabled()) {
			log
					.warn("Index and cache have become out of date for key - Probable cause element expiry - flushing entry for '"
							+ key +"'");
		}
	}
	
	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		return index;
	}
	
	/* (non-Javadoc)
	 * @see com.jofti.api.IndexedListener#setCacheImpl(java.lang.Object)
	 */
	public void setCacheImpl(Object cache) {
		this.cache = (Cache)cache;
		this.cache.addCacheEventListener(eventListener,CacheEntryEventListener.class);
		
	}
	
	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}
}