package com.jofti.cache.adapter;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.websphere.objectgrid.BackingMap;
import com.ibm.websphere.objectgrid.IndexAlreadyDefinedException;
import com.ibm.websphere.objectgrid.IndexUndefinedException;
import com.ibm.websphere.objectgrid.ObjectGridException;
import com.ibm.websphere.objectgrid.ObjectMap;
import com.ibm.websphere.objectgrid.Session;
import com.ibm.websphere.objectgrid.TxID;
import com.ibm.websphere.objectgrid.plugins.index.DynamicIndexCallback;
import com.jofti.api.CacheAccessor;
import com.jofti.api.IndexQuery;
import com.jofti.cache.BaseAdaptor;
import com.jofti.cache.CacheListenerAdapter;
import com.jofti.cache.adapter.listener.ObjectGridEventListener;
import com.jofti.core.IParsedQuery;
import com.jofti.core.InternalIndex;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ClassFieldMethods;
import com.jofti.util.CompositeComparator;
import com.jofti.util.ObjectProcedureAdapter;
import com.jofti.util.OpenHashMap;
import com.jofti.util.ValueTreeMap;
import com.tangosol.net.NamedCache;
import com.tangosol.util.MapEvent;

/**

 * 
 * The adapter implementation specific to ObjectGrid.</p>
 * 
 * The adapter takes care of the implementation specific details for converting
 * between the process the index expects and the behaviour of ObjectGrid.</p>
 * 
 * The Listener adapter is for use as a listener to ObjectGrid and does not provide the get, set, remove methods 
 * that the wrapper adapter provides. The addition, removal and getting of values from the Cache must be 
 * done on the Cache implementation directly. This is the preferred way of using an already existing Cache.</p>
 * 
 * The start up of the adapter will also try and index any elements already in
 * the cache.<P>
 * 
 * @author Steve Woodcock
 *         <p>
 * @version 1.0
 * @since 1.1
 */
public class ObjectGridListenerAdapter extends BaseAdaptor implements CacheAccessor, CacheListenerAdapter
{


    private static Log                  log = LogFactory
                                                    .getLog(ObjectGridListenerAdapter.class);


    private String                      name;

    public InternalIndex               index;

    private ObjectGridEventListener eventListener = null;
    
    
    private BackingMap cache =null;

    private Object loadLock = new Object();
    
    private boolean loaded =true;

    private boolean dynamicIndex =false;
    
    private boolean plugIn =false;
    
    public ObjectGridListenerAdapter()
    {
    }

    public ObjectGridListenerAdapter(Object cache)
    {
        this.cache = (BackingMap) cache;

    }
    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#init(java.util.Properties)
     */
    public synchronized void init(Properties properties) throws JoftiException
    {
        if (properties != null) {
            String key = null;
            for (Iterator it = properties.keySet().iterator(); it.hasNext();) {
                key = (String) it.next();
                if (MUTABLE_VALUES.equalsIgnoreCase(key)) {
                    checkMutable = Boolean.valueOf(
                            properties.getProperty(key)).booleanValue();
                    if (log.isInfoEnabled()) {
                        log.info("Mutability checking is set to "
                                + checkMutable);
                    }
                }
                if ("file".equalsIgnoreCase(key)) {
                    log.warn("Listener adapters cannot be used to start up a Cache - check docs for Adapter Type");
                    throw new JoftiException("Listener adapters cannot be used to start up a Cache -  check docs for Adapter Type");
                }
                if ("plugin".equalsIgnoreCase(key)) {
                    log.info("Listener used as plug-in");
                    plugIn =true;
                }
            }
        }

     
        if (cache == null) {
            throw new JoftiException("Cache cannot be null for listener adapter");
        } else {
        	    log.info("Index started using cache "+ cache.getName());
        }
        eventListener = new ObjectGridEventListener(this, name);
    
    }

    
 
   

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#destroy()
     */
    public synchronized void destroy() throws JoftiException
    {
    	
        try {
            cache.removeMapEventListener(eventListener);
        } catch (IllegalStateException e) {
        	log.warn("Unable to remove event listener:",e);
        }
            if (!dynamicIndex){
	            List indexes = cache.getMapIndexPlugins();
	            for(int i=0;i<indexes.size();i++){
	            	if (indexes.get(i) == eventListener){
	            		indexes.remove(i);
	            	}
	            }
            }else{          
            		try {
						cache.removeDynamicIndex(name);
					} catch (IndexUndefinedException e) {
						log.warn("Unable to remove dynamic Index:",e);
					} catch (IllegalArgumentException e) {
						
						log.warn("Unable to remove dynamic Index:",e);
					}
            	
            }
           if (index != null){
                index.removeAll();
           }
           log.info("Removed index from cache:"+name);
        
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#getName()
     */
    public String getName()
    {
        return name;
    }

    /* (non-Javadoc)
     * @see com.jofti.cache.LifeCycleAdapter#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    public String toString()
    {
        return "ObjectGridCache(" + getName() + ')';
    }

  


    /*
     * (non-Javadoc)
     * 
     * @see com.jofticache.CacheAdapter#setInternalIndex(com.jofti.core.InternalIndex)
     */
    public void setInternalIndex(InternalIndex index)
    {
        this.index = index;

    }

    /* (non-Javadoc)
     * @see com.jofti.api.IndexCache#query(com.jofti.api.IndexQuery)
     */
    public Map query(IndexQuery query) throws JoftiException
    {
    	
    	while (!loaded){
    		synchronized (loadLock) {
	    		try {
	    			loadLock.wait();
	    		}catch (InterruptedException e){
	    			
	    		}
    		}
    	}
    	Map temp = null;
    	
    	query = processQuery(query,index.getParserManager());
       
    	acquireQueryLock();
		try {
        
        temp=index.query(query);
    	} finally {
			releaseQueryLock();
		}
    	return getCacheValues(temp, (IParsedQuery)query,index.getIntrospector());

    }

	

    /* (non-Javadoc)
     * @see com.jofti.cache.CacheLocking#checkMutable(java.lang.Object, java.lang.Object)
     */
    protected Object checkMutable(Object key, Object result)
    {
        try {
            // first parse the object - again
            Map cacheObjectValues = index.getIntrospector().getAttributeValues(
                    result);

            Map indexObjectValues = index.getAttributesByKey(decorateKey( key));
            
            if (cacheObjectValues.equals(indexObjectValues)) {
                return result;
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Object under Key " + key
                            + " -  attributes changed without re-insert");
                }
            }
            // then get the values for the key

        } catch (JoftiException e) {
            log.warn("Error checking mutability", e);
        }

        return null;
    }

	/**
	 * @param cache
	 */
	public void setCacheImpl(Object cache){
		ObjectGridListenerAdapter.this.cache = (BackingMap)cache;
		
		
	}
	
	public Map getCacheValues(Map col, final IParsedQuery query,
			final ClassIntrospector introspector) throws JoftiException {

		
		final Map returnClasses = query.getResultFieldsMap();
		final CompositeComparator comp = query.getOrderingComparator();
		final int maxResults = query.getMaxResults();
		final int startEntry = query.getFirstResult();
		
		Map interim = null;
		if (comp ==null || comp.getSize() ==0){
			interim =new HashMap(col.size() + 2, 1.00f);
		}else{
			interim = new ValueTreeMap(comp);
		}
		
		// return map
		final Map temp = interim;
		
		

		// the results should always be an OpenHashMap
		OpenHashMap originalMap = (OpenHashMap) col;

		Session session =null;
		// get the session that we aredoing our stuff in
		try {
		 session= cache.getObjectGrid().getSession();
		
		
		
		final ObjectMap map =session.getMap(cache.getName());
		
		
		session.begin();
		// populate the return Map
		originalMap.forEachPair(new ObjectProcedureAdapter() {
			
			public boolean apply(Object key, Object value) throws RuntimeException{
				
				//this will be an alias or null or it is not in the return set and we can ignore it
				if (value == null || returnClasses ==null || returnClasses.containsKey(value)) {
					// see if we have the class in the return map

					// strip the key of wrapper if it is noncomparable
					key = stripKey(key);

					Object result = null;
					try {
						result =getCacheValue(map, key);
					} catch (Exception e) {
						throw new RuntimeException(e);
					}

					if (result == null) {
						if (log.isWarnEnabled()) {
							log
									.warn("Index and cache have become out of date for key "
											+ key);
						}
					} else {
						// get the methods to be operated on the value
						if (checkMutable) {
							result = checkMutable(key, result);
							// mutability check failed
							if (result == null) {
								if (log.isDebugEnabled()) {
									log
											.debug("Object under key:"
													+ key
													+ " has changed in cache since it was indexed");
								}
								return true;
							}
						}
						
						
						if (returnClasses != null) {
							// we need to get the value from result if we are using fields
							ClassFieldMethods fieldSet = (ClassFieldMethods) returnClasses
									.get(value);

							Map tempMap = fieldSet != null ? fieldSet
									.getFieldMap() : null;

							if (tempMap != null && tempMap.size() > 0) {
								// get the fields and apply the values to them
				
									Map resultMap = new LinkedHashMap();
				
									resultMap = introspector.getResultValuesFromObject(resultMap,result,tempMap);
									
									// no values means no mapping in results
									if (resultMap.size()>0){
										Object[] res = new Object[resultMap.size()];
										Iterator it = resultMap.entrySet().iterator();
										for (int i = 0;i<resultMap.size();i++){
											Map.Entry entry = (Map.Entry)it.next();
											res[i]= entry.getValue();
										}
										temp.put(key,res);
									}
									
							} else {
								// there are no fields just return the value
								temp.put(key, result);
							}
						} else {
							temp.put(key, result);
						}
					}

				}
				return true;
			}
		});

		} catch(Exception e){
			throw new JoftiException(e);
		}finally{
			if (session != null){
				try {
					if(log.isDebugEnabled()){
						log.debug("finished search: terminating search tx "+session.getTxID());
					}
				session.commit();
				} catch(Exception e){
					log.warn("unable to close transaction for query", e);
				}
			}
		}
		// now limit result size
		if (maxResults >0 || startEntry >0){
			return limitResults(temp, startEntry, maxResults);
		}else if (startEntry <0 || maxResults <0){
			if (startEntry <0){
				throw new IllegalArgumentException("startResult cannot be less than 0:"+startEntry); 
			}
			if (maxResults <0){
				throw new IllegalArgumentException("maxResults cannot be less than 0:"+maxResults); 
			}
		}
		
		
		return temp;
	}

  	protected Object getCacheValue(ObjectMap map, Object key) throws ObjectGridException{
		
		return map.get(key);
		
	}

	/* (non-Javadoc)
	 * @see com.jofti.cache.LifeCycleAdapter#start()
	 */
	public void start() throws JoftiException {
		if (cache != null){			
			addListener();          
	        
		}
		
		
		
	}

	private void addListener() throws JoftiException {
		cache.addMapEventListener(eventListener);
		
		if (!plugIn){
			try {
				cache.addMapIndexPlugin(eventListener);
			} catch (IllegalStateException e) {
				//add the dynamic listener
				loaded =false;
				try {
					cache.createDynamicIndex(eventListener, new DynamicListener());
					dynamicIndex =true;
					// load the initial values
				//	loadInitialValues();
				} catch (Exception e1) {
					throw new JoftiException(e1);
				}
				
			} catch (IndexAlreadyDefinedException iae){
				throw new JoftiException(iae);
			}
		}
	}

	class DynamicListener implements DynamicIndexCallback{

		public void destroy(String s) {
			log.info("dynamic index destroyed:"+s);
			
		}

		public void error(String s, Throwable throwable) {
			log.error(s, throwable);
			synchronized (loadLock) {
				loaded =true;
				loadLock.notifyAll();
			}
			
			
		}

		public void ready(String s) {
			log.info("dynamic index ready:"+s);
			synchronized (loadLock) {
				loaded =true;
				loadLock.notifyAll();
			}
			
			
		}
		
	}
	 

	/* (non-Javadoc)
	 * @see com.jofti.cache.CacheLocking#getIndex()
	 */
	public InternalIndex getIndex() {
		
		return index;
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.CacheAccessor#getCacheImpl()
	 */
	public Object getCacheImpl() {
		
		return cache;
	}

	public IndexQuery addQuery(String name, IndexQuery query)throws JoftiException {
		
		return index.getParserManager().addQuery(name, query);
	}

	/* (non-Javadoc)
	 * @see com.jofti.api.Index#getQuery(java.lang.String)
	 */
	public IndexQuery getQuery(String name)  {
		
		return index.getParserManager().getQuery(name);
	}

	protected Object getCacheValue(Object key) {
		// intentional no-op
		return null;
	}

	public ObjectGridEventListener getEventListener() {
		return eventListener;
	}
	
}