/*
 * Created on 31-Mar-2004
 *
 */
package com.jofti.config;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import com.jofti.exception.JoftiException;

/**

 * 
 * The parser is responsible for parsing the XMl configuration file and returns a Map of IndexConfig objects (one for each index in the file).<p>
 * The current implementation uses KXML as its parser (as it is small and we probably will not get any version interference with 
 * other XML parsers).
 *
  * @author Steve Woodcock (steve@jofti.com)<p>
  * @version 1.0
 */
public class ConfigFileParser {
	
	
 /**
  * Parses the config file from the file name. The method uses getResourceAsStream() so the filename must be 
  * on the classpath.<p>
  * 
 * @param configFile
 * @return Map of IndexConfig objects.
 * @throws JoftiException
 */
public Map parseIndexConfig(String configFile) throws JoftiException
 {
 	
 	Map temp =null;
 	
	InputStream stream = null;
	
	try {
		stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(configFile);

		if (stream == null){
			throw new JoftiException("Unable to find config file '" + configFile +"' on classpath - ensure the file name is correctly configured");
		}
		temp= parseConfig(stream);
	
	} catch (JoftiException e) {
		e.printStackTrace();
	} catch(Throwable t){
		t.printStackTrace();
	}finally{
	
		if (stream != null){
			try{
				stream.close();
			} catch(Exception e){
				// we do not care about this
			}
		}
	}
	return temp;
}

/**
 * Parses the config file from an inputstream. The method allows calling code to load the configfile however it wants and 
 * pass the location independent stream containing the file. 
 * on the classpath.<p>
 * 
* @param stream
* @return Map of IndexConfig objects.
* @throws JoftiException
*/
 public Map parseIndexConfig(InputStream stream) throws JoftiException
 {
 	
 	Map temp =null;
 	
	
	try {
	
		if (stream == null){
			throw new JoftiException("InputStream is null ensure the file name is correctly configured");
		}
		temp= parseConfig(stream);
	
	} catch (JoftiException e) {
		e.printStackTrace();
	} catch(Throwable t){
		t.printStackTrace();
	}finally{
	
		if (stream != null){
			try{
				stream.close();
			} catch(Exception e){
				// we do not care about this
			}
		}
	}
	return temp;
}
 
 
 private XmlPullParser getParser() throws Exception{
	 return XmlPullParserFactory.newInstance("org.kxml2.io.KXmlParser",Thread.currentThread().getContextClassLoader().getClass()).newPullParser();

 }
 
 
 private Map parseConfig(InputStream stream) throws JoftiException
 {
 	Map indexMap = new HashMap();
 	
 	

	XmlPullParser parser = null;
	
	try {
	
		if (stream == null){
			throw new JoftiException("InputStrean is null - ensure the file name is correctly configured");
		}
		
		parser = getParser();

		parser.setInput(stream, null);

		int eventType = parser.getEventType();
		do {
			if(eventType == XmlPullParser.START_DOCUMENT) {
				
					indexMap = processIndexes(parser);
				
			}
			
			if (parser.getEventType() !=XmlPullParser.END_DOCUMENT ){
				eventType = parser.next();
			}
		}while (eventType != XmlPullParser.END_DOCUMENT);		
		parser.setInput (null);
	} catch (JoftiException e) {
		
		e.printStackTrace();
	} catch (XmlPullParserException e) {
		
		e.printStackTrace();
	} catch (IOException e) {
		
		e.printStackTrace();
	} catch(Throwable t){
		t.printStackTrace();
	}finally{
	
		if (stream != null){
			try{
				stream.close();
			} catch(Exception e){
				// we do not care about this
			}
		}
	}


    return indexMap;
}

 
 private Map processIndexes(XmlPullParser parser) throws XmlPullParserException, IOException, JoftiException{
 	
 	
 	Map temp = new HashMap();
 	//get the next tag - this should be indexes
 	while(true){
	 	int tagType = parser.nextTag();
	 	
	 	if (tagType == XmlPullParser.START_TAG){
		 	String name = parser.getName();
		 	
		 	if ("indexes".equalsIgnoreCase(name)){
			 	if (!parser.isEmptyElementTag()){
			 		// the next one should be an index tag
			 		tagType = parser.nextTag();
			 		// should be index
			 		while ( tagType == XmlPullParser.START_TAG && "index".equalsIgnoreCase(parser.getName())){
			 			IndexConfig config = processIndex(parser);
			 			temp.put(config.getName(),config);
			 			parser.nextTag();
			 		}
			 		
			 	}else{
			 		throw new JoftiException("Indexes definition in config file is empty");
			 	}
		 	}else{
		 		throw new JoftiException("Unrecognised tag " + name + " at root of document");
		 	}
	 	}else if (tagType == XmlPullParser.END_TAG){
	 		break;
	 	}
	 	
	 	//parser.nextTag();
	 	parser.require(XmlPullParser.END_TAG,null,"indexes");
	 	break;
	 }
 	return temp;
 }
 
  
 IndexConfig processIndex(XmlPullParser parser)
    throws IOException, XmlPullParserException,JoftiException {
    
 	
 	
 	DefaultIndexConfig config = new DefaultIndexConfig();
		
 	String name = parser.getName();
 	
 	
 	if ("index".equalsIgnoreCase(name)){
 			
 			 //process the attributes for the index tag
 		    int numAttributes = parser.getAttributeCount();
 		    if (numAttributes >= 0){
 		    	for (int i=0;i<numAttributes;i++){
 		    		String attributeName = parser.getAttributeName(i);
 		    		if (attributeName.equalsIgnoreCase("name")){
 		    			config.setName(parser.getAttributeValue(i));
 		    		} else
 		    		if (attributeName.equalsIgnoreCase("init-method")){
 		    			//config.setName(parser.getAttributeValue(i));
 		    		} else
 		    		if (attributeName.equalsIgnoreCase("destroy-method")){
 		    			//config.setName(parser.getAttributeValue(i));
 		    		}
 		    		else {
 		    			//log warning here
 		    		}
 		    	}
 		    }	
 		    
 		    // now we have either cache/classes/type
 		    int event = parser.nextTag();

 		   name= parser.getName();
 		    while ((event == XmlPullParser.START_TAG) && ("cache".equalsIgnoreCase(name) ||
 		    		"classes".equalsIgnoreCase(name) ||"parser".equalsIgnoreCase(name) || "type".equalsIgnoreCase(name)
					|| "disk-overflow".equalsIgnoreCase(name)
					|| "queries".equalsIgnoreCase(name)))
 		    {
 		    	
 		    
	 			if (name.equalsIgnoreCase("cache")){
	 				//see if empty
	 				config = processCache(parser,config);
	 				parser.require(XmlPullParser.END_TAG,"","cache");
	 	      		
	 	      	}else if (name.equalsIgnoreCase("classes")){
	 	      		config = (DefaultIndexConfig)processClasses(parser, config);
	 	      		parser.require(XmlPullParser.END_TAG,"","classes");
	 	      		
	 	      	}else if (name.equalsIgnoreCase("queries")){
	 	      		config = (DefaultIndexConfig)processQueries(parser, config);
	 	      		parser.require(XmlPullParser.END_TAG,"","queries");
	 	      		
	 	      	}else if (name.equalsIgnoreCase("parser")){
	 	      		config.setParserType(parser.nextText().trim());
	 	      		parser.require(XmlPullParser.END_TAG,"","parser");
	 	      		
	 	      	}else if (name.equalsIgnoreCase("type")){
	 	      		config.setIndexType(parser.nextText().trim());
	 	      		parser.require(XmlPullParser.END_TAG,"","type");
	 	      	}else if (name.equalsIgnoreCase("disk-overflow")){
	 	      		config = (DefaultIndexConfig)processOverflow(parser, config);
	 	      		parser.require(XmlPullParser.END_TAG,"","disk-overflow");
	 	      	}
	 		    
	 			event = parser.nextTag();
	 			name= parser.getName();
 		    
 		}
 		
 		parser.require(XmlPullParser.END_TAG, "", "index");
 		
 	}
 	return config;
 }
 
 private DefaultIndexConfig processCache(XmlPullParser parser, DefaultIndexConfig config) 
 throws IOException, XmlPullParserException, JoftiException{
 	
			// we may have an adapter here
			int event = parser.nextTag();
			String name = parser.getName();
			if(name.equalsIgnoreCase("adapter")){
	       		int adapterAttribs = parser.getAttributeCount();
	         	for (int i=0;i<adapterAttribs;i++){
	         		String temp = parser.getAttributeName(i);
	         		if (temp.equalsIgnoreCase("init-method")){
	         			//config.setCacheInitMethod(parser.getAttributeValue(i));
	         		} else
	         		if (temp.equalsIgnoreCase("destroy-method")){
	         			//config.setCacheDestroyMethod(parser.getAttributeValue(i));
	         		} else {
	         			//log warning here
	         		}
	         	}
	         	//now see if we have a type or an end tag
	         	event = parser.nextTag();
	         	while (event == XmlPullParser.START_TAG){
	         		//must be the type tag
	         		name = parser.getName();
	         		if (name.equalsIgnoreCase("type")){
	         			String content = parser.nextText();
	         			config.setCacheAdapter(content.trim());
		         		parser.require(XmlPullParser.END_TAG,"","type");
		         		event = parser.nextTag();
		         		name = parser.getName();
	         		}
	         		if (name.equalsIgnoreCase("init-properties")){
	         			Properties props = new Properties();
	         			if (!parser.isEmptyElementTag()){
	         				//loop through attributes
	         				while (parser.nextTag() == XmlPullParser.START_TAG){
	         					name = parser.getName();
	         					if (name.equalsIgnoreCase("property")){
	         						String temp = parser.getAttributeValue(0);
	         						if (temp != null){
	         							props.put(temp, parser.nextText().trim());
	         						}
	         					} 
	         					parser.require(XmlPullParser.END_TAG,"",name);
	         				}
	         				config.setAdapterProperties(props);
	         				//assert the
	         				parser.require(XmlPullParser.END_TAG,"","init-properties");
	         				event = parser.nextTag();
	         				
	         			}
	         			
	         		}	
	         	}
	         	
	         	parser.require(XmlPullParser.END_TAG,"","adapter");
	         	parser.nextTag();
			} 

			return config;
 		}

 
 private IndexConfig processOverflow(XmlPullParser parser, IndexConfig config) 
 throws IOException, XmlPullParserException, JoftiException{
 	
 	Properties props = config.getIndexProperties();
	         	//now see if we have a type or an end tag
	         	int event = parser.nextTag();
	         	while (event == XmlPullParser.START_TAG){
	         		//must be the type tag
	         		String name = parser.getName();
	         		if (name.equalsIgnoreCase("provider")){
	         			String content = parser.nextText();
	         			if (content !=null){
	         				content = content.trim();
	         			}
	         			props.put("provider",content);
		         		parser.require(XmlPullParser.END_TAG,"","provider");
		         		event = parser.nextTag();
		         		name = parser.getName();
	         		}
	         		
	         		if (name.equalsIgnoreCase("init-properties")){
	         			
	         			if (!parser.isEmptyElementTag()){
	         				//loop through attributes
	         				while (parser.nextTag() == XmlPullParser.START_TAG){
	         					name = parser.getName();
	         					if (name.equalsIgnoreCase("property")){
	         						String temp = parser.getAttributeValue(0);
	         						if (temp != null){
	         							props.put(temp, parser.nextText().trim());
	         						}
	         					} 
	         					parser.require(XmlPullParser.END_TAG,"",name);
	         				}
	         				
	         				//assert the
	         				parser.require(XmlPullParser.END_TAG,"","init-properties");
	         				event = parser.nextTag();
	         				
	         			}
	         			
	         		}	
	         	}
	         	
	       

			return config;
 		}
 private IndexConfig processClasses(XmlPullParser parser, IndexConfig config) 
 throws IOException, XmlPullParserException, JoftiException{
 	
 	Map temp = new HashMap();
 	//loop  through classes
 	while (parser.nextTag() == XmlPullParser.START_TAG){
 		String name = parser.getName();
 		if (name.equalsIgnoreCase("class")){
 			List list = new ArrayList();
 			String className = parser.getAttributeValue(0);
 			if (!parser.isEmptyElementTag()){
 				;
 				//loop through attributes
 				while (parser.nextTag() == XmlPullParser.START_TAG){
 					name = parser.getName();
 					if (name.equalsIgnoreCase("property")){
 						list.add(parser.nextText());
 					}
 					parser.require(XmlPullParser.END_TAG,"",name);
 				}
 				//assert the
 				parser.require(XmlPullParser.END_TAG,"","class");
 				temp.put(className,list);
 			}
 			config.addMapping(className,list);
 		}
 	}
 	
 	return config;
 }
 
 private IndexConfig processQueries(XmlPullParser parser, IndexConfig config) 
 throws IOException, XmlPullParserException, JoftiException{
 	
 	//loop  through queries
 	while (parser.nextTag() == XmlPullParser.START_TAG){
 		String name = parser.getName();
 		if (name.equalsIgnoreCase("query")){
 			String queryText = null;
 			String queryName = parser.getAttributeValue(0);
 				
 				queryText = parser.nextText().trim();
 				
 				config.addQuery(queryName,queryText);
 				parser.require(XmlPullParser.END_TAG,"","query");
 		}
 	}
 	
 	return config;
 }
 /**
  * Used to parse only the classmappings fragment of an xml file. Mot common usage is when manual configuration 
  * is being used with no config file and a file containing only the classes is passed in.<p>
 * @param config
 * @param stream
 * @return IndexConfig containing the parsed classes.
 * @throws JoftiException
 */
public IndexConfig parseClassMapping(IndexConfig config, InputStream stream) throws JoftiException{
     return parseClassFromStream(config,stream);
 }

/**
 * Used to parse only the classmappings fragment of an xml file. Mot common usage is when manual configuration 
 * is being used with no config file and a file containing only the classes is passed in. Note: uses getResourceAsStream and so 
 * file name must be on classpath.<p>
* @param config
* @param classFileName
* @return IndexConfig containing the parsed classes.
* @throws JoftiException
*/
 public IndexConfig parseClassMapping(IndexConfig config, String  classFileName) throws JoftiException{
     InputStream stream =null;
     try {
			stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(classFileName);

			if (stream == null){
				throw new JoftiException("Unable to find config file '" + classFileName +"' on classpath - ensure the file name is correctly configured");
			}
			config = parseClassFromStream(config,stream);
     }	catch(Throwable t){
				if(t instanceof JoftiException){
				        throw (JoftiException )t;
                }else{
                    throw new JoftiException(t);
                }
	}
     return config;
	
 }
 
 private IndexConfig parseClassFromStream(IndexConfig config, InputStream stream) throws JoftiException{
	 	
		
		try {
	
			if (stream == null){
				throw new JoftiException("Input Stream for classes is null");
			}
			
			XmlPullParser parser = getParser();
			parser.setInput(stream, null);
			
			int eventType = parser.getEventType();
			
			do {
				if(eventType == XmlPullParser.START_DOCUMENT){
					
					eventType = parser.nextTag();
					String name = parser.getName();
					while (eventType == XmlPullParser.START_TAG) {
					 		   
					
					if (eventType == XmlPullParser.START_TAG && ("classes".equalsIgnoreCase(name)) 
							 ) {
				
					
						config = processClasses(parser, config);
						
						parser.require(XmlPullParser.END_TAG,"","classes");
						eventType =parser.next();
						name = parser.getName();
						
					}else if (eventType == XmlPullParser.START_TAG && ("queries".equalsIgnoreCase(name)) 
					 ) {
						config = processQueries(parser, config);
						
						parser.require(XmlPullParser.END_TAG,"","queries");
						eventType =parser.next();
						name = parser.getName();
					}else if (eventType == XmlPullParser.START_TAG && ("disk-overflow".equalsIgnoreCase(name)) 
                     ) {
                        config =processOverflow(parser, config);
                        parser.require(XmlPullParser.END_TAG,"","disk-overflow");
                        eventType =parser.next();
                        name = parser.getName();
                    }else{
						throw new JoftiException("Classes definitions must contain only '<classes>' or '<queries>' or '<disk-overflow>' tag");
					}
					}
					
				}
				
				//eventType = parser.getEventType();
				if (eventType !=XmlPullParser.END_DOCUMENT ){
					throw new JoftiException("Document not formatted correctly extra tag found "+ parser.getName());
				}
			}while (eventType != XmlPullParser.END_DOCUMENT);		
			parser.setInput (null);
		} catch (Exception e) {
			
            throw new JoftiException(e);
		}finally{
		
			if (stream != null){
				try{
					stream.close();
				} catch(Exception e){
					// we do not care about this
				}
			}
		}
		return config;
 }
 
 
 
}

