package com.jofti.core;

import java.util.Properties;

import com.jofti.btree.EntrySplitWrapper;
import com.jofti.btree.IPage;
import com.jofti.exception.JoftiException;
import com.jofti.store.StoreWrapper;

/**
 * <p>The store manager is responsible for mediating between the store implementations and 
 * the the index.
 * </p>
 * <p>
 * After retrieving a page the caller SHOULD ensure they call either store, remove or release after finishing with a page. Failure to do so may well 
 * have an impact on the performance of the store in being able to recycle pages.
 * </p>
 * 
 * @author xenephon
 * @since Jofti 1.2
 */
public interface IStoreManager {

	/**
     * Initialises the store manager.
     * 
	 * @param properties
	 * @throws JoftiException
	 */
	public abstract void init(Properties properties) throws JoftiException;
	
	public abstract String getName();

	public abstract void setName(String name);

	/**
     * <p>
     * Stores the page in the store. Any changes to the storekey are made in the return key object. 
     * </p>
     * <p>
     * </p>
	 * @param key - containing the locations to store the page under (possibly more than one)
	 * @param obj - the page to store.
	 * @return
	 * @throws JoftiException - thrown if the key cannot be stored.
	 */
	public IStoreKey store(IStoreKey key,IPage obj) throws JoftiException;
	
	/**
     * Called by the client to notify the manager that they have finished with the page.
     *  
	 * @param key
	 * @param obj
	 */
	public void releasePage(IStoreKey key,IPage obj);
	
	/**
     * Returns the next available storeKey to the caller.
     * 
	 * @return
	 * @throws JoftiException
	 */
	public IStoreKey getNextKey() throws JoftiException;
	
	
	/**
     * Returns a StoreWrapper object which containes the StoreKey and the IPage the 
     * key refers to.
     * 
	 * @param key
	 * @return
	 * @throws JoftiException
	 */
	public StoreWrapper retrieve(IStoreKey key) throws JoftiException;
	
	/**
     * Removes all entries from the store.
	 * @throws JoftiException
	 */
	public void removeAll() throws JoftiException;
	
	/**
     * Removes the page from the store and frees up the positions specified in the storeKey.
	 * @param key
	 * @param page
	 * @throws JoftiException
	 */
	public void remove(IStoreKey key, IPage page) throws JoftiException;
	
	/**
     * Splits an IPage into two IPage objects with a fair distribution of entries. The EntrySplitWrapper 
     * contains both the page and the number of entries in the page.
     * 
	 * @param page
	 * @param entryNumber
	 * @return
	 */
	public EntrySplitWrapper[] split(IPage page, int entryNumber);
    
  
}