/*
 * Created on 08-Apr-2005
 *
 */
package com.jofti.core;

import java.util.Map;

import com.jofti.api.IndexQuery;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ParserManager;

/**
 * 
 *
 * The Internal IndexCache is the core interface that IndexCache implmentations must implement. 
 * This allows Adapters and the IndexCache Manager to interact with the IndexCache implementation.<p>
 * 
 * @author xenephon (xenephon@jofti.com)
 */
public interface InternalIndex {

	 /**
     * Puts an object into the underlying index instance and indexes the object according to
     * the class definitions in the index config file. Some indexes may only accept certain types of 
     * object.<p>
     * @param key value to use as key in the index
     * @param value value to put into cache and to be indexed
     * @throws JoftiException thrown as a wrapper exception that contains any index specific exceptions.
     */
	public void insert(Object key, Object value) throws IllegalArgumentException, JoftiException;
	
	
	public void insertEntry(Object key, Object value) throws IllegalArgumentException, JoftiException;
	
	 /**
     * Deletes an object from the index. Attempting to remove a non-existent, or expired object will 
     * not generate an exception. 
     * <p>
     * @param key  -the key to retrieve the cached object
     * @param value - the object to be removed
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
	public void remove(Object key, Object value) throws IllegalArgumentException, JoftiException;
	 
	/**
     * Deletes all entries for the key from the index. Attempting to remove a non-existent, or expired object will 
     * not generate an exception. 
     * <p>
     * @param key  -the key to retrieve the cached object
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
	public void removeByKey(Object key) throws  JoftiException;
	
	
	/**
     * Returns a Map of attribute values that the index has for the key from the index.  
     * <p>
     * @param key  -the key to retrieve the cached object
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
	public Map getAttributesByKey(Object key) throws  JoftiException;
	
	/**
     * Returns whether the index contains the key. 
     * <p>
     * @param key  -the key to check
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
	
	public boolean contains(Object key) throws JoftiException;
	
	
	/**
     * Returns the keys that are stored under the entry. 
     * <p>
     * @param key  -the object to check
     * @throws JoftiException - A wrapper for any cache specific exceptions or exceptions generated by the index.
     */
	public Map getEntries(Object key) throws JoftiException;
	
	 /**
     * This will remove all the values in the cache and dump the current index.<p>
     * @throws JoftiException
     */
	public void removeAll() throws JoftiException;
	
    /**
     * This queries the index and retrieves a map of matching elements (if any). The map contains 
     * key/value pairs and the result is ordered by the attribute being searched on (if a single attribute is used in the query).
     * The ordering is preserved in the iteration as it is a @seejava.util.LinkedHashMap.
     * <p>
     * If you are using a NameSpacedIndex the keys returned 
     * in this map are of type @see NameSpaceKey. 
     * <p>
     * @param query - the type of query to perform against the index and cache.
     * <p>
     * @return Map a map of the results<br>
     * @throws JoftiException a wrapper exception for errors in the query. <br>
     */
	public Map query(IndexQuery query) throws JoftiException;
	 
	
	
	public long getKeyNumber();
	
	public ClassIntrospector getIntrospector();
	
	
	/**
	 * Returns the parser manager configured in this index.
	 * @return ParserManager
	 */
	public ParserManager getParserManager();
	
	
	
	  
}
