/*
 * ObjectInspector.java
 *
 * Created on 11 May 2003, 18:59
 */

package com.jofti.introspect;

import java.io.File;
import java.io.ObjectStreamField;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.DoubleBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import java.nio.charset.Charset;
import java.sql.Timestamp;
import java.text.CollationKey;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.btree.ValueObject;
import com.jofti.config.IndexConfig;
import com.jofti.exception.JoftiException;
import com.jofti.exception.PropertyNotIndexedException;
import com.jofti.model.ComparableBoolean;
import com.jofti.model.ParsedObject;
import com.jofti.util.ReflectionUtil;

import edu.emory.mathcs.backport.java.util.concurrent.ConcurrentHashMap;


/**
 * A JavaBean idiom compliant introspector. This means that Objects have to implement thet getter/setter 
 * metaphor in order to be correctly interrogated.</p>
 * 
 * The general usage of the parsing is package.Class.property. This value needs to be specified int he config file 
 * in this format and the introspector will use this mechanism to navigate to the correct property. </p>
 * @author  xenephon (xenephon@jofti.com)
 */
public class JavaBeanClassIntrospector implements ClassIntrospector
{
    
    private static Log log =  LogFactory.getLog(JavaBeanClassIntrospector.class);


    private static final String METHOD_PREFIX = "get";
    
    private static final Class[] noArgs = new Class[]{};    
    
    private static final int KEY_DIMENSION_VALUE = -1;
    
    private static int keyDimensionCount = KEY_DIMENSION_VALUE;
    
    private static final String NULL_ATTRIBUTE = "NULL METHOD";
    
    private static final String ARRAY_ATTRIBUTE = "[element]";
    
    
    private static  Method ARRAY_METHOD = null;
    
    private static  Method COLLECTIONS_METHOD = null;
    
    private static Method NULL_METHOD = null;
    
    private static final Set defaultClasses = new HashSet();
    
    static {
    		defaultClasses.add( java.lang.String.class);
    		defaultClasses.add( java.lang.Integer.class);
    		defaultClasses.add( java.lang.Double.class);
    		defaultClasses.add( java.lang.Float.class);
    		defaultClasses.add( java.lang.Long.class);
    		defaultClasses.add( java.lang.Short.class);
    		defaultClasses.add( java.lang.Byte.class);
    		defaultClasses.add( java.lang.Character.class);
    		defaultClasses.add(java.math.BigInteger.class);
    		defaultClasses.add(java.math.BigDecimal.class);
    		defaultClasses.add(java.util.Date.class);
    		defaultClasses.add(Timestamp.class);
    		defaultClasses.add(java.net.URI.class);
    		defaultClasses.add(ComparableBoolean.class);

	        defaultClasses.add( java.lang.String[].class);
	        defaultClasses.add( java.lang.Integer[].class);
	        defaultClasses.add( java.lang.Double[].class);
	        defaultClasses.add( java.lang.Float[].class);
	        defaultClasses.add( java.lang.Long[].class);
	        defaultClasses.add( java.lang.Short[].class);
	        defaultClasses.add( java.lang.Byte[].class);
	        defaultClasses.add( java.lang.Character[].class);
	        defaultClasses.add(java.math.BigInteger[].class);
	        defaultClasses.add(java.math.BigDecimal[].class);
	        defaultClasses.add(java.util.Date[].class);
	        defaultClasses.add(Timestamp[].class);
	        defaultClasses.add(java.net.URI[].class);
	        defaultClasses.add(ComparableBoolean[].class);
	        defaultClasses.add(java.lang.Boolean[].class);
	        defaultClasses.add(int[].class);
	        defaultClasses.add(long[].class);
	        defaultClasses.add(short[].class);
	        defaultClasses.add(byte[].class);
	        defaultClasses.add(boolean[].class);
	        defaultClasses.add(char[].class);
	        defaultClasses.add(float[].class);
	        defaultClasses.add(double[].class);
	    }
    
    private static final Set excludedComparableClasses = new HashSet();
    
    static {
        excludedComparableClasses.add( ByteBuffer.class);
        excludedComparableClasses.add( CharBuffer.class);
        excludedComparableClasses.add( Charset.class);
        excludedComparableClasses.add( CollationKey.class);
        excludedComparableClasses.add( DoubleBuffer.class);
        excludedComparableClasses.add( File.class);
        excludedComparableClasses.add( FloatBuffer.class);
        excludedComparableClasses.add( IntBuffer.class);
        excludedComparableClasses.add( LongBuffer.class);
        excludedComparableClasses.add( ObjectStreamField.class);
        excludedComparableClasses.add( ShortBuffer.class);
		
}
    
    
    static {
    	try{
    		ARRAY_METHOD = JavaBeanClassIntrospector.class.getDeclaredMethod("arrayMethod",
                    new Class[]{});
    		
    		COLLECTIONS_METHOD = JavaBeanClassIntrospector.class.getDeclaredMethod("collectionsMethod",
                    new Class[]{}); 
    		
    		NULL_METHOD = JavaBeanClassIntrospector.class.getDeclaredMethod("nullMethod",
                    new Class[]{});
    	}catch (Exception e){
    		
    	}
    }
    
    private void arrayMethod(){
    	
    }
    
    private void nullMethod(){
    	
    }
    
    private void collectionsMethod(){
    	
    }

   private Set primitiveSet = java.util.Collections.unmodifiableSet(defaultClasses);
   
    Map dimensionMap = new ConcurrentHashMap();    
    
    Map interfaceMap = new ConcurrentHashMap();
    
    Map concreteMap = new ConcurrentHashMap();
    
    Map keyDimensionMap =  new ConcurrentHashMap();
  
    
    private ClassUtils utils = new ClassUtils();
    
    /* (non-Javadoc)
     * @see com.jofti.introspect.ClassIntrospector#parseConfig(com.jofti.config.IndexConfig)
     */
    public void parseConfig(IndexConfig config) throws JoftiException {

		Map configMap = config.getIndexMappings();
		// check if we have a config map
		if (configMap == null) {
			return;
		}

		// starts at 0
		int dimensionCount = 0;

		// look through mappings to construct methods
		for (Iterator it = configMap.keySet().iterator(); it.hasNext();) {
			// what class are we dealing with
			String className = (String) it.next();
			Class clazz = null;

			// see if the JVM knows about the class
			try {
				clazz = ReflectionUtil.classForName(className);
			} catch (Exception e) {
				log.info("No class found for  " + className);
				throw new JoftiException(e);
			}

			// set up the parsed Object
			ParsedObject parsedObject = new ParsedObject();

			parsedObject.setClassValue(clazz);
			Map classMap = new ConcurrentHashMap();
			Map methodMap = new ConcurrentHashMap();
			Map dimensionClassMap = new ConcurrentHashMap();
			Map fieldMethodMap = new ConcurrentHashMap();
			
			parsedObject.setFieldValues(classMap);
			parsedObject.setMethodValues(methodMap);
			parsedObject.setDimensionClassMappings(dimensionClassMap);
			parsedObject.setFieldMethods(fieldMethodMap);

			// we need to look at proxy or interface stuff here
			List properties = (List) configMap.get(className);
			if (clazz.isInterface()) {
				parsedObject.setInterface(true);
			}
			// now we get the map of attributes to index

			dimensionCount = parseAttribtes(dimensionCount, clazz, classMap,
					methodMap, dimensionClassMap, fieldMethodMap,properties);

			dimensionMap.put(clazz, parsedObject);

		}

		parseBuiltInClasses(dimensionCount);
	}

    /**
	 * @param dimensionCount
	 * @param clazz
	 * @param classMap
	 * @param methodMap
	 * @param dimensionClassMap
	 * @param attributes
	 * @return
	 * @throws JoftiException
	 */
    
    
    
    private int parseAttribtes(int dimensionCount, Class clazz, Map classMap, Map methodMap, Map dimensionClassMap, Map fieldMethodMap, List attributes) throws JoftiException
    {
    	
    	
        if (attributes != null){
        	
        
   		 int size = attributes.size();
	        Iterator attribIterator = attributes.iterator();
	        for (int i=0;i<size;i++) {
        		String attribute = (String)attribIterator.next();
        		
        			Object[] methods = getMethodArray(clazz, attribute);
        			if (methods == null || methods.length ==0){
        				throw new JoftiException("No getter method found for attribute " + attribute + " in class " + clazz);
        			} else{
        				
        				
        					Class returnType = null;
        					if (methods[methods.length-1] == ARRAY_METHOD){
        				
        						throw new JoftiException("terminating value cannot be an array type");
        					}else if (methods[methods.length-1] instanceof Method){
        						returnType = ((Method)methods[methods.length-1]).getReturnType();
        					}else{
        						// must contain the return value as an object
        						returnType = (Class)methods[methods.length-1];
        					}
        				
        				
        				// see if method type is a primitive we can change
        				if (returnType.isPrimitive() || returnType == Boolean.class){
        					returnType = boxPrimitive(returnType);
        				} 
        				
        				// see if the final value from method array is comparable
  
        				if (Comparable.class.isAssignableFrom(returnType) && !excludedComparableClasses.contains(returnType) ){
        					Integer dimension = new Integer(dimensionCount);
        					classMap.put(attribute,dimension);
        					methodMap.put(methods,dimension );
        					dimensionClassMap.put(dimension,returnType);
        					fieldMethodMap.put(attribute,methods);
        					dimensionCount++;
        				} else{
        					throw new JoftiException("Attribute " + attribute + " is not Comparable in class " +clazz + " or is an excluded Comparable type");
        				}
        			
        			}
        	}
        	
        }
        return dimensionCount;
    }

    /**
	 * @param clazz
	 * @param attribute
	 * @return
	 */
    public Object[] getMethodsForAttribute(Class clazz,String attribute) throws JoftiException{
    	
    	Object[] results  =null;
    	
   	// first see if we already have the field
    	ParsedObject obj = (ParsedObject)dimensionMap.get(clazz);
    	if (obj != null){
    		// see if it one we have already specified
    		Map fieldMethods = obj.getFieldMethods();
    		// we have a parsed class
    		
    		// see if it is a primitive class and value is value
    		if (fieldMethods.containsKey(NULL_ATTRIBUTE)){
    			 if ("value".equalsIgnoreCase(attribute)){
    			 	return (Object[])fieldMethods.get(NULL_ATTRIBUTE);
    			 }else{
    			 	throw new JoftiException("Attribute "+ attribute + " is not allowed for Class "+ clazz + " can only have attribute VALUE or must be a whole object ");
    			 }
    		}
    		results = (Object[])fieldMethods.get(attribute);
    		
    		if (results == null){
    			
    			// oj lets see if it is a primitive so it must have 
    			// a null method
    			
    			results = getMethodArray(clazz,attribute);
    		}
    		return results;
    	}else{
    		results = getMethodArray(clazz,attribute);
    	}
    	return results;
    }
    
	private Object[] getMethodArray(Class clazz, String attribute)throws JoftiException {
		Object[] methods =null;
		try {
			methods = preProcessAttribute(attribute);
			methods =constructMethodArray(methods, clazz);
			
		} catch (JoftiException e){
			log.warn("No getter method found for attribute " + attribute + ": " + e);
			throw new JoftiException("No getter method found for attribute " + attribute +" on " + clazz,e);
		}
		return methods;
	}

	/**
     * @param dimensionCount
     */
    private void parseBuiltInClasses(int dimensionCount)
    {
        // now parse in the default classes
        for (Iterator it = defaultClasses.iterator();it.hasNext();){
        	
	        	ParsedObject parsedObject = new ParsedObject();
	    		Class temp = (Class)it.next();
	    		parsedObject.setClassValue(temp);
	    		Map classMap = new HashMap();
	    		Map methodMap = new HashMap();
	    		Map fieldMethodMap = new HashMap();
	    		
	    		parsedObject.setFieldValues(classMap);
	    		parsedObject.setMethodValues(methodMap);
	    		parsedObject.setFieldMethods(fieldMethodMap);
	    		
	    		Integer dimension = new Integer(dimensionCount);
	    		String attribute =null;
    			
    			Object[] meth =null;
                if (temp.isArray()){
                	meth = new Object[2];
                	meth[0]=ARRAY_METHOD;
                	meth[1]=temp.getComponentType();
                	attribute= ARRAY_ATTRIBUTE;
                }else{
                	meth = new Object[1];
                	meth[0]=NULL_METHOD;
                	attribute = NULL_ATTRIBUTE;
                }
                
                methodMap.put(meth,dimension);
                classMap.put(attribute,dimension);
                fieldMethodMap.put(attribute,meth);

     
			dimensionCount++;
			dimensionMap.put(temp,parsedObject);
        }
    }
    
    public Class boxPrimitive(Class clazz){
    	if (clazz == Integer.TYPE){
    		return Integer.class;
    	}else if (clazz == Double.TYPE){
    		return Double.class;
    	}else if (clazz == Long.TYPE){
    		return Long.class;
    	}else if (clazz == Float.TYPE){
    		return Float.class;
    	}else if (clazz == Short.TYPE){
    		return Short.class;
    	}else if (clazz == Byte.TYPE){
    		return Byte.class;
    	}else if (clazz == Character.TYPE){
    		return Character.class;
    	}else{
    		return ComparableBoolean.class;
    	}
    }
    
    /* (non-Javadoc)
     * @see com.jofti.introspect.ClassIntrospector#getClassForAttribute(java.lang.Class, java.lang.String)
     */
    public Class getClassForAttribute(Class className, String attribute) throws JoftiException{
    	
    	ParsedObject obj = (ParsedObject)dimensionMap.get(className);
		if (obj != null){
			Object returnObj =null;
			// first see if it has the null method
			if ( (returnObj= obj.getFieldValues().get(NULL_ATTRIBUTE)) != null){
				// we should return the dimension ofr the primitive class itself
				if (attribute != null && !"value".equalsIgnoreCase(attribute)){
					throw new JoftiException ("Only field value 'VALUE' can be used for class "+ className);
				}
				return className;
			} else if ( (returnObj= obj.getFieldValues().get(ARRAY_ATTRIBUTE)) != null){
                // we should return the dimension ofr the primitive class itself
					if (attribute.equalsIgnoreCase(ARRAY_ATTRIBUTE)){
						return className;
					}else{
						throw new PropertyNotIndexedException(" property "+ attribute + " cannot refer to an element in "+ className + " use "+ ARRAY_ATTRIBUTE);
						
					}
				
            }else {			
				try {
					return (Class) obj.getDimensionClassMappings().get(obj.getFieldValues().get(attribute));
				} catch (Throwable e){
					throw new PropertyNotIndexedException(" property "+ attribute + " not indexed in " + className);
				}
			}
		}else{
			throw new PropertyNotIndexedException(" class "+ className + " not in index");
		}
    }
    
    
    private Object[] preProcessAttribute(String attribute) throws JoftiException{
    	List tempList = new ArrayList();
    	int startFragment =0;
    	int endFragment =0;
    	boolean inArray=false;
    	if (attribute.length() ==1){
    		return new Object[]{attribute};
    	}
    	for (int i=0;i<attribute.length();i++){
    		if (attribute.charAt(i)=='['){
    			inArray = true;
    			startFragment =i;
    		}else if (attribute.charAt(i)==']'){
    			
    			endFragment =i;
    		}else if ( i ==attribute.length()-1){
    			endFragment =i;
    		}
    		else if (! inArray){
    			if (attribute.charAt(i)=='.'){
    				endFragment = i;
    			}
    		}else {
    			
    		}
    		if (endFragment > startFragment ){
    			// we should extract the string
    			if (inArray || i== attribute.length()-1){
    				endFragment = endFragment+1;
    			}
    			String temp = attribute.substring(startFragment,endFragment);
    			if (temp.startsWith(".")){
    				temp = temp.substring(1);
    			}

    			tempList.add(temp);
    			startFragment = endFragment;
    			inArray =false;
    		}
    	}
    	return tempList.toArray();
    	
    }
    
    private Object[] constructMethodArray(Object[] attributes,
			Class originalClass) throws JoftiException {
    	
    	List methods = new ArrayList();

		try {
			// we have an array thing
	
			
			
			Method tempMethod = null;
			int j= attributes.length;
			int k=0;
			for (int i = 0; i < j; i++) {
				
				if (originalClass.isArray() && attributes[i].equals("[element]")){
					
					// check that the component type is not an object
					if (originalClass.getComponentType().equals(Object.class)){
						throw new JoftiException("Types to be indexed in an Object array must be specified");
					}
					
					//	add in the extra method that signifies an array
					// make sure that element is the attribute
					
					methods.add(ARRAY_METHOD);
					
					// increment j
					
					originalClass = originalClass.getComponentType();
					methods.add(originalClass);
					
				}else if (originalClass.isArray()){
					// we have to get they type out of it
					String tempClass = ((String)attributes[i]).substring(1,((String)attributes[i]).length()-1);
					methods.add(ARRAY_METHOD);
					Class elementClass =null;
					try{
						elementClass =  ReflectionUtil.classForName(tempClass);
					} catch (Exception e){
						throw new JoftiException(e);
					}
					
					if (originalClass.getComponentType().isAssignableFrom(elementClass) ){
						originalClass = elementClass;
						
						methods.add(originalClass);
						
					}else{
						throw new JoftiException("element type "+ elementClass + " is not assignable to " + originalClass.getComponentType());
					}
					
				} else if (originalClass.isAssignableFrom(Collection.class) && attributes[i].equals("[element]")){
					
					// check that the component type is not an object
					if (originalClass.getComponentType() == null || originalClass.getComponentType().equals(Object.class)){
						throw new JoftiException("Types to be indexed in a Collection must be specified by Type");
					}
					
					
				}else if (Collection.class.isAssignableFrom(originalClass)){
					
					String tempClass = ((String)attributes[i]).substring(1,((String)attributes[i]).length()-1);
					methods.add(COLLECTIONS_METHOD);
					Class elementClass =null;
					try{
						elementClass =  ReflectionUtil.classForName(tempClass);
					} catch (Exception e){
						throw new JoftiException(e);
					}
					
					methods.add(elementClass);
					originalClass = elementClass;
				}else{
					
					
				
				
					tempMethod = getMethod(originalClass,(String)attributes[i],noArgs);
		
					methods.add( tempMethod);
					originalClass = tempMethod.getReturnType();
					
				}	
				
			}

		} catch (NoSuchMethodException nme) {
			throw new JoftiException(nme);
		}
			

    		
    		
    		return methods.toArray();
    }
   
    private Method getMethod(Class originalClass, String attribute, Class[] args) throws NoSuchMethodException
    {
	    	StringBuffer buf = new StringBuffer(attribute);
			buf.setCharAt(0, Character.toUpperCase(buf.charAt(0)));
			buf.insert(0, METHOD_PREFIX);
			return originalClass.getMethod(buf.toString(), args);

    }
    

	/* (non-Javadoc)
	 * @see com.jofti.introspect.ClassIntrospector#getDimension(java.lang.String, java.lang.String)
	 */
    
	public int getDimension(String className, String propertyName) throws JoftiException {
		Class clazz = null;
		
		try {
			clazz = ReflectionUtil.classForName(className);
		} catch (Exception e){
			log.info("No class found for  " + className);
			throw new JoftiException(e);
		}
		return getDimension(clazz, propertyName);
	}
	public int getDimension(Class clazz, String propertyName) throws JoftiException {

		int temp =0;
		ParsedObject obj = (ParsedObject)dimensionMap.get(clazz);
		if (obj != null){
			Object returnObj =null;
			// first see if it has the null method
			if ( (returnObj= obj.getFieldValues().get(NULL_ATTRIBUTE)) != null){
				// we should return the dimension ofr the primitive class itself
				temp = ((Integer)returnObj).intValue();
			} else if ( (returnObj= obj.getFieldValues().get(ARRAY_ATTRIBUTE)) != null){
                // we should return the dimension ofr the array class itself
                temp = ((Integer)returnObj).intValue();
            } else {			
				try {
					temp = ((Integer)obj.getFieldValues().get(propertyName)).intValue();
				} catch (Throwable e){
					throw new PropertyNotIndexedException("no dimension found for property "+ propertyName);
				}
			}
		}else{
			throw new PropertyNotIndexedException("no dimension found for class "+ clazz);
		}
		return temp;
	}

	/* (non-Javadoc)
	 * @see com.jofti.introspect.ClassIntrospector#getKeyDimension()
	 */
	public int getKeyDimension() {
		
		return KEY_DIMENSION_VALUE;
	}
	
	public synchronized int getKeyDimension(Class className) {
		
		Integer temp = (Integer)keyDimensionMap.get(className);
		if (temp == null){
			temp = new Integer(keyDimensionCount--);
			keyDimensionMap.put(className, temp);
		}
		return temp.intValue();
	}

	/* (non-Javadoc)
	 * @see com.jofti.introspect.ClassIntrospector#getAttributeValues()
	 */
	public Map getAttributeValues(Object obj) throws JoftiException {


		 if (obj == null){
			return new HashMap(1);
		 }
         
         
		 obj =utils.wrapObject(obj);
         
         Class objClass = obj.getClass();
         List parsedList = getParsedObjectsForClass(objClass);
         
		 //
         if (parsedList.size() >0){
             return getValuesFromParsedObjects(obj,parsedList);
         } else{

	         if (log.isDebugEnabled()){
	             log.debug("no mapping found for class "+ objClass);
	         }
	        return new HashMap(1);
         }
	}

	public List getParsedObjectsForClass(Class clazz){
		List temp = new ArrayList();
		
		// first get the interfaces
		Class[] interfaceArray = (Class[])interfaceMap.get(clazz);
		if (interfaceArray == null){
			interfaceArray = utils.getInterfaces(clazz);
			// we can't cache proxy xlasses as they might change
			if (!java.lang.reflect.Proxy.isProxyClass(clazz)){
				interfaceMap.put(clazz,interfaceArray);
			}
		}
		temp = getParsedClassesForArray(interfaceArray,temp);
		
		// if not a proxy then do the concrete classes
		if (!java.lang.reflect.Proxy.isProxyClass(clazz)){
			Class[] concreteArray = (Class[])concreteMap.get(clazz);
			if (concreteArray == null){
				concreteArray = utils.getClasses(clazz);
				concreteMap.put(clazz,concreteArray);
			}
			temp = getParsedClassesForArray(concreteArray,temp);
		}
		return temp;
		
	}
    
	private List getParsedClassesForArray(Class[] array, List temp){
		for (int i=0;i<array.length;i++){
			// get all the parsedObjects that match
			ParsedObject parsedObj = (ParsedObject)dimensionMap.get(array[i]);
			if (parsedObj != null){
				temp.add(parsedObj);
			}
		}
		return temp;
	}
	
	 private Map getValuesFromParsedObjects(Object obj,List parsedObjects){
		 Map temp = new HashMap(2);
		 
		 for (int i=0;i<parsedObjects.size();i++){
			 ParsedObject parsedObj = (ParsedObject)parsedObjects.get(i);
			 temp = getValuesFromParsedObject(temp,obj,parsedObj);
		 }
		 return temp;
	 
	 }
	
	  public Map getResultValuesFromObject(Map temp, Object obj,Map methodMap){
	  	int size = methodMap.size();
	  	Iterator it = methodMap.entrySet().iterator();
	       for (int j=0;j<size;j++) {
            Object tempObj = obj;
            // loop through the objects and index the value
            // turn this into a queue
            Map.Entry entry = (Map.Entry)it.next();
            Object[] methods = (Object[])entry.getValue();
            if (methods ==null){
                
                temp.put(entry.getKey(),tempObj);
            } else {
            	List results = new ArrayList();
                try {
                    tempObj = processMethodArray(tempObj, methods);
                    
                    
                    if (tempObj != null){
                       
                        temp.put(entry.getKey(),tempObj);
                    }else{
                    	temp.put(entry.getKey(),null);
                    }
                } catch (Exception e){
                    throw new IllegalArgumentException(e.getMessage());
                    
                } 
            }
                
        }
	       return temp;
	  	
	  }
    private Map getValuesFromParsedObject(Map temp, Object obj,ParsedObject parsedObj){
        
        Set tempSet = parsedObj.getMethodValues().entrySet();
        int size = tempSet.size();
        Iterator it = tempSet.iterator();
        for (int j=0;j<size;j++) {
                Object tempObj = obj;
                // loop through the objects and index the value
                // turn this into a queue
                Map.Entry entry = (Map.Entry)it.next();
                Object[] methods = (Object[])entry.getKey();
                if (methods ==null){
                    
                    temp.put(entry.getValue(),utils.wrapObject(tempObj));
                } else {
                	List results = new ArrayList();
                    try {
                        tempObj = processMethodArray(tempObj, methods);
                        
                        
                        if (tempObj != null){
                           
                            temp.put(entry.getValue(),utils.wrapObject(tempObj));
                        }else{
                        	temp.put(entry.getValue(),ValueObject.MIN_COMAPARBLE_VALUE);
                        }
                    } catch (Exception e){
                        throw new IllegalArgumentException(e.getMessage());
                        
                    } 
                }
                    
            }
            return temp;
    }
    
    public Object getResultFromMethods(Object obj, Object[] methods) throws JoftiException{
    	Object retVal =null;
    	try {
    		retVal = processMethodArray(obj, methods);
    	}catch (Exception e){
    		throw new JoftiException(e);
    	}
    	return retVal;
    	
    }
	
	/**
	 * @param tempObj
	 * @param methods
	 * @return
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	private Object processMethodArray(Object tempObj, Object[] methods) throws IllegalAccessException, InvocationTargetException {
		for (int i =0;i <methods.length;i++){
			if (tempObj == null){
				break;
			}
			if (methods[i] == ARRAY_METHOD){
				// we need to get the values
				tempObj = getRealArrayValues(tempObj,(Class)methods[i+1]);
			}else if (methods[i] == NULL_METHOD){
				// we need to use the actual value
				// we do not need to do anything here
			}else if (tempObj.getClass().isArray() && methods[i] instanceof Method){
				//apply the method to each object
				tempObj = applyMethodsToArray(tempObj,(Method)methods[i]);
			}else if (tempObj.getClass().isArray() && methods[i] instanceof Class){
				// get the values out of the array
				tempObj = getArrayValues(tempObj,(Class)methods[i] );
			}else if (methods[i] == COLLECTIONS_METHOD){
				// we need to get the values
				tempObj = getRealCollectionValues((Collection)tempObj,(Class)methods[i+1]);
			} else if (Collection.class.isAssignableFrom(tempObj.getClass()) && methods[i] instanceof Method){
				//apply the method to each object
				tempObj = applyMethodsToCollection((Collection)tempObj,(Method)methods[i]);
			}else if (Collection.class.isAssignableFrom(tempObj.getClass())  && methods[i] instanceof Class){
				// get the values out of the array
				tempObj = getCollectionValues((Collection)tempObj,(Class)methods[i] );
			}else {
			
				tempObj = ((Method)methods[i]).invoke(tempObj,noArgs);
			}
		}
		//tempObject = method.invoke(obj,noArgs);
		return tempObj;
	}

	private Object[] applyMethodsToArray( Object array, Method meth){
       
        int size = Array.getLength(array);
        List res = new ArrayList();
        for (int i=0;i<size;i++){
            
            Object temp = Array.get(array,i);
           
            if (temp != null ){
            	
            	try {
				temp=	meth.invoke(temp,noArgs);
				} catch (IllegalArgumentException e) {
					
					log.warn(e);
				} catch (IllegalAccessException e) {
					
					log.warn(e);
				} catch (InvocationTargetException e) {
					log.warn(e);
				}
            	
            }
            if (temp != null){
            	res.add(temp);
            }
        }
        if (res.size() == 0){
            return null;
        }else{
            return res.toArray();
        }

    }
	
	private Object[] applyMethodsToCollection( Collection col, Method meth){
	       
        
        List res = new ArrayList();
        
        int size = col.size();
        Iterator it = col.iterator();
        for (int i=0;i<size;i++) {
            
            Object temp = it.next();
           
            if (temp != null ){
            	
            	try {
				temp=	meth.invoke(temp,noArgs);
				} catch (IllegalArgumentException e) {
					log.warn(e);
				} catch (IllegalAccessException e) {
					log.warn(e);
				
				} catch (InvocationTargetException e) {
					log.warn(e);
				}
            	
            }
            if (temp != null){
            	res.add(temp);
            }
        }
        if (res.size() == 0){
            return null;
        }else{
            return res.toArray();
        }

    }
	private Object[] getArrayValues( Object array, Class type){
	       
	        int size = Array.getLength(array);
	        List res = new ArrayList();
	        for (int i=0;i<size;i++){
	            
	            Object temp = Array.get(array,i);
	            if (type.isPrimitive() ){
		        	   res.add(utils.wrapObject(temp));
		           }else
	            if (temp != null && (type.isAssignableFrom(temp.getClass()) ||type.isAssignableFrom(utils.wrapObject(temp.getClass()).getClass())) ){
	                res.add(utils.wrapObject(temp));
	            }
	        }
	        if (res.size() == 0){
	            return null;
	        }else{
	            return res.toArray();
	        }

	    }
	
	private Object[] getCollectionValues( Collection col, Class type){
	       

        List res = new ArrayList();
        int size = col.size();
        Iterator it = col.iterator();
        for (int i=0;i<size;i++) {
            
            Object temp = it.next();

            if (temp != null && (type.isAssignableFrom(temp.getClass()) ||type.isAssignableFrom(utils.wrapObject(temp.getClass()).getClass())) ){
                res.add(utils.wrapObject(temp));
            }
        }
        if (res.size() == 0){
            return null;
        }else{
            return res.toArray();
        }

    }

	private Object[] getRealCollectionValues( Collection col, Class type){
	       

        List res = new ArrayList();
        
        int size = col.size();
        Iterator it = col.iterator();
        for (int i=0;i<size;i++) {
            
            Object temp = it.next();
           
            if (temp != null && (type.isAssignableFrom(temp.getClass()) ||type.isAssignableFrom(utils.wrapObject(temp.getClass()).getClass())) ){
                res.add(temp);
            }
        }
        if (res.size() == 0){
            return null;
        }else{
            return res.toArray();
        }

    }
	
	
	
	private Object[] getRealArrayValues( Object array, Class type){
	       
	        int size = Array.getLength(array);
	        List res = new ArrayList();
	        for (int i=0;i<size;i++){
	            
	            Object temp = Array.get(array,i);
	           if (array.getClass().getComponentType().isPrimitive()){
	        	   res.add(temp);
	           }else
	            if (temp != null && (type.isAssignableFrom(temp.getClass()) ||type.isAssignableFrom(utils.wrapObject(temp.getClass()).getClass())) ){
	                res.add(temp);
	            }
	        }
	        if (res.size() == 0){
	            return null;
	        }else{
	            return res.toArray();
	        }

	    }
	/* (non-Javadoc)
	 * @see com.jofti.introspect.ClassIntrospector#getKeyDimensions()
	 */
	public Map getKeyDimensions() {
		return keyDimensionMap;
	}


	/* (non-Javadoc)
	 * @see com.jofti.introspect.ClassIntrospector#getPrimitiveClasses()
	 */
	public Set getPrimitiveClasses() {
		return primitiveSet;
	}



	public Map getDimensions() {

		return dimensionMap;
	}
	


}
