/*
 * IndexManager.java
 *
 * Created on 01 June 2003, 08:17
 */

package com.jofti.manager;

import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


import com.jofti.api.Index;
import com.jofti.api.IndexManager;
import com.jofti.api.NameSpacedIndex;
import com.jofti.cache.LifeCycleAdapter;
import com.jofti.cache.NameSpacedCacheAdapter;
import com.jofti.config.ConfigFileParser;
import com.jofti.config.IndexConfig;
import com.jofti.core.GenericIndexFactory;
import com.jofti.core.InternalIndex;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.util.ReflectionUtil;

import edu.emory.mathcs.backport.java.util.concurrent.ConcurrentHashMap;

/**
 *

 * 
 * 
 * The manager is responsible for providing access to caches. This class is not a singleton 
 * and is not intended to be used in that manner.<p>
 * 
 * The usage for the cache manager is:<p>
 * 
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.setConfigFile("configFile");<br>
 * manager.init();	<br>
 * IndexCache index = manager.getIndex("name");<br>
 * <p>
 * or:
 * <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init(inputStream);	<br>
 * IndexCache index = manager.getIndex("name");<br>
 * <p>
 * or
 * <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init(configFile);	<br>
 * IndexCache index = manager.getIndex("name");<br>
 * <p>
 * The name of the index is one of the indexs configured in the configFile. 
 * <p>
 * NameSpaced caches such as JBossCache are obtained using:
 * <p>
 * IndexManager manager = new IndexManagerImpl();<br>
 * manager.init(configFile);	<br>
 * NameSpacedIndex index = manager.getNameSpacedIndex("name");<br>
 * <p>
 * @author  Steve Woodcock
 * @version 1.0
 *  
 */
public class IndexManagerImpl   implements IndexManager
{

    private static Log log         = LogFactory.getLog(IndexManagerImpl.class);

    private final Map  cacheMap    = new ConcurrentHashMap();

    private final Map  configMap   = new ConcurrentHashMap();

    private String     configFile  = null;

    private boolean    initialised = false;
    

    /** Creates a new instance of IndexManagerImpl */
    public IndexManagerImpl()
    {
        
    }

    /**
     * Sets a config file to use in the indexManager. This file must be on the
     * classpath.
     */
    public void setConfigFile(String configFile)
    {
        this.configFile = configFile;
    }

    /**
     * 
     * Initialise method that takes a config file name. This over-rides the
     * fileName (if any) set in the setConfigFile() method.
     * 
     * @param configFileName
     * @throws JoftiException
     */
    public synchronized void init(String configFileName) throws JoftiException
    {

        if (!initialised) {

            Map temp = null;
            if (log.isInfoEnabled()) {
                log.info("Loading config from file '" + configFileName + "'");
            }
            temp = loadConfig(configFileName);
            if (log.isInfoEnabled()) {
                log.info("Loaded config from file '" + configFileName + "'");
            }
            configureIndexes(temp);
            initialised = true;
        } else {
            if (log.isWarnEnabled()) {
                log
                        .warn("IndexCache is already initialised - ignoring repeat attempt using '"
                                + configFileName + "'");
            }
        }
    }

    /**
     * 
     * Initialise method that no parameters. This configures the Indexes with the
     * fileName set in the setConfigFile() method.
     * 
     * @throws JoftiException
     */
    public synchronized void init() throws JoftiException
    {

        if (!initialised) {

            Map temp = null;
            if (configFile != null) {
                if (log.isInfoEnabled()) {
                    log.info("Loading config from file '" + configFile + "'");
                }

                temp = loadConfig(configFile);

                if (log.isInfoEnabled()) {
                    log.info("Loaded config from file '" + configFile + "'");
                }
                configureIndexes(temp);
            } else {
                if (log.isInfoEnabled()) {
                    log
                            .info("No config file configured - not loading any caches");
                }
            }
            initialised = true;
        } else {
            if (log.isWarnEnabled()) {
                log
                        .warn("IndexCache is already initialised - ignoring repeat attempt ");
            }
        }
    }

    /**
     * 
     * Initialise method that takes an inputstream. This over-rides the fileName
     * (if any) set in the setConfigFile() method.
     * 
     * @param configStream
     * @throws JoftiException
     */
    public synchronized void init(InputStream configStream)
            throws JoftiException
    {

        if (!initialised) {
           
            Map temp = null;
            if (log.isInfoEnabled()) {
                log.info("Loading config from stream");
            }
            temp = loadConfig(configStream);
            if (log.isInfoEnabled()) {
                log.info("Loaded config from stream");
            }
            configureIndexes(temp);
            initialised = true;
        } else {
            if (log.isWarnEnabled()) {
                log.warn("IndexCache is already initialised - ignoring repeat attempt ");
            }
        }
    }

    /**
     * This method allows cache instances to be added to the cache
     * programatically rather than at start-up. <p>
     * The usage is for example:<p>
     * 
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test");<br>
     *  IndexCache index = (IndexCache)<br>
     * manager.addIndexedCache(config, <cacheImpl>,
     * <xml-filename-with-classes-in-it>);<br>
     * <p>
     * any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class.
     * <p>
     * If you are using a nameSpaced cache like JBossCache then the usage would
     * be:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * NameSpacedIndex index = (NameSpacedIndex)
     * manager.addIndexedCache(config, <cacheImpl>, <xml-filename-with-classes-in-it>);<br>
     * <p>
     * Note: The cache implementations must be started correctly before they are
     * passed into this method. Added caches are assumed to have been started
     * and the manager will NOT attempt to initialise the actual cache
     * implementation.
     * <p>
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @param cache -
     *            the cache implementation. Passing NULL will result in the adapter creating a new
     * 			 cache instance.
     * @param classesFileName -
     *            the xml file containing the classes definitions for the cache.
     *            This file must be available on the classpath.
     * @return The added cache.
     * @throws JoftiException
     */
    public Index addIndex(IndexConfig config, Object cache,
            String classesFileName) throws JoftiException
    {

        config = parseClassesForConfig(config, classesFileName);
        return createIndexedCache(config, cache);

    }

    /**
     * This method allows cache instances to be added to the cache
     * programatically rather than at start-up. This method will result in the
     * adapter used creating a new instance of its cache type.<p>
     * 
     * 
     * 
     * Any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class.
     * <p>
     * Note: This method is the equivalent of an index entry in the
     * configuration file. The manager will atttempt to construct and initialise
     * a new indexed cache based on the attributes in the cacheConfig class and
     * the class definition file.
     * <p>
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @param classesFileName -
     *            the xml file containing the classes definitions for the
     *            cache.This file must be available on the classpath.
     * @return The added cache.
     * @throws JoftiException
     */
    public Index addIndexCache(IndexConfig config,
            String classesFileName) throws JoftiException
    {

        config = parseClassesForConfig(config, classesFileName);
        return createIndexedCache(config, null);

    }

    /**
     * This method allows cache instances to be added to the cache
     * programatically rather than at start-up.
     * <p>
     * 
     * 
     * Any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the
     * @link IndexConfig class. This method will result in the
     * adapter used creating a new instance of its cache type.
     * <p>
     * Note: This method is the equivalent of an index entry in the
     * configuration file. The manager will atttempt to construct and initialise
     * a new indexed cache based solely on the attributes in the cacheConfig
     * class.
     * <p>
     * Class definitions can be added using attribute classMappings in the
     * IndexConfig class. See this class for details on how to configure these.
     * <p>
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @return The added cache.
     * @throws JoftiException
     */

    public Index addIndexCache(IndexConfig config)
            throws JoftiException
    {

        return createIndexedCache(config, null);

    }

    /**
     * This method allows cache instances to be added to the cache
     * programatically rather than at start-up. The usage is for example:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test"); <br>
     * IndexCache index = (IndexCache)
     * manager.addIndexedCache(config, <cacheImpl>,
     * <xml-filename-with-classes-in-it>);<br>
     * <p>
     * any type of cache can be added in this manner, providing a CacheAdapter
     * exists for it and the correct adapter has been configured in the 
     * @link IndexConfig class.
     * <p>
     * If you are using a nameSpaced cache like JBossCache then the usage would
     * be:
     * <p>
     * DefaultIndexConfig config = new DefaultIndexConfig();<br>
     * config.setName("test");<br>
     *  NameSpacedIndex index = (NameSpacedIndex)
     * manager.addIndexedCache(config, <cacheImpl>);<br>
     * <p>
     * Note: The cache implementations must be started correctly before they are
     * passed into this method. Added caches are assumed to have been started
     * and the manager will NOT attempt to initialise the actual cache
     * implementation.
     * <p>
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @param cache -
     *            the cache implementation. Passing NULL will result in the adapter creating a new
     * 			 cache instance.
     * @return The added cache.
     * @throws JoftiException
     */
    public Index addIndex(IndexConfig config, Object cache)
    throws JoftiException
    {

        return createIndexedCache(config, cache);

    }
    
    
    /* (non-Javadoc)
     * @see com.jofti.api.IndexManager#addIndexedCache(com.jofti.config.IndexConfig, java.lang.Object, java.io.InputStream)
     */
    public Index addIndex(IndexConfig config, Object cache, InputStream stream)
    throws JoftiException 
    {
        config = parseClassesForConfig(config,stream);
        return createIndexedCache(config, cache);

    }
    
    
    private IndexConfig parseClassesForConfig(IndexConfig config,
            InputStream stream) throws JoftiException
    {
        if (log.isDebugEnabled()) {
            log.info("parsing classes from inputStream ");
        }
        // load the classes to be indexed from the supplied file
        config = loadConfig(config,stream);
        

        if (log.isDebugEnabled()) {
            log.info("successfully parsed classes from inputStream ");
        }
        // return the config with the parsed classes in it.
        return config;
    } 
    /**
     * This method does the actual work of loading in the class definitions from the file.
     *      
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @param classesFileName -
     *            the xml file containing the classes definitions for the
     *            cache.This file must be available on the classpath.
     * @return The cacheConfig object with the class definitions added.
     * @throws JoftiException
     */
    private IndexConfig parseClassesForConfig(IndexConfig config,
            String fileName) throws JoftiException
    {
        if (log.isDebugEnabled()) {
            log.info("parsing classes from file " + fileName);
        }
        // load the classes to be indexed from the supplied file
        config = loadConfig(config, fileName);
        
        if (log.isDebugEnabled()) {
            log.info("successfully parsed classes from file " + fileName);
        }
        // return the config with the parsed classes in it.
        return config;
    }

  
    /**
     * This method does the actual work of creating the cache for the addIndexCache methods.
     *      
     * 
     * @param config -
     *            the config class containing definitions of the adapter, index
     *            type and parser to use.
     * @param cache -
     *            the cache implementation.
     * @return The added cache.
     * @throws JoftiException
     */
    private Index createIndexedCache(IndexConfig config, Object cache)
            throws JoftiException
    {

        // load up the adaptor
        if (log.isInfoEnabled()) {
            log.info("loading cache '" + config.getName() + "' with adapter " + config.getCacheAdapter() );
        }
        LifeCycleAdapter cacheAdaptor = null;
        
        // if we do not have a supplied cache then the adapter will create one
        if (cache == null) {
            cacheAdaptor = loadAdaptor(config);
        } else {
            cacheAdaptor = loadAdaptor(config, cache);
        }
        
        // set the name for the adapter - used as the key in the cacheMap
        cacheAdaptor.setName(config.getName());
        
        if (cacheAdaptor.getName() == null){
            throw new JoftiException("A cache must have a name to be inserted into the cache");
        }
        // we should now call init - it is up to the dapter to protect itself
        
        cacheAdaptor.init(config.getAdapterProperties());
        
        if (log.isInfoEnabled()) {
            log.info("Configuring cache '" +  cacheAdaptor.getName() + "'");
        }
        
        // create the wrapper for the cache
        cacheAdaptor = configureIndexedCache(config, cacheAdaptor);
        
        if (log.isInfoEnabled()) {
            log.info("Starting cache '" + cacheAdaptor.getName() + "'");
        }
        // start the wrapper to start the caches
        cacheAdaptor.start();


        
        if (cacheMap.get(config.getName()) != null) {
            log.warn("added cache '" + cacheAdaptor.getName() + 
                    "' will replace an already existing cache of the same name in the loaded cache mappings");
        }
        cacheMap.put(config.getName(), cacheAdaptor);
        if (log.isInfoEnabled()) {
            log.info("Added cache '" + cacheAdaptor.getName() + "' to loaded caches");
        }
        return (Index)cacheAdaptor;
    }

    
    /**
     * Configures each of the caches in turn from the config file used to initialise the manager.
     * 
     * @param configuredMap
     * @throws JoftiException
     */
    private void configureIndexes(Map configuredMap) throws JoftiException
    {

        configMap.putAll(configuredMap);

        // we have loaded the map so lets now create our internal stuff
        if (log.isInfoEnabled()) {
            log.info("Initialising indexes and caches");
        }
        // loop through the map of config entries and try and inialise each one
        for (Iterator it = configMap.entrySet().iterator(); it.hasNext();) {
            Map.Entry entry = (Map.Entry) it.next();
            IndexConfig config = (IndexConfig) entry.getValue();
            LifeCycleAdapter adapter = null;

            // see if cache is lazy loaded
            
                if (log.isInfoEnabled()) {
                    log.info("instantiating adapter '" + config.getName() + "'");
                }
                adapter = instantiateAdapter(config);
                if (log.isInfoEnabled()) {
                    log.info("instantiated adapter '" + config.getName() + "'");
                }
           
            // carry on and do the indexes even if we have no cache started
            // configure each cache
            if (log.isInfoEnabled()) {
                log.info("Configuring index for cache '" + config.getName()
                        + "'");
            }
            // configure the wrapper
            adapter = configureIndexedCache(config, adapter);
            
            //add the queries
            
            // start the caches
            adapter.start();
            if (log.isInfoEnabled()) {
                log.info("IndexCache Configured for cache '" + config.getName()
                        + "'");
            }

            cacheMap.put(config.getName(), adapter);
        }

    }

  
    private synchronized LifeCycleAdapter instantiateAdapter(IndexConfig config)
            throws JoftiException
    {
        LifeCycleAdapter adapter = null;

        if (config == null) {
            return adapter;
        } else {

            // create the cache adapter
            adapter = loadAdaptor(config);
            adapter.setName(config.getName());
            // now try and do the complicated stuff
            if (log.isInfoEnabled()) {
                log.info("initialising adapter '" + config.getName() + "'");
            }
            try {
                // initialise the cache - we should pass in some properties here
                adapter.init(config.getAdapterProperties());
            } catch (Throwable e) {
                log.error("Unable to instantiate adapter for '"
                        + config.getName() + "':" + e);
                try {
                    if (adapter != null) {
                        adapter.destroy();
                    }
                } catch (Throwable t) {
                    throw new JoftiException(t);
                }
                throw new JoftiException(e);
            }
            if (log.isInfoEnabled()) {
                log.info("initialised adapter '" + config.getName() + "'");
            }
        }
        return adapter;
    }

    /**
     * @param config
     * @return
     * @throws JoftiException
     */

    private LifeCycleAdapter loadAdaptor(IndexConfig config, Object cacheImpl)
            throws JoftiException
    {
        LifeCycleAdapter cache;
        if (log.isDebugEnabled()) {
            log.debug("Creating adaptor " + config.getCacheAdapter() + " for '"
                    + config.getName() + "'");
        }

        cache = (LifeCycleAdapter) createClass(config.getCacheAdapter(),
                cacheImpl);
        if (log.isDebugEnabled()) {
            log.debug("Created adaptor " + config.getCacheAdapter() + " for '"
                    + config.getName() + "'");
        }
        return cache;
    }

    private LifeCycleAdapter loadAdaptor(IndexConfig config)
            throws JoftiException
    {
        LifeCycleAdapter cache;
        if (log.isDebugEnabled()) {
            log.debug("Creating adaptor " + config.getCacheAdapter() + " for '"
                    + config.getName() + "'");
        }
        cache = (LifeCycleAdapter) createClass(config.getCacheAdapter());
        if (log.isDebugEnabled()) {
            log.debug("Created adaptor " + config.getCacheAdapter() + " for '"
                    + config.getName() + "'");
        }
        return cache;
    }

    private LifeCycleAdapter configureIndexedCache(IndexConfig config,
            LifeCycleAdapter adapter) throws JoftiException
    {


        if (config == null) {
            return null;
        } else {

            try {
                // create the parser
                if (log.isDebugEnabled()) {
                    log.debug("Creating parser " + config.getParserType()
                            + " for '" + config.getName() + "'");
                }
                ClassIntrospector parser = (ClassIntrospector) createClass(config
                        .getParserType());

                if (log.isDebugEnabled()) {
                    log.debug("Created parser " + config.getParserType()
                            + " for '" + config.getName() + "'");
                }

                //create the index
                InternalIndex index = null;

                // construct the class/method map so we can do all the
                // reflection stuff early
                if (log.isDebugEnabled()) {
                    log.debug("Parsing Config " + config.getIndexMappings()
                            + " for '" + config.getName() + "'");
                }
                parser.parseConfig(config);
                
                if (log.isDebugEnabled()) {
                    log.debug("Parsed Config for '" + config.getName() + "'");
                }
                
                if (log.isDebugEnabled()) {
                    log.debug("Creating index " + config.getIndexType()
                            + " for '" + config.getName() + "'");
                }
                
                index = GenericIndexFactory.getInstance().createIndex(
                        config.getIndexType(), parser,
                        config.getIndexProperties(),config.getName());
                
                if (log.isDebugEnabled()) {
                    log.debug("Created index " + config.getIndexType()
                            + " for '" + config.getName() + "'");
                }
                if (log.isDebugEnabled()) {
                    log.debug("adding preconfigured queries " + config.getQueryMappings()
                            + " for '" + config.getName() + "'");
                }
                
                for (Iterator it = config.getQueryMappings().entrySet().iterator();it.hasNext();){
                	Map.Entry entry = (Map.Entry)it.next();
                	index.getParserManager().addQuery((String)entry.getKey(),(String)entry.getValue());
                }
                if (log.isDebugEnabled()) {
                    log.debug("added preconfigured queries " 
                            + " for '" + config.getName() + "'");
                }
                adapter.setInternalIndex(index);

            } catch (Exception e) {
                log.error("Unable to instantiate index for '"
                        + config.getName() + "':" + e);
                // shut down the cache if we get an error configuring stuff
                try {
                    if (adapter != null) {
                        adapter.destroy();

                    }

                } catch (Throwable t) {
                    throw new JoftiException(t);
                }
                throw new JoftiException(e);

            }
        }
        return adapter;
    }

    private Object createClass(String className) throws JoftiException
    {

        Object obj = null;
        try {
            obj = ReflectionUtil.classForName(className).newInstance();
        } catch (Exception e) {
            throw new JoftiException(e);
        }
        return obj;

    }

    private Object createClass(String className, Object param)
            throws JoftiException
    {

        Object obj = null;
        try {
            Class types[] = new Class[1];
            types[0] = Object.class;
            Class clazz = ReflectionUtil.classForName(className);
            Constructor cons = clazz.getConstructor(types);
            obj = cons.newInstance(new Object[] { param });
        } catch (Throwable e) {
            throw new JoftiException(e);
        }
        return obj;

    }

    /**
     * Retrieves a cache from the manager. If the cache does not exist in the manager the 
     * method returns NULL, rather than throw an exception.
     * 
     * Attempting to retrieve a name spaced cache using this method will result in an 
     * exception.
     * @param indexName - the key name to retrive the cache. This set in the config as the cacheName.
     * @return - the cache or NULL if no cache can be found under that name.
     * 
     * @throws JoftiException
     */
    public Index getIndexCache(String indexName) throws JoftiException
    {

        if (!initialised) {
            throw new JoftiException("IndexCache has not been initialised");
        }
       
        LifeCycleAdapter adapter = (LifeCycleAdapter) cacheMap.get(indexName);

        if (adapter == null) {
            // check if lazily loaded
        	log.warn("No cache exists under the name "+indexName);
        }
        
        return (Index)adapter;
    }

    /**
     * Retrieves a name spaced indexed cache from the manager. If the cache does not exist in the manager the 
     * method returns NULL, rather than throw an exception.
     * 
     * Attempting to retrieve a non-name spaced cache using this method will result in an 
     * exception.
     * @param indexName - the key name to retrive the cache. This set in the config as the cacheName.
     * @return - the cache or NULL if no cache can be found under that name.
     * 
     * @throws JoftiException
     */
    public NameSpacedIndex getNameSpacedIndex(String indexName)
            throws JoftiException
    {

        if (!initialised) {
            throw new JoftiException("IndexCache has not been initialised");
        }
       
        LifeCycleAdapter adapter = (LifeCycleAdapter) cacheMap.get(indexName);

        if (adapter == null) {
         
                    log.warn("No cache exists under name " + indexName);
           
            if (!(adapter instanceof NameSpacedCacheAdapter)) {
                return null;
            }

        }
        return (NameSpacedIndex) adapter;
    }

   

    private synchronized Map loadConfig(String configfile)
            throws JoftiException
    {

        ConfigFileParser confParser = new ConfigFileParser();
        return confParser.parseIndexConfig(configfile);

    }

    private synchronized IndexConfig loadConfig(IndexConfig config,
            String classesFileName) throws JoftiException
    {

        ConfigFileParser confParser = new ConfigFileParser();
        return confParser.parseClassMapping(config, classesFileName);

    }
    
    private synchronized IndexConfig loadConfig(IndexConfig config,
            InputStream classes) throws JoftiException
    {

        ConfigFileParser confParser = new ConfigFileParser();
        return confParser.parseClassMapping(config, classes);

    }
    private synchronized Map loadConfig(InputStream configStream)
            throws JoftiException
    {

        ConfigFileParser confParser = new ConfigFileParser();
        return confParser.parseIndexConfig(configStream);

    }

    
    /**
     * Used to shutdown and remove a cache from the manager. You should always use this method
     * to remove a cache from the manager - as some cache implementations explicitly require a 
     * shutdown phase to be run before they can be removed.
     * 
     * @param cache - the cache to be destroyed.
     */
    public synchronized void destroyIndex(Object cache)
    {
        if (cache instanceof LifeCycleAdapter) {
        	LifeCycleAdapter wrapper = (LifeCycleAdapter) cache;
            removeCache(wrapper);
        }
    }

    
    protected synchronized void removeCache(LifeCycleAdapter wrapper)
    {

        if (wrapper != null) {
            try {
                wrapper.destroy();
            } catch (Exception e) {
                log.error(e);
            } finally {
                cacheMap.remove(wrapper.getName());
            }
        }

    }


    /* (non-Javadoc)
     * @see com.jofti.api.IndexManager#destroy()
     */
    public synchronized void destroy()
    {
        for(Iterator it = cacheMap.values().iterator();it.hasNext();){
        	LifeCycleAdapter adapter= (LifeCycleAdapter)it.next();
            removeCache(adapter);
        }
        configMap.clear();
        initialised =false;
        
    }
    

    /* (non-Javadoc)
     * @see com.jofti.api.IndexManager#removeIndexedCache(java.lang.Object)
     */
    public synchronized Object removeIndex(Object cache)
    {
    	 if (cache instanceof LifeCycleAdapter) {
            
             Object obj = cacheMap.remove(((LifeCycleAdapter)cache).getName());
             return obj;
         }
    	 return null;
        
    }

}
