package com.jofti.parser;

import java.io.Reader;
import java.io.StringReader;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import antlr.collections.AST;

import com.jofti.api.IndexQuery;
import com.jofti.core.IParsedQuery;
import com.jofti.core.IPredicate;
import com.jofti.exception.JoftiException;
import com.jofti.introspect.ClassIntrospector;
import com.jofti.parser.ejb.EJB3QueryParser;
import com.jofti.query.EJBQuery;
import com.jofti.util.ArrayComparator;
import com.jofti.util.CompositeComparator;
import com.jofti.util.ReflectionComparator;
import com.jofti.util.ReflectionUtil;

/**
 * This class does the work in turning a test EJBQuery into an IParsedQuery, which is the 
 * internal query representation.</p>
 * 
 * 
 * @author xenephon
 * @version 1.0
 */
public class EJBQueryParser extends AbstractParser implements IQueryParser {

	// constructs a parser with a set number of cache queries
	public EJBQueryParser(final int cachedQueries, ClassIntrospector introspector) {
		super(cachedQueries, introspector);
	}

	// default constructor does 100 queries
	public EJBQueryParser( ClassIntrospector introspector) {
		this(100, introspector);
	}

	private static Log log = LogFactory.getLog(ParserManager.class);

	/* (non-Javadoc)
	 * @see com.jofti.parser.IQueryParser#parseQuery(com.jofti.api.IndexQuery)
	 */
	public IParsedQuery parseQuery(IndexQuery originalQuery)
			throws JoftiException {


		return parseQuery(((EJBQuery)originalQuery).getQuery(),originalQuery);

	}

	public IParsedQuery  getQuery(String name){

		ParsedQuery query = null;
		synchronized (lockObject) {
			query = (ParsedQuery) compiledQueries.get(name);
		}
		if (query != null){
			ParsedQueryWrapper wrapper = new ParsedQueryWrapper(query);
			
//		 add in the name parameters if we have any
			wrapper.setNamedValueMap(new HashMap());

			return wrapper;
		}
		return null;

	}
	
	public IParsedQuery parseQuery(String name, IndexQuery originalQuery)
	throws JoftiException {

// get text string
EJBQuery tempQuery = (EJBQuery) originalQuery;



ParsedQuery parsedQuery = null;

//see if query is already cached - this ignores parameters - and only checks the String format
synchronized (lockObject) {
	parsedQuery = (ParsedQuery) compiledQueries.get(name);
}

// we have a cached query here
if (parsedQuery == null) {			
	// we have to parse the query
	if (log.isDebugEnabled()) {
		log.debug("parsing new query " + tempQuery.getQuery());
	}
	// parse the query
	
	parsedQuery = parseQuery(new ParsedQuery(), tempQuery.getQuery());
	
	parsedQuery.setFirstResult(tempQuery.getFirstResult());
	
	//add returned query into cache
	synchronized (lockObject) {
		compiledQueries.put(name, parsedQuery);
	}
}

//  we use a wrapper here as queries are actually cached for reuse
// and wrapper is where we set the values - otherwise we would have to
// produce a copy of the query

ParsedQueryWrapper wrapper = new ParsedQueryWrapper(parsedQuery);

//		 add in the name parameters if we have any
	wrapper.setNamedValueMap(tempQuery.getParameterMap());
	

	wrapper.setFirstResult(tempQuery.getFirstResult());
	wrapper.setMaxResults(tempQuery.getMaxResults());

return wrapper;
}
	
	private ParsedQuery parseQuery(ParsedQuery parsedQuery, String selectClause)
			throws JoftiException {
		Reader input = null;
		try {
			input = new StringReader(selectClause);

			CommonLexer lexer = new CommonLexer(input);

			EJB3QueryParser parser = new EJB3QueryParser(lexer);

			parser.queryStatement();

			AST ast = parser.getAST();
			AST selectStatement = null;

			if (ast == null || ast.getFirstChild() == null) {

				throw new JoftiException(
						"Query '"
								+ selectClause
								+ "' is not of format 'SELECT variable[[,] variable]* FROM Class [AS] identifier[,Class [AS] identifier] WHERE predicate [ORDER BY identifier [ASC|DESC]]*'");
			}
			if (log.isDebugEnabled()) {
				log.debug(ast.toStringTree());
			}

			parsedQuery = parseClause(ast, parsedQuery);

		} catch (Exception e) {
			throw new JoftiException(e);
		} finally {
			try {
				input.close();
			} catch (Exception e) {
				log.warn("problem closing reader", e);
			}
		}
		return parsedQuery;
	}

	private ParsedQuery parseClause(AST ast, ParsedQuery parsedQuery)
			throws JoftiException {
		AST orderNode =null;
		do {
			//we have no where clause
			switch (ast.getType()) {
			case CommonLexerTokenTypes.NAMESPACE:
				// we need to do from first
				parsedQuery = parseNameSpace(ast.getFirstChild()
						.getNextSibling(), parsedQuery);

				//              now reset the node to the where or from clause node
				ast = ast.getFirstChild();
				break;
			case CommonLexerTokenTypes.WHERE:
				// we need to do from first
				parsedQuery = parseFrom(ast.getFirstChild().getFirstChild()
						.getNextSibling(), parsedQuery);

				//            	 now get the stuff out of the where clause

				parsedQuery = parseWhere(ast.getFirstChild().getNextSibling(),
						parsedQuery);

				//                  now reset the node to the select node
				ast = ast.getFirstChild().getFirstChild();
				break;
			//	                  now reset the node to the select node
			case CommonLexerTokenTypes.FROM:
				parsedQuery = parseFrom(ast.getFirstChild().getNextSibling(),
						parsedQuery);

				// we can now parse the select part
				ast = ast.getFirstChild();
				break;
			// where is first

			case CommonLexerTokenTypes.SELECT:
				parsedQuery = parseSelect(ast, parsedQuery);

				// we can now parse the select part
				ast = ast.getFirstChild();
				break;
			case CommonLexerTokenTypes.ORDER:
				orderNode =ast;
				//mark for later
			//	parsedQuery = parseOrder(ast, parsedQuery);
				ast = ast.getFirstChild();
				break;
			// skip through the remianing arguments in the select
			default:
				ast = ast.getFirstChild();
				break;
			}
		} while (ast != null);
		
		if (orderNode != null){
			parsedQuery = parseOrder(orderNode, parsedQuery);
		}
		return parsedQuery;
	}

	private ParsedQuery parseWhere(AST ast, ParsedQuery parsedQuery)
			throws JoftiException {
		Stack stack = new Stack();

		stack = parseWherePredicates(ast, stack, parsedQuery);

		if (log.isDebugEnabled()) {
			log.debug(stack);
		}

		// we need to reverse the stack here
		Stack temp = new Stack();

		boolean predicateElement = false;

		while (stack.size() != 0) {
			Object obj = stack.pop();
			if (predicateElement) {
				if (obj instanceof IPredicate) {
					throw new JoftiException(
							"Query not formatted correctly a join operator must seperate two predicates");
				}
				predicateElement = false;
			} else {
				if (obj instanceof Operator) {
					throw new JoftiException(
							"Query not formatted correctly a join operator must seperate two predicates");
				}
				predicateElement = true;
			}
			temp.push(obj);
		}

		parsedQuery.setPredicates(temp);
		return parsedQuery;
	}

	private ParsedQuery parseNameSpace(AST expression, ParsedQuery query)
			throws JoftiException {

		AST node = expression;
		String nameSpace = "";
		do {
			if (log.isDebugEnabled()) {
				log.debug("parsing select for " + node.toStringTree());
			}

			nameSpace = nameSpace + node.getText();
			node = node.getNextSibling();

		} while (node != null);

		query.setNameSpace(nameSpace);
		return query;

	}

	private ParsedQuery parseSelect(AST expression, ParsedQuery query)
			throws JoftiException {

		// will be alias for field
		AST node = expression.getFirstChild();

		do {
			// we have an as

			if (log.isDebugEnabled()) {
				log.debug("parsing select for " + node.toStringTree());
			}
			// put the aliases into the alias and return map
			addToFieldMap(query.getAliasMap(), query.getResultFieldsMap(), node
					.getText());
			node = node.getNextSibling();

		} while (node != null);

		return query;

	}
	
	

	private ParsedQuery parseFrom(AST expression, ParsedQuery query)
			throws JoftiException {

		//first one must be an alias identifier
		AST node = expression;

		do {
			// we have an as
			if (node.getType() == CommonLexerTokenTypes.ALIAS_IDENTIFIER) {
				if (log.isDebugEnabled()) {
					log.debug("alias found for class "
							+ node.getFirstChild().toStringTree()
							+ " of "
							+ node.getFirstChild().getNextSibling()
									.toStringTree());
				}

				// try and construct a className from the class identifier
				Class clazz = null;
				try {
					clazz = ReflectionUtil.classForName(node.getFirstChild()
							.getText());
				} catch (Exception e) {
					throw new JoftiException("Class "
							+ node.getFirstChild().getText()
							+ " cannot be parsed in From clause", e);
				}
				if (clazz.isPrimitive() || clazz == Boolean.class){
					clazz = introspector.boxPrimitive(clazz);
				}
				query.getAliasMap().put(
						node.getFirstChild().getNextSibling().getText(), clazz);
				node = node.getNextSibling();
			} else {
				// if we want to deal with in() we should do it here
				// even though the AST copes with it I do not think the 
				// feature lends itself to the index query
				
				// we have no as - we just use it as normal
				
				// not dealing with in stuff here
				Class clazz = null;
				try {
					clazz = ReflectionUtil.classForName(node.getText());
				} catch (Exception e) {
					throw new JoftiException("Class " + node.getText()
							+ " cannot be parsed in From clause", e);
				}
				if (clazz.isPrimitive() || clazz == Boolean.class){
					clazz = introspector.boxPrimitive(clazz);
				}
				query.getAliasMap().put(node.getNextSibling().getText(), clazz);
				//skip the sibling
				node = node.getNextSibling().getNextSibling();
			}

		} while (node != null);

		return query;

	}

	
	
	
	
	
	
	private void addToFieldMap(Map aliasMap, Map returnMap, String alias)
			throws JoftiException {
		int aliasSeperator = alias.indexOf('.');
		
		if (aliasSeperator > 0) {
			Class clazz = getClassForAlias(aliasMap, alias);
			ClassFieldMethods fieldSet = (ClassFieldMethods) returnMap.get(clazz);
			if (fieldSet == null) {
				fieldSet = new ClassFieldMethods(clazz, null);
				if (fieldSet.getFieldMap() == null) {
					fieldSet.setFieldMap(new LinkedHashMap());
				}
				returnMap.put(clazz,fieldSet);
			} 
				
//				 we need to get the methods for the field - 
				// we do it here as it is only done once for the query
				
				String attribute = alias.substring(aliasSeperator + 1, alias.length());
				
				if (fieldSet.getFieldMap().containsKey(attribute)){
					throw new JoftiException("Field "+ attribute + " is already part of the return set for "+clazz);
				}
				Object[] methods = introspector.getMethodsForAttribute(clazz,attribute);
				if (methods == null || methods.length ==0){
					// log.warn
					throw new JoftiException("Attribute "+ attribute + " is not valid for "+ clazz);
				}else{
					fieldSet.getFieldMap().put(attribute,methods);
				}

		} else {
			if (!aliasMap.containsKey(alias)) {
				throw new JoftiException("select for identifier " + alias
						+ " is not mentioned in from clause");
			} else if (returnMap.containsKey(aliasMap.get(alias))) {
				if (returnMap.get(aliasMap.get(alias)) == null || 
						((ClassFieldMethods)returnMap.get(aliasMap.get(alias))).getFieldMap() == null ||
						((ClassFieldMethods)returnMap.get(aliasMap.get(alias))).getFieldMap().size()==0 ){
					throw new JoftiException(""+ aliasMap.get(alias) +
							" under alias " + alias + "  is listed more than once in select clause");
				}else 
				throw new JoftiException(" "+ aliasMap.get(alias) +
						 " as whole object under alias " + alias +"  cannot be selected if fields from class are already in select clause");
			}

			returnMap.put(aliasMap.get(alias), new ClassFieldMethods(
					(Class) aliasMap.get(alias), null));

		}
	}
}