/*
 * Created on 13-Jul-2005
 *
 */
package com.jofti.query;


import java.util.HashMap;
import java.util.Map;

import com.jofti.api.IndexQuery;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;

/**
 * 
 *
 * Basic usage is:<br> "select ${identifier} from ${classname} AS ${identifier} where ${identifier}.${propertyname} [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter]"
 * 
 * <p></p>
 * multiproperty queries are of the form<br>
 * "select ${identifier} from ${classname} AS ${identifier} where  ${identifier}.${propertyname}  [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter] [and,or] ${identifier}.${propertyname} 
 * [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter]"
 * <p>
 * compound queries are supported as ${query} [and,or] ${query}
 * <p>
 * ordering for compound queries is achieved through use of bracket groups<br>
  * "... where ( ${identifier}.${propertyname}   [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter]  [and,or] ${identifier}.${propertyname} [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter] ) 
  * [and,or]  ${identifier}.${propertyname}  [=,<,>,!=,>=,<=, LIKE, IN] ['value',:namedParameter,?positionalParameter] "
  <p>
 * 
 * multiclass queries are of the form
 * <p>
 * select ${identifierA}, ${identifierB} From ${classname} AS ${identifierA}, ${classname} AS ${identifierB} where ${identifierA}.${propertyname} [=,<,>,!=,>=,<=] 'value' [and.or] ${identifierB}.${propertyname}  [=,<,>,!=,>=,<=] 'value'"
 * <p>
 * 
 * All values are created using a reflective String constructor .So if you want to Compare an Integer(22) you just specify
 * value=22  - the value type is chosen based on the declared type of the field.
 * <p>
 * For primitives, primitive wrappers or some natively comparable classes in the JVM the propertyname is always the literal string 'VALUE' (case insensitive)
 * * 		<ul>
 * 			<li>java.lang.String.class,
    		<li>java.lang.Integer.class,
    		<li>java.lang.Double.class,
    		<li>java.lang.Float.class,
    		<li>java.lang.Long.class,
    		<li>java.lang.Short.class,
    		<li>java.math.BigInteger.class,
    		<li>java.math.BigDecimal.class,
    		<li>java.util.Date.class,
    		<li>java.net.URI.class 
    		<li>java.lang.Byte.class
    		<li>java.lang.Character.class
    		<li>java.sql.Timestamp.class
    		</ul>  
 * <p>
 * For example to query an integer you would write
 * <p>
 * select i from java.lang.Integer i  where i.VALUE = 20
 * <p>
 * or <br>
 * select i from java.lang.Integer as i where i.VALUE =20
 * <p>
 * 
 * Dates are constructed using the default encoding of the JVM. So a UK encoding would be:
 * <p>
 * select d from java.util.Date d where d.value <='4/10/1901 00:00:00' 
 * <p>
 * For user defined classes in order to query the property type must have a String constructor and must 
 * implement equals and hashCode correctly.
 * 
 *  <p>
 * 
 * For those classes that do not support String constructors or if you wish to use already existing values then you can use either 
 * the Named parameter or Positional parameter format.
 * </p>
 * 
 * <p>
 * For instance to query for an integer as above we can use:<br>
 * new EJBQuery("select i from java.lang.Integer as i where i.value =:intVal").setParameter("intVal", new Integer(20));
 *  <br>
 * or<br>
 * new EJBQuery("select i from java.lang.Integer as i where i.value =?1").setParameter(1, new Integer(20));
 * </p>
 * 
 * 
 *  @author Steve Woodcock
 *  @version 1.0
 */
public class EJBQuery implements IndexQuery, QueryId {

	String query = null;
	
	int hashCode =0;
	Map parameterMap = new HashMap();
	
	int maxResults;
	int firstResult;
	
    static final QueryType QUERY_ID=QueryType.UNPARSED_QUERY;
    
	private static final String TERMINATOR =";";
	
	public EJBQuery(String query){
		this.query = query + TERMINATOR;
	}
	
	public String getQuery(){
		return this.query;
	}
    

    public QueryType getQueryType()
    {
        
        return QUERY_ID;
    }
    
    /**
     * Sets a parameter that must match a named parameter in the query string.<br>
     * The nameParameter in the Query is in the form :name and the parameter is set as "name" 
     * without the leading ':'. The value can be any object but its type must be assignable to the parameter 
     * type using normal Java assignability rules. So if the parameter is a String we cannot set an Integer 
     * and expect the type to be correctly coerced.<p>
     * @param name - the name of the parameter
     * @param value - the value the query should compare against
     * @return - itself to allow method chaining.
     */
    public IndexQuery setParameter(String name, Object value){
    	parameterMap.put(name, value);
    	return this;
    }
    
    
    /**
     * Sets a parameter that must match a poitional parameter in the query string.<br>
     * The positional Parameter in the Query is in the form ?number and the parameter is set as an int. The value can be any object but its type must be assignable to the parameter 
     * type using normal Java assignability rules. So if the parameter is a String we cannot set an Integer 
     * and expect the type to be correctly coerced.<p>
     * @param parameter - the position of the parameter
     * @param value - the value the query should compare against
     * @return - itself to allow method chaining.
     */
    public IndexQuery setParameter(int parameter, Object value){
    	parameterMap.put(""+parameter, value);
    	return this;
    }
    
    /**
     * Returns a Map of the parameters set on the query keyed by name and/or position. Positional parameters 
     * are stored in the Map as String representation of the int 
     * @return - the map of parameters.
     */
    public Map getParameterMap(){
    	return parameterMap;
    }
    
    public void clearParameters(){
    	parameterMap.clear();
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    public int hashCode(){
    	if (hashCode ==0){
    		hashCode = query.hashCode();
    	}
    	return hashCode;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj){
    	return query.equals(obj);
    }
    
    public String toString(){
    	return query;
    }

	public IndexQuery setFirstResult(int firstResult) {
		this.firstResult= firstResult;
		return this;
	}

	public int getFirstResult() {
		
		return firstResult;
	}
	
	public int getMaxResults() {
		
		return maxResults;
	}

	public IndexQuery setMaxResults(int maxResults) {
		this.maxResults=maxResults;
		return this;
	}
}
