package com.jofti.query;

import com.jofti.api.IndexQuery;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.util.ReflectionUtil;

/**
 * 
 *
 * A utility class to enable simple queries to be done easily. The Query matches everything in the 
 * value array passed in for that particular field. For example, x IN [val1, val2, val3] is equivalent to testing [x = val1, or x= = val2, or x = val3].
 * 
 * 
 *  This query cannot be combined with any other. iF you want to construct more complex queries use the @link com.jofti.query.Query or EJBQuery class.
  *<p>
  *@author Steve Woodcock
 */
public class MatchInQuery implements IndexQuery, QueryId {

	int hashCode =0;
	Class className;
	String propertyName;
	Comparable[] values;
	final public Object alias;
    static final QueryType QUERY_ID=QueryType.IN_QUERY;
	
	/**
	 * Construct a query supplying the classname of the object type to be returned,and the 
	 * field to be queried and the values to be used. <p>
	 * 
	 * The field type in the object and each of the values must be of the same type or be able 
	 * to be coerced using java's assingability rules.
	 * 
	 * <p>
	 * An example usage  would be:
	 * <p>
	 * new MatchInQuery(org.package.Myclass.class, "myProperty" ,[20,21,22]);
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param propertyName - the field name to use
	 * @param values - the values that are used as the search value.
	 */
	public MatchInQuery(Class className, String propertyName, Comparable[] values){
		this(className, propertyName,values,null);
	}
	
	
	public MatchInQuery(Class className, String propertyName, Comparable[] values, Object alias){
		this.className = className;
		this.propertyName = propertyName;
		this.values = values;
		this.alias =alias;
	}
	
	/**
	 * Construct a query supplying the classname of the object type to be returned,and the 
	 * field to be queried and the values to be used. <p>
	 * 
	 * The field type in the object and each of the values must be of the same type or be able 
	 * to be coerced using java's assingability rules.
	 * 
	 * <p>
	 * An example usage  would be:
	 * <p>
	 * new MatchInQuery("org.package.Myclass", "myProperty" ,[20,21,22]);
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param propertyName - the field name to use
	 * @param values - the values that are used as the search value.
	 */
	
	public MatchInQuery(String className, String propertyName, Comparable[] values){
		this(className, propertyName,values,null);
	}
	
	public MatchInQuery(String className, String propertyName, Comparable[] values, Object alias){
		Class clazz = null;
		try{
			clazz = ReflectionUtil.classForName(className);
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		this.className = clazz;
		this.propertyName = propertyName;
		this.values = values;
		this.alias =alias;
	}
	/**
	 * Construct a query supplying the values to be searched against. 
	 * This is a convenience method for classes (such as String,Integer etc)
	 * that have no property value as such. Instead the value is the class type of the values 
	 * in the array.
	 * <p>
	 * 
	 * An example usage  would be:
	 * <p>
	 * new MatchInQuery(["test","test1"]);
	 * <p>
	 * This is so you do not have to use the methods above in the manner of
	 * <p>
	 * new MatchInQuery("java.lang.String", null ,["test","test1"]); 
	 * <p>
	 * @param values - the values that are used as the search value.
	 */
	public MatchInQuery(Comparable[] values){
		this.values = values;
		this.alias =null;
	}
	/**
	 * @return Returns the className.
	 */
	public Class getClassName() {
		return className;
	}

	/**
	 * @return Returns the propertyName.
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * @return Returns the values.
	 */
	public Comparable[] getValues() {
		return values;
	}

    public QueryType getQueryType()
    {
        return  QUERY_ID;
    }

  
	public Object getAlias() {
		return alias;
	}
	
	public IndexQuery setParameter(String name, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");
	}
	/* (non-Javadoc)
	 * @see com.jofti.api.IndexQuery#setParameter(int, java.lang.Object)
	 */
	public IndexQuery setParameter(int position, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");

	}
	
	public IndexQuery setFirstResult(int firstResult) {
		throw new UnsupportedOperationException("result limits are not supported for convenience classes");

	}
	public IndexQuery setMaxResults(int maxResults) {
		throw new UnsupportedOperationException("result limits are not supported for convenience classes");

	}
}
