package com.jofti.query;

import com.jofti.api.IndexQuery;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.util.ReflectionUtil;

/**
 * 
 *
 * A utility class to enable simple queries to be done easily. The Query matches everything not equal the 
 * value passed in for that particular field. This is equivalent to !=.<p>
 * 
 * 
 *  This query cannot be combined with any other. iF you want to construct more complex queries use the @see com.jofti.query.Query class.
 *  <p>
 *  @author Steve Woodcock
 */
public class MatchNotQuery implements IndexQuery, QueryId {

	Class className;
	String propertyName;
	Comparable value;
    
	public final Object alias;
    final QueryType QUERY_ID =QueryType.NOT_QUERY;
	
	/**
	 * Construct a query supplying the classname of the object type to be returned,and the 
	 * field to be queried and the value to be used. <p>
	 * 
	 * The field type in the object and the value must be of the same type.
	 * <p>
	 * 
	 * An example usage  would be:
	 * <p>
	 * new MatchNotQuery(org.package.Myclass.class, "myProperty" ,"some value");
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param propertyName - the field name to use
	 * @param value - the value that is used as the search value.
	 */
	
	public MatchNotQuery(Class className, String propertyName, Comparable value){
		this(className,propertyName,value,null);
	}
	
	public MatchNotQuery(Class className, String propertyName, Comparable value,Object alias){
		this.className = className;
		this.propertyName = propertyName;
		this.value = value;
		this.alias =alias;
	}
	
	/**
	 * Construct a query supplying the classname of the object type to be returned,and the 
	 * field to be queried and the value to be used. <p>
	 * 
	 * The field type in the object and the value must be of the same type.
	 * <p>
	 * 
	 * An example usage  would be:
	 * <p>
	 * new MatchNotQuery("org.package.Myclass", "myProperty" ,"some value");
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param propertyName - the field name to use
	 * @param value - the value that is used as the search value.
	 */
	
	public MatchNotQuery(String className, String propertyName, Comparable value){
		this(className,propertyName,value,null);
	}
	
	public MatchNotQuery(String className, String propertyName, Comparable value,Object alias){
		Class clazz = null;
		try{
			clazz = ReflectionUtil.classForName(className);
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		this.className = clazz;
		this.propertyName = propertyName;
		this.value = value;
		this.alias =alias;
	}
	/**
	 * Construct a query supplying the value to be searched against. 
	 * This is a convenience method for classes (such as String,Integer etc)
	 * that have no property value as such. Instead the value is the class type.
	 * <p>
	 * 
	 * An example usage  would be:
	 * <p>
	 * new MatchNotQuery("some value");
	 * <p>
	 * This is so you do not have to use the methods above in the manner of
	 * <p>
	 * new MatchNotQuery("java.lang.String", null ,"some value"); 
	 * <p>

	 * @param value - the value that is used as the search value.
	 */
	public MatchNotQuery(Comparable value){
		this.value = value;
		this.alias =null;
	}
	/**
	 * @return Returns the className.
	 */
	public Class getClassName() {
		return className;
	}

	/**
	 * @return Returns the propertyName.
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * @return Returns the value.
	 */
	public Comparable getValue() {
		return value;
	}

    public QueryType getQueryType()
    {
       
        return QUERY_ID;
    }

	
	public Object getAlias() {
		return alias;
	}
	
	public IndexQuery setParameter(String name, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");
	}
	/* (non-Javadoc)
	 * @see com.jofti.api.IndexQuery#setParameter(int, java.lang.Object)
	 */
	public IndexQuery setParameter(int position, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");

	}
	
	public IndexQuery setFirstResult(int firstResult) {
		throw new UnsupportedOperationException("result limits are not supported for convenience classes");

	}
	public IndexQuery setMaxResults(int maxResults) {
		throw new UnsupportedOperationException("result limits are not supported for convenience classes");

	}
}
