package com.jofti.query.namespace;

import com.jofti.api.IndexQuery;
import com.jofti.cache.adapter.NameSpaceWrapper;
import com.jofti.core.INameSpace;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.util.ReflectionUtil;

/**
 * 
 *
* A utility class to enable simple queries to be done easily. The Query matches everything does not match 
* the value for particular field. This is equivalent to !=.
 * <p>
 * All nameSpace queries must provide a correct namespace type for the implementation. Some nameSPaces 
 * are hierachical and so the search will use the nameSpace as starting point. Others are flat and so there is no 
 * traversal step.
 * <p>
 * This query cannot be combined with any other. iF you want to construct more complex queries use the @link com.jofti.query.Query class.
 *@author steve Woodcock
 */
public class MatchNSNotQuery implements IndexQuery,INameSpaceAware,QueryId  {

	Class className;
	Object nameSpace;
	String propertyName;
	Comparable value;
	public final Object alias;
	static final QueryType QUERY_ID=QueryType.NOT_NS_QUERY;
	/**
	 * Construct a query supplying the classname of the object type to be returned, the namespace 
	 * under which to start the search. The field to be queried and the value to be used. 
	 * <p>
	 * The field type in the object and the value must be of the same type.
	 * <p>
	 * 
	 * An example usage (for JBossCache) would be:
	 * <p>
	 * new MatchNSNotQuery("org.package.Myclass", "/test", "myProperty" ,"some value");
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param nameSpace - the name space to be searched.
	 * @param propertyName - the field name to use
	 * @param value - the value that is used as the search value.
	 */
	
	public MatchNSNotQuery(Class className, Object nameSpace,String propertyName, Comparable value){
		this(className,nameSpace,propertyName,value,null);
	}
	public MatchNSNotQuery(Class className, Object nameSpace,String propertyName, Comparable value,Object alias){
		this.className = className;
		this.propertyName = propertyName;
		this.value = value;
		this.nameSpace = nameSpace;
		this.alias = alias;
	}
	
	public MatchNSNotQuery(String className, Object nameSpace,String propertyName, Comparable value){
		this(className,nameSpace, propertyName,value,null);
	}
	
	public MatchNSNotQuery(String className, Object nameSpace,String propertyName, Comparable value,Object alias){
		Class clazz = null;
		try{
			clazz = ReflectionUtil.classForName(className);
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		this.className = clazz;
		this.propertyName = propertyName;
		this.value = value;
		this.nameSpace = nameSpace;
		this.alias = alias;
	}
	
	
	/**
	 * Construct a query supplying the value to be searched against and the namespace 
	 * under which to start the search. This is a convenience method for classes (such as String,Integer etc)
	 * that have no property value as such. Instead the value is the class type.
	 * <p>
	 * 
	 * An example usage (for JBossCache) would be:
	 * <p>
	 * new MatchNSNotQuery("/test","some value");
	 * <p>
	 * This is so you do not have to use the methods above in the manner of
	 * <p>
	 * new MatchNSNotQuery("java.lang.String", "/test", null ,"some value"); 
	 * <p>

	 * @param nameSpace - the name space to be searched.
	 * @param value - the value that is used as the search value.
	 */
	public MatchNSNotQuery(Object nameSpace,Comparable value){
		this(nameSpace,value,null);
	}
	public MatchNSNotQuery(Object nameSpace,Comparable value,Object alias){
		this.value = value;
		this.nameSpace = nameSpace;
		this.alias = alias;
		
	}
	
	/**
	 * @return Returns the className.
	 */
	public Class getClassName() {
		return className;
	}

	/**
	 * @return Returns the propertyName.
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * @return Returns the value.
	 */
	public Comparable getValue() {
		return value;
	}

	
	
	
	/**
	 * @return Returns the wrapper.
	 */
	public synchronized INameSpace getNameSpaceWrapper() {
		
		if (nameSpace instanceof INameSpace){
			return (INameSpace) nameSpace;
		}else{
			return new NameSpaceWrapper(nameSpace);
		}
	}
	
	/**
	 * @return Returns the nameSpace.
	 */
	public synchronized Object getNameSpace() {
		return nameSpace;
	}
	/**
	 * @param nameSpace The nameSpace to set.
	 */
	public synchronized void setNameSpace(Object nameSpace) {
		this.nameSpace = nameSpace;
	}
	
	public boolean equals(Object o){
		if (o instanceof MatchNSNotQuery){
			MatchNSNotQuery temp = (MatchNSNotQuery)o;
			boolean result = nameSpace.equals(temp.nameSpace);
			if (!result){
				return result;
			}
			if (className != null && temp.className != null){
				result = className.equals(temp.className);
				if (!result){
					return result;
				}
			}else {
				result = className == null & temp.className == null;
				if (!result){
					return result;
				}
			}
	
			if (propertyName != null && temp.propertyName != null){
				result = propertyName.equals(temp.propertyName);
				if (!result){
					return result;
				}
			}else {
				result = propertyName == null & temp.propertyName == null;
				if (!result){
					return result;
				}
			}
			if (value != null && temp.value != null){
				result = value.equals(temp.value);
				if (!result){
					return result;
				}
			}else {
				result = value == null & temp.value == null;
				if (!result){
					return result;
				}
			}
			return result;
		}
		return false;
	}
	
	public int hashCode(){
		int temp = nameSpace.hashCode(); 
		if (className != null){
			temp+= className.hashCode();
		}
		if (propertyName != null){
			temp+= propertyName.hashCode();
		}
		if (value != null){
			temp+= value.hashCode();
		}
		return temp;
		
		
	}
	public Object getAlias() {
		return alias;
	}
	/* (non-Javadoc)
	 * @see com.jofti.core.QueryId#getQueryType()
	 */
	public QueryType getQueryType() {
		
		return QUERY_ID;
	}
	
	public IndexQuery setParameter(String name, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");
	}
	/* (non-Javadoc)
	 * @see com.jofti.api.IndexQuery#setParameter(int, java.lang.Object)
	 */
	public IndexQuery setParameter(int position, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");

	}
	public IndexQuery setFirstResult(int firstResult) {
		throw new UnsupportedOperationException("Result limitation is not supported for convenience classes");
	}
	public IndexQuery setMaxResults(int maxResults) {
		throw new UnsupportedOperationException("Result limitation is not supported for convenience classes");
	}
}
