package com.jofti.query.namespace;

import com.jofti.api.IndexQuery;
import com.jofti.cache.adapter.NameSpaceWrapper;
import com.jofti.core.INameSpace;
import com.jofti.core.INameSpaceAware;
import com.jofti.core.QueryId;
import com.jofti.core.QueryType;
import com.jofti.util.ReflectionUtil;

/**
 * 
 *
* A utility class to enable simple queries to be done easily. The Query matches  
* ranges  for particular field. This is equivalent to (>= and <=) or (> and <).
 * <p>
 * All nameSpace queries must provide a correct namespace type for the implementation. Some nameSpaces 
 * are hierachical and so the search will use the nameSpace as starting point. Others are flat and so there is no 
 * traversal step.
 * <p>
 * This query cannot be combined with any other. iF you want to construct more complex queries use the @link com.jofti.query.Query class.
 *<p>
 * @author steve Woodcock
 */
public class MatchNSRangeQuery implements IndexQuery, INameSpaceAware,QueryId {

	public final Object alias;
	Object nameSpace;
	Class className;
	String propertyName;
	Comparable startValue;
	Comparable endValue;
	NameSpaceWrapper wrapper;
	boolean inclusive =true;
	
	static final QueryType QUERY_ID=QueryType.RANGE_NS_QUERY;
	/**
	 * Construct a query supplying the classname of the object type to be returned, the namespace 
	 * under which to start the search. The field to be queried and the start and finish values to be used. 
	 * <p>
	 * The field type in the object and the values must be of the same type. In addition the end value must
	 * be greater (comparaison >0 ) than the start value;
	 * <p>
	 * An example usage (for JBossCache) would be:
	 * <p>
	 * new MatchNSQuery("org.package.Myclass", "/test", "myProperty" ,new Integer(10),new Integer(20));
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param nameSpace - the name space to be searched.
	 * @param propertyName - the field name to use
	 * @param startValue - the value that is used as the start search value.
	 * @param endValue - the value that is used as the end search value.
	 */
	public MatchNSRangeQuery(Class className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue){
		this(className,nameSpace,propertyName,startValue,endValue,null);
	}
	public MatchNSRangeQuery(Class className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue,Object alias){
		this.className = className;
		this.propertyName = propertyName;
		this.startValue = startValue;
		this.endValue = endValue;
		this.nameSpace = nameSpace;
		this.alias=alias;
	}
	
	public MatchNSRangeQuery(String className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue){
		this(className,nameSpace,propertyName,startValue,endValue,null);
	}
	public MatchNSRangeQuery(String className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue,Object alias){
		Class clazz = null;
		try{
			clazz = ReflectionUtil.classForName(className);
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		this.className = clazz;
		this.propertyName = propertyName;
		this.startValue = startValue;
		this.endValue = endValue;
		this.nameSpace = nameSpace;
		this.alias =alias;
	}
	
	/**
	 * Construct a query supplying the classname of the object type to be returned, the namespace 
	 * under which to start the search. The field to be queried and the start and finish values to be used. 
	 * This method also allows inclusivity to be set.
	 * <p>
	 * The field type in the object and the values must be of the same type. In addition the end value must
	 * be greater (comparaison >0 ) than the start value; 
	 * <p>
	 * An example usage (for JBossCache) would be:
	 * <p>
	 * new MatchNSRangeQuery("org.package.Myclass", "/test", "myProperty" ,new Integer(10),new Integer(20), true);
	 * <p>
	 * @param className - the class of object to be returned.
	 * @param nameSpace - the name space to be searched.
	 * @param propertyName - the field name to use
	 * @param startValue - the value that is used as the start search value.
	 * @param endValue - the value that is used as the end search value.
	 * @param inclusive - whether the seacrh should be inclusive of values.
	 */
	
	public MatchNSRangeQuery(Class className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue, boolean inclusive){
		this(className,nameSpace,propertyName,startValue,endValue,inclusive,null);
	}
	public MatchNSRangeQuery(Class className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue, boolean inclusive,Object alias){
		this.className = className;
		this.propertyName = propertyName;
		this.startValue = startValue;
		this.endValue = endValue;
		this.nameSpace = nameSpace;
		this.inclusive = inclusive;
		this.alias = alias;
	}
	
	public MatchNSRangeQuery(String className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue, boolean inclusive){
		this(className,nameSpace,propertyName,startValue,endValue,inclusive,null);
	}
	public MatchNSRangeQuery(String className, Object nameSpace, String propertyName, Comparable startValue,Comparable endValue, boolean inclusive,Object alias){
		Class clazz = null;
		try{
			clazz = ReflectionUtil.classForName(className);
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		this.className = clazz;
		this.propertyName = propertyName;
		this.startValue = startValue;
		this.endValue = endValue;
		this.nameSpace = nameSpace;
		this.inclusive = inclusive;
		this.alias = alias;
	}
	
	/**
	 * Construct a query supplying the value to be searched against and the namespace 
	 * under which to start the search. This is a convenience method for classes (such as String,Integer etc)
	 * that have no property value as such. Instead the value is the class type.
	 * <p>
	 * 
	 * An example usage (for JBossCache) would be:
	 * <p>
	 * new MatchNSRangeQuery("/test", "ab","zz",true);
	 * <p>
	 * This is so you do not have to use the methods above in the manner of
	 * <p>
	 * new MatchNSRangeQuery("java.lang.String", "/test", null ,"ab","zz",true); 
	 * <p>
	 * @param nameSpace - the name space to be searched.
	 * @param startValue - the value that is used as the start search value.
	 * @param endValue - the value that is used as the end search value.
	 * @param inclusive - whether the seacrh should be inclusive of values.
	 */
	
	public MatchNSRangeQuery(Object nameSpace,Comparable startValue,Comparable endValue, boolean inclusive){
		this(nameSpace,startValue,endValue,inclusive,null);
	}
	public MatchNSRangeQuery(Object nameSpace,Comparable startValue,Comparable endValue, boolean inclusive,Object alias){
		this.nameSpace = nameSpace;
		this.startValue = startValue;
		this.endValue = endValue;
		this.inclusive = inclusive;
		this.alias=alias;
	}
	/**
	 * @return Returns the className.
	 */
	public Class getClassName() {
		return className;
	}

	/**
	 * @return Returns the propertyName.
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * @return Returns the value.
	 */
	
	
	/**
	 * @return Returns the endValue.
	 */
	public Comparable getEndValue() {
		return endValue;
	}
	
	/**
	 * @return Returns the startValue.
	 */
	public Comparable getStartValue() {
		return startValue;
	}
	
	public synchronized INameSpace getNameSpaceWrapper() {
		
		if (nameSpace instanceof INameSpace){
			return (INameSpace) nameSpace;
		}else{
			return new NameSpaceWrapper(nameSpace);
		}
	}
	/**
	 * @return Returns the nameSpace.
	 */
	public synchronized Object getNameSpace() {
		return nameSpace;
	}
	/**
	 * @param nameSpace The nameSpace to set.
	 */
	public synchronized void setNameSpace(Object nameSpace) {
		this.nameSpace = nameSpace;
	}
	public boolean isInclusive() {
		return inclusive;
	}
	public void setInclusive(boolean inclusive) {
		this.inclusive = inclusive;
	}
	public Object getAlias() {
		return alias;
	}
	
	public QueryType getQueryType() {
		
		return QUERY_ID;
	}
	
	public IndexQuery setParameter(String name, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");
	}
	/* (non-Javadoc)
	 * @see com.jofti.api.IndexQuery#setParameter(int, java.lang.Object)
	 */
	public IndexQuery setParameter(int position, Object value) {
		throw new UnsupportedOperationException("Parameters are not supported for convenience classes");

	}
	public IndexQuery setFirstResult(int firstResult) {
		throw new UnsupportedOperationException("Result limitation is not supported for convenience classes");
	}
	public IndexQuery setMaxResults(int maxResults) {
		throw new UnsupportedOperationException("Result limitation is not supported for convenience classes");
	}
}
