package com.jofti.store;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.btree.BTree;
import com.jofti.btree.EntrySplitWrapper;
import com.jofti.btree.IPage;
import com.jofti.btree.LeafNodeEntry;
import com.jofti.core.IStoreKey;
import com.jofti.core.IStoreManager;
import com.jofti.exception.JoftiException;

import edu.emory.mathcs.backport.java.util.concurrent.atomic.AtomicLong;


/**
 * <p>
 * Base store manager that deals with non-implementation specific actions. The majority of the 
 * functions are to do with storing and retrieving pages to and from disk. 
 * </p>
 * <p>
 * Sub classesshould make use of the bstract methods provided to allow diffirent types of 
 * cache optimisations to make alleviate some of the direct to disk behaviour.
 * </p>
 * @author xenephon
 *
 */
public abstract class AbstractStoreManager implements IStoreManager, IEntrySerializer {



    // values used for initial properties
    final String OVERFLOW_DIRECTORY ="directory";
	final String BLOCK_SIZE = "block-size";
	final String FILE_SIZE = "file-size";
	final String MAX_NODES = "max-nodes";
	final String MAX_FILES ="max-files";
	final String FILE_NAME ="filename";
	
	
	/* objects that manage the reusable objects */
	
	//manages the file access and writing
    FileManager logManager = null;
    
    // buffers used to transfer data to the files
    BufferManager bufferManager =null;
    
    // serialization helpers
    HelperManager helperManager =null;
    
    //manager for pages
    PageManager pageManager =null;
    
    //marker for curent storepointer
    protected int currentStore =0;
	
    // used to issue new keys per node
    final AtomicLong keyId =new AtomicLong(0);
   
    // properties
    protected Properties configProperties =null;
    protected String name =null;
	
	private static Log log =  LogFactory.getLog(AbstractStoreManager.class);
	
     
	int misses=0;
	int hits =0;
	int pendingHits =0;
	int storeRetrieves=0;
	
	int size =0;


	
	int blockSize =4096; //default is 4k
	int fileSize=30485760; // default is 300Mb
	int maxFiles =10; //default is 10
	String overflowDirectory ="/tmp"; //temp directory for operating system
	String fileName ="jofti";
	
	
	int maxBuffers=100; //maximum reusable buffer objects - depends on threads
	int bufferNumber=10; // number of buffers to create initially
	
	 int maxNodes =3000; //default max nodes to keep in memory
	int helperNumber =30; // number of helpers - should be able to cope with maximum number of threads


 

	 
	/* (non-Javadoc)
	 * @see com.jofti.store.IStoreManager#init(java.util.Properties)
	 */
     public synchronized void init(Properties properties) throws JoftiException{


         setUpProperties(properties);

         if (log.isInfoEnabled()){
                log.info("block size is " + blockSize);
                log.info("max file length is " + fileSize);
                log.info("max nodes are " + maxNodes);
                log.info("max file stores is  " + maxFiles);
                log.info("Overflow directory is " + overflowDirectory);
                log.info("filename is " + fileName);

            }
         //set up the log files
         logManager = new FileManager();
         logManager.init(blockSize,overflowDirectory,fileName,fileSize,maxFiles);
         
         //set up the buffers that transfer data to the files
         bufferManager = new BufferManager();
         bufferManager.init(blockSize,maxBuffers,bufferNumber);
                       
         //set up a serialization helper
         helperManager = new HelperManager();
         helperManager.init(helperNumber,blockSize,BTree.getMaxNodeSize(),this);
    
         //set up the page manager
         pageManager = new PageManager();
         pageManager.init(blockSize,100,10,this,this);
         
 
        }
	

   
    /**
     * @param properties
     * @throws JoftiException
     */
    protected void setUpProperties(Properties properties) throws JoftiException
    {
        if (properties != null){
			configProperties = properties;
			String key = null;
			for (Iterator it = properties.keySet().iterator();it.hasNext();){
				key = (String)it.next();
				if (BLOCK_SIZE.equalsIgnoreCase(key)){
					String tempSize =properties.getProperty(key);
					try {
						blockSize =Integer.parseInt(tempSize);
					}catch (Exception e){
						throw new JoftiException("block-size of " + tempSize +" is not valid");
					}					
				}
				
				if (FILE_SIZE.equalsIgnoreCase(key)){
					String tempFile =properties.getProperty(key);
					try {
						fileSize =Integer.parseInt(tempFile);
					}catch (Exception e){
						throw new JoftiException("file lengths of " + tempFile +" is not valid");
					}
				}
				if (MAX_NODES.equalsIgnoreCase(key)){
					String temp =properties.getProperty(key);
					try {
						maxNodes =Integer.parseInt(temp);
					}catch (Exception e){
						throw new JoftiException("max nodes of " + temp +" is not valid");
					}
				}
			
				if (MAX_FILES.equalsIgnoreCase(key)){
					String temp =properties.getProperty(key);
					try {
						maxFiles =Integer.parseInt(temp);
					}catch (Exception e){
						throw new JoftiException("max store number of " + temp +" is not valid");
					}

				}
                 if (OVERFLOW_DIRECTORY.equalsIgnoreCase(key)){

                     overflowDirectory = properties.getProperty(key);

                 }
                 if (FILE_NAME.equalsIgnoreCase(key)){

                    fileName = properties.getProperty(key);



                }
                
			}
		}
    }
	
  
   
 
	/* (non-Javadoc)
	 * @see com.jofti.store.IStoreManager#getName()
	 */
	public String getName() {
		return name;
	}
	/* (non-Javadoc)
	 * @see com.jofti.store.IStoreManager#setName(java.lang.String)
	 */
	public void setName(String name) {
		this.name = name;
	}

	/* (non-Javadoc)
	 * @see com.jofti.core.IStoreManager#getNextKey()
	 */
	public IStoreKey getNextKey(){
		StoreKey key = new StoreKey(keyId.getAndIncrement());
		try {
			key.setFilePositions(new FilePositionHolder[]{logManager.getNextPosition()});
		} catch(JoftiException e){
			throw new RuntimeException(e);
		}
		key.newKey=true;
		return key;
	}

	
        
	/* (non-Javadoc)
	 * @see com.jofti.core.IStoreManager#store(com.jofti.core.IStoreKey, com.jofti.btree.IPage)
	 */
	public  abstract IStoreKey store(IStoreKey key, IPage obj) throws JoftiException;
	   
    /* (non-Javadoc)
     * @see com.jofti.core.IStoreManager#retrieve(com.jofti.core.IStoreKey)
     */
    public  abstract StoreWrapper retrieve(IStoreKey key) throws JoftiException;
  
    
    protected FilePositionHolder[] allocatePositions(FilePositionHolder[] filePositions, int size) throws JoftiException{
        return logManager.allocateBlocks(filePositions,size);
        
    }


    /**
     * <p>
     * Method that does the actual work for storing a byteBuffer to disk. The Byte buffer can be 
     * of any size and the filepositions in the key object are used to determine where the data is written.
     * </p>
     * @param key
     * @param buffer
     * @return
     * @throws JoftiException
     */
    protected IStoreKey doStore(IStoreKey key, ByteBuffer buffer)
            throws JoftiException
    {

        buffer.rewind();
        FilePositionHolder[] positions = key.getFilePositions();

        if (positions.length == 1) {
            BlockBuffer buf = (BlockBuffer) bufferManager
                    .acquireBuffer(positions[0]);
            // write a single buffer
            try {
                buf.writeFurniture(key.getNumber());
                buf.put(buffer);
                FileStore file = logManager.getFileStore(positions[0].file);
                buf.file = file;
                buf.write();
            } catch (IOException ie) {
                throw new JoftiException(ie);
            } finally {
                bufferManager.releaseBuffer(buf);
            }
        } else {
            // we need to loop through the positions and write out the buffers
            // as we have a node that is too big for single buffer
            int offSet = 0;
            int originalLimit = buffer.limit();
            
            for (int i = 0; i < positions.length; i++) {
                BlockBuffer buf = (BlockBuffer) bufferManager
                        .acquireBuffer(positions[i]);
                try {
                    buf.writeFurniture(key.getNumber());

                    // set the pos to be the first offset
                    buffer.position(offSet);
                    // set a temp limit
                    if (originalLimit - offSet > buf.remaining()) {
                        buffer.limit(offSet + buf.remaining());
                        offSet = buffer.limit();
                    } else {
                        buffer.limit(originalLimit);
                        offSet = originalLimit;
                    }

                    buf.put(buffer);

                    buf.file = logManager.getFileStore(positions[i].file);
                    buf.write();
                } catch (IOException ie) {
                    throw new JoftiException(ie);
                } finally {
                    bufferManager.releaseBuffer(buf);
                }

            }

        }

        return key;
    }
    
     /* (non-Javadoc)
     * @see com.jofti.core.IStoreManager#releasePage(com.jofti.core.IStoreKey, com.jofti.btree.IPage)
     */
    public abstract void releasePage(IStoreKey key, IPage page);
   

    /**
     * <p>
     * Used to retrieve a new page. Sub classes should use this method to decorate the 
     * pages in any way they want in order to improve any cache optimisations.
     * </p>
     * @param size
     * @return
     */
    protected abstract IPage getNewPage(int size);  
   
	/**
     * <p>
     * The method that does the work of retrieving the page from the backing store.
     * </p>
	 * @param key
	 * @return
	 * @throws JoftiException
	 */

	protected IPage doRetrieve(IStoreKey key) throws JoftiException
    {

        if (key.isNewKey()) {
            return doGetNewPage(blockSize);
        }

        IPage page = null;

        try {
            FilePositionHolder[] positions = key.getFilePositions();

            if (positions.length == 1) {
                // just get the values straight
                BlockBuffer buf = bufferManager.acquireBuffer(positions[0]);
                ExternalisableHelper helper = helperManager.acquireHelper();
                try {
                    FileStore file = logManager.getFileStore(positions[0].file);
                    buf.read(file);

                    page = helper.readExternalBuffer(buf.buffer, key
                            .getNumber());
                    page.setManager(this);
                    return page;
                } finally {
                    bufferManager.releaseBuffer(buf);
                    helperManager.releaseHelper(helper);
                }
            } else {
                // merge all the buffers into a single buffer
                ByteBuffer totalBuf = pageManager
                        .acquireBuffer(positions.length * blockSize);
                totalBuf.clear();
                ExternalisableHelper helper = helperManager.acquireHelper();
                // Object[] obj =null;
                int number = 0;
                try {
                    for (int i = 0; i < positions.length; i++) {
                        BlockBuffer buf = bufferManager
                                .acquireBuffer(positions[i]);

                        try {
                            FileStore file = logManager
                                    .getFileStore(positions[i].file);
                            buf.read(file);
                            totalBuf.put(buf.buffer);
                            if (i == 0) {
                                number = buf.numberEntries;
                            }
                        } finally {
                            bufferManager.releaseBuffer(buf);
                        }
                    }
                    //				
                    totalBuf.flip();
                    try {
                        page = helper.readExternalBuffer(totalBuf, number);
                    } catch (Throwable t) {
                        log.fatal("buffer read failure ", t);
                        throw t;
                    }
                    page.setManager(this);
                } finally {
                    helperManager.releaseHelper(helper);
                    pageManager.releaseBuffer(totalBuf);
                }		
                return page;
            }
        } catch (Throwable t) {
            log.fatal("error retrieving record ", t);
            throw new JoftiException("error retrieving record ", t);
        }

    }

    /* (non-Javadoc)
     * @see com.jofti.core.IStoreManager#remove(com.jofti.core.IStoreKey, com.jofti.btree.IPage)
     */
    public abstract void remove(IStoreKey key, IPage page) throws JoftiException ;
	
    
    /* (non-Javadoc)
     * @see com.jofti.core.IStoreManager#removeAll()
     */
    public abstract void  removeAll() throws JoftiException;
    
    
    protected void doRemoveAll() throws JoftiException{
        
        logManager.reset();
    }
    
	protected void doRemove(IStoreKey key) throws JoftiException {
		
			FilePositionHolder[] holder = key.getFilePositions();
		
			if (holder != null){
				for (int i=0;i<holder.length;i++){
					logManager.removePosition(holder[i]);
				}
				
			}

		
	}
	
	/*  Serializer methods */

	
    /* (non-Javadoc)
     * @see com.jofti.store.IEntrySerializer#convertForStorage(java.lang.Object)
     */
    public byte[] convertForStorage(Object obj) throws JoftiException{
    	    ExternalisableHelper helper =helperManager.acquireHelper();
    	    try {
    	        return helper.convertForStore((LeafNodeEntry)obj);
    	    }finally {
    	        helperManager.releaseHelper(helper);
    	    }
    }
	
    
    
 
    /* (non-Javadoc)
     * @see com.jofti.store.IEntrySerializer#convertFromStorage(java.nio.ByteBuffer)
     */
    public LeafNodeEntry convertFromStorage(ByteBuffer buf) throws JoftiException{
    	    ExternalisableHelper helper =helperManager.acquireHelper();
    	    try {
    	        return (LeafNodeEntry) helper.convertFromStore(buf);
    	    }finally {
    	        helperManager.releaseHelper(helper);
    	    }
    }
    
    /**
     * @param size
     * @return
     */
    protected IPage doGetNewPage(int size)
    {

        IPage page = null;

        try {
            page = pageManager.acquirePage(size);
        } catch (Exception e) {
            throw new RuntimeException("unable to acquire page for size "
                    + size);
        }
        return page;

    }
    
  
    /* (non-Javadoc)
     * @see com.jofti.core.IStoreManager#split(com.jofti.btree.IPage, int)
     */
    public EntrySplitWrapper[] split(IPage page, int entryNumber){
		

		int splitPoint = entryNumber/2;
			
		//set up the new buffer
		ByteBuffer buf = page.getBuffer();
		int[] pointers =page.getPointers();
		// see if we need more than block size
		int endPoint = pointers[entryNumber-1];

		 buf.position(endPoint);
		int endLength =  buf.getInt()+4;
		
		int bufferSize =(endLength+endPoint) - pointers[splitPoint];
		
		IPage newPage =getNewPage(bufferSize);
		

	
		newPage.getBuffer().limit(bufferSize);
		newPage.getBuffer().rewind();
		
		int length = entryNumber -splitPoint;
		System.arraycopy(pointers,splitPoint,newPage.getPointers(),0,length);
		
		
		// write the buffer
		buf.position(newPage.getPointers()[0]);
		
		buf.mark();
		try {
			newPage.getBuffer().put(buf);
		} catch (Throwable t){
			log.fatal("error "+ newPage.getBuffer() + " "+ buf);
			throw new RuntimeException(t);
		}
		newPage.getBuffer().flip();

		buf.reset();
		buf.flip();
		

		
		
//		erase the higher entries
		Arrays.fill(pointers,splitPoint,pointers.length-1,-1);
	
		// ok nor reset the higher fields 
		int offSet = newPage.getPointers()[0];
		
		for (int i=0;i<length;i++){
			newPage.getPointers()[i]= newPage.getPointers()[i]-offSet;
		}
		
		EntrySplitWrapper[] entries =new EntrySplitWrapper[2];
		EntrySplitWrapper splitWrapper = new EntrySplitWrapper();
		splitWrapper.entries =page;
		splitWrapper.size=length;
		entries[0] = splitWrapper;
		
		
		EntrySplitWrapper newWrapper = new EntrySplitWrapper();
		newWrapper.entries =newPage;
		newWrapper.size=splitPoint;
		
		entries[1]=newWrapper;
		
		return entries;
		
	}
  


}
