package com.jofti.store;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.exception.JoftiException;

/**
 *
 * <p>Each block contains a header, zero or more entries,
 * and a footer.
 */
class BlockBuffer {

	ByteBuffer buffer = null;

	FileStore file = null;

	// default size
	int bufferSize = 4096;

	int recordSize = 0;

	int numberEntries=0;
	/**
	 * Number of used data bytes in the buffer.
	 * 
	 * <p>This is different than buffer capacity().  The bytes used is the
	 * number of bytes of usable data in a buffer. 
	 * 
	 * <p>set by operations that read data from files into the
	 * buffer such as read().
	 * <p>checked by operations that retrieve logical records
	 * from the buffer get().
	 */
	int bytesUsed = 0;

	FilePositionHolder positionHolder = null;

	private static Log log = LogFactory.getLog(BlockBuffer.class);

	/**
	 * number of times this buffer was used.
	 * <p>In general, should be about the same for all buffers in a pool.
	 */
	int initCounter = 0;

	/**
	 * size of a buffer header.
	 * <pre>
	 * <b>buffer Header format</b>
	 * byte[]  HEADER_ID              [1] "H"
	 * short   size of bytes written  [2]
	 * short   total number of items  [2]
	 * byte[]  CRLF                   [2] to make it easier to read buffers in an editor
	 * </pre>
	 */
	final static int bufferHeaderSize = 7;

	final int bytesUsedOffSet = 1;

	/**
	 * The number of bytes to reserve for block footer information.
	 */
	final static int bufferFooterSize = 3;

	/* 
	 * byte FOOTER_ID             	[1] "F"
	 * byte CRLF                  	[2] to make it easier to read buffers in an editor
	 */

	/**
	 * Size of the header for each data record in the block.
	 * 
	 * <p>Record header format:
	 * int record length of user data [4]
	 */
	private int recordHeaderSize = 4;

	// block header & footer fields
	/**
	 * Carriage Return Line Feed sequence used for debugging purposes.
	 */
	private byte[] CRLF = "\r\n".getBytes();

	private byte[] crlf = new byte[CRLF.length];

	/**
	 * Signature for each logical block header.
	 */
	private byte[] HEADER_ID = "H".getBytes();

	/**
	 * Signature for each logical block footer.
	 */
	private byte[] FOOTER_ID = "F".getBytes();

	/**
	 * maximum size of user data record.
	 * 
	 * <p>Although this member is local to a LogBuffer instance,
	 * it is assumed to have the same value in all instances.
	 */
	private int payloadSize;

	/**
	 * default constructor calls super class constructor.
	 */
	BlockBuffer() {

	}

	public int remaining() {
		return buffer.remaining() - (bufferFooterSize);
	}

	/**
	 * initialize members for buffer reuse.
	 * 
	 * @param bsn Logic Block Sequence Number of the buffer.
	 * BufferManager maintains a list of block sequence numbers
	 * to ensure correct order of writes to disk.  Some implementations
	 * of LogBuffer may include the BSN as part of a record or
	 * block header.
	 */
	public void init(int bufferSize) throws JoftiException {
		// sun Bug ID: 4879883 - can't allocate more than 64mb without resetting
		// in jdk earlier than  1.4.2_05 - workaround -XX:MaxDirectMemorySize=XXXM

		buffer = ByteBuffer.allocateDirect(bufferSize);

		++initCounter;

		buffer.clear();

		payloadSize = ((bufferSize) - (bufferHeaderSize + bufferFooterSize));

	}

	public void reset() {
		buffer.clear();
		positionHolder = null;
		file = null;
		recordSize = 0;
		bytesUsed = 0;

	}

	void writeFurniture(int numberRecords) {
		buffer.put(HEADER_ID);

		// placeholder for length of bytes in record
		buffer.putShort((short) 0);
		buffer.putShort((short)numberRecords);
		buffer.put(CRLF);

	}


	int put(ByteBuffer buf, int offset) throws JoftiException {

		int written = 0;

		int remaining = buffer.remaining() - bufferFooterSize;

		// write the length of the array element
		// ensure that it is first element of array - or we do not write this
		if (offset == 0 && recordHeaderSize <= remaining) {
		//	buffer.putInt(data.length);
			remaining -= 4;
		} else if (offset == 0 && recordHeaderSize > remaining) {
			// we can ignore this as there is not enough space
			// to write even the header 
			return written;
		}
		// we do this to make sure that we always have the size in front of the array

		// try and write some bytes from the data - this could be partial for a large array
		if (remaining > 0) {

			int length = remaining;
			// can we fit all the data element into the buffer
//			if ((data.length - offset) < length) {
//				length = data.length - offset;
//			}
//			buffer.put(data, offset, length);
			// update how much we have written
			written = length;
			// update what we have remaining

		}

		bytesUsed = buffer.position();
		return written;
	}
	
	void put(ByteBuffer buf) throws JoftiException {

		// write the length of the array element
		// ensure that it is first element of array - or we do not write this
			buffer.put(buf);
			// update how much we have written

		bytesUsed = buffer.position();
	}



	

	void writeFooter() {
		if (buffer.position() != payloadSize + bufferHeaderSize) {
			buffer.position(payloadSize + bufferHeaderSize);
		}

		buffer.put(FOOTER_ID);
		buffer.put(CRLF);

	}

	/**
	 * write ByteBuffer to the log file.
	 */
	void write() throws IOException {
		// assert lf != null: "FileStore lf is null";

		//write bytesUsed
		buffer.putShort(bytesUsedOffSet, (short) bytesUsed);

		writeFooter();
		try {

			buffer.flip();
			file.write(this);

		} catch (IOException e) {
			throw e;
		}
	}

	/**
	 * Reads a block from FileStore <i> lf </i> and validates
	 * header and footer information.
	 * 
	 * @see LogBuffer#read(FileStore, long)
	 * @throws IOException
	 * if anything goes wrong during the file read.
	 * @throws InvalidLogBufferException
	 * if any of the block header or footer fields are invalid.
	 */
	BlockBuffer read(FileStore store) throws JoftiException {
		long position = positionHolder.position;

		try {
			// fill our ByteBuffer with a block of data from the file
			int bytesRead = -1;
			if (store.channel.size() > position) //  JRockit issue identified in HOWL
				bytesRead = store.channel.read(buffer, position);
			if (bytesRead == -1) {
				// end of file
				return null;
			}

			if (bytesRead != buffer.capacity())
				throw new JoftiException("FILESIZE Error: bytesRead="
						+ bytesRead);

			// verify header
			buffer.clear();
			byte head = buffer.get();
			if (head != HEADER_ID[0])
				throw new JoftiException(
						"HEADER_ID does not match for position " + position
								+ " in store " + store.fileId);

			// get data used (short)
			bytesUsed = buffer.getShort();
			if (bytesUsed < 0)
				throw new JoftiException("data used: " + bytesUsed
						+ "  for position " + position + " in store "
						+ store.fileId);
			// get the number of records
			numberEntries = buffer.getShort();
			if (numberEntries < 0)
				throw new JoftiException("entries: " + numberEntries
						+ "  for position " + position + " in store "
						+ store.fileId);
			
			// get CRLF
			buffer.get(crlf);
			if (!Arrays.equals(crlf, CRLF))
				throw new JoftiException(
						"HEADER_CRLF not present  for position " + position
								+ " in store " + store.fileId);

			// mark start of first data record
			buffer.mark();

			// get FOOTER_ID and compare 
			buffer.position(payloadSize + bufferHeaderSize);

			byte footer = buffer.get();
			if (footer != FOOTER_ID[0]) {
				throw new JoftiException("FOOTER_ID not present for position "
						+ position + " in store " + store.fileId);
			}

			// get FOOTER_CRLF
			buffer.get(crlf);
			if (!Arrays.equals(crlf, CRLF))
				throw new JoftiException("FOOTER_CRLF not found for position "
						+ position + " in store " + store.fileId);

			//set the limit
			buffer.limit(bytesUsed);
			// reset position to first data record

			buffer.reset();
		} catch (Exception e) {
			throw new JoftiException(e);
		}
		return this;
	}

	/**
	 * return statistics for this buffer.
	 * 
	 * @return String containing statistics as an XML node
	 */
	String getStats() {
		String name = this.getClass().getName();

		String result = "<LogBuffer class='"
				+ name
				+ "\n  <timesUsed value='"
				+ initCounter
				+ "'>Number of times this buffer was initialized for use</timesUsed>"
				+ "\n</LogBuffer>" + "\n";

		return result;
	}

	public FileStore getFileStore() {
		return file;
	}

	public void setFileStore(FileStore file) {
		this.file = file;
	}

	public FilePositionHolder getPositionHolder() {
		return positionHolder;
	}

	public void setPositionHolder(FilePositionHolder positionHolder) {
		this.positionHolder = positionHolder;
	}
}