package com.jofti.store;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.btree.IPage;
import com.jofti.btree.KeyValueObject;
import com.jofti.btree.LeafNodeEntry;
import com.jofti.btree.MaxLeafNodeEntry;
import com.jofti.btree.ValueObject;
import com.jofti.core.IStoreManager;
import com.jofti.exception.JoftiException;
import com.jofti.util.ByteBufferArrayInputStream;
import com.jofti.util.FastByteArrayOutputStream;

/**
 * 
 * Helper to aid in the serialization and deserialization of Stored Leaf Nodes.
 * <p>
 * @author xenephon (xenephon@jofti.com)
 *
 */
public class ExternalisableHelper {

	private static final byte STRING = 0;

	private static final byte LONG = 1;

	private static final byte INTEGER = 2;

	private static final byte BOOLEAN = 3;

	private static final byte OTHER = 4;

	private static final byte MAX_ENTRY = 0;

	private static final byte LEAF_ENTRY = 1;

	private static final byte NULL_ENTRY = 2;

	private static final byte KEY_VALUE = 1;

	private static final byte VALUE_OBJECT = 2;
	
	private static Log log =  LogFactory.getLog(ExternalisableHelper.class);

	FastByteArrayOutputStream nodeFbos = null;

	ByteBufferArrayInputStream bybuf = null;

	ObjectOutputStream nodeOut = null;

	AbstractStoreManager manager = null;

	int nodeSize = 0;

	public void init(int nodeSize, int blockSize, IStoreManager manager)
			throws JoftiException 
	{
		bybuf = new ByteBufferArrayInputStream(null);
		nodeFbos = new FastByteArrayOutputStream(blockSize);
		try {
			nodeOut = new ObjectOutputStream(nodeFbos);

			nodeOut.useProtocolVersion(1);
		} catch (Exception e) {
			throw new JoftiException(e);
		}
		this.nodeSize = nodeSize;
		this.manager = (AbstractStoreManager) manager;
	}


	public LeafNodeEntry convertFromStore(ByteBuffer buf)
			throws JoftiException 
	{

		ObjectInputStream in = null;
		LeafNodeEntry entry = null;

		try {

			bybuf.resetData(buf);
			// unfortunately we have to create a new stream here
			in = new ObjectInputStream(bybuf);

			// get the type of entry
			byte type = in.readByte();

			if (type == NULL_ENTRY) {
				//we can leave here
				return null;
			} else if (type == MAX_ENTRY) {

				entry = new MaxLeafNodeEntry();
			} else {
				if (entry == null) {
					entry = new LeafNodeEntry();
				}
			}

			// get the value object
			Object obj = readValue(in);

			// get value object type
			type = in.readByte();

			//get the dimension
			int dimension = in.readInt();

			if (type == KEY_VALUE) {

				// get the number of entries in hashmap
				// we assume that there are no more than 256 dimensions for a 
				// key value
				byte j = in.readByte();

				Map map = new HashMap(j);
				for (int k = 0; k < j; k++) {
					map.put(readValue(in), readValue(in));
				}
				entry.value = new KeyValueObject(dimension, (Comparable) obj,
						map);

			} else {

				entry.value = new ValueObject(dimension, (Comparable) obj);

			}
			// get number of entries in set - mostly will be few
			int l = in.readInt();

			for (int k = 0; k < l; k++) {
				entry.addUniqueId(readValue(in));
			}

		} catch (Exception e) {
			throw new JoftiException("Unable to read entry " + buf, e);
		} finally {
			// make sure we close the input stream
			if (in != null) {
				try {
					in.close();
				} catch (Exception e) {
					// put a warning here
				}
				in = null;
			}
		}

		return entry;

	}

	public byte[] convertForStore(LeafNodeEntry entry) throws JoftiException 
	{
		byte[] res =null;
		try {
		if (entry instanceof MaxLeafNodeEntry) {
			// maxLeafNodeEntry
			nodeOut.writeByte(MAX_ENTRY);

		} else {
			//LeafNodeEntry
			nodeOut.writeByte(LEAF_ENTRY);
		}
		// write out the value

		ValueObject valObj = (ValueObject) entry.value;

		writeValue(nodeOut, valObj.getRealValue());

		//write out the key value
		if (entry.value instanceof KeyValueObject) {
			//type for ke yvalue object
			nodeOut.writeByte(KEY_VALUE);

			//write the dimension
			nodeOut.writeInt(((ValueObject) entry.value).getDimension());

			// write out the attributes 
			Map map = ((KeyValueObject) valObj).getAttributes();

			// write out the size of the key value attributes
			nodeOut.writeByte((byte) map.size());
			Iterator it = map.entrySet().iterator();
			for (int i = 0; i < map.size(); i++) {
				Map.Entry mapEntry = (Map.Entry) it.next();
				writeValue(nodeOut, mapEntry.getKey());
				writeValue(nodeOut, mapEntry.getValue());

			}

		} else {
			//write out type of value object
			nodeOut.writeByte(VALUE_OBJECT);

			//write out dimension
			nodeOut.writeInt(((ValueObject) entry.value).getDimension());
		}

		int size = entry.getUniqueIdSize();

		nodeOut.writeInt(size);
		if (size > 0) {
			Iterator it = entry.uniqueIdSet.iterator();
			for (int i = 0; i < size; i++) {
				writeValue(nodeOut, it.next());
			}
		}

		nodeOut.flush();
		size = nodeFbos.getSize();
		byte[] tempData = nodeFbos.getByteArray();
		res = new byte[size];

		System.arraycopy(tempData, 0, res, 0, size);

		nodeOut.reset();
		nodeFbos.reset();

		} catch (Throwable t){
			throw new JoftiException("Unable to serialize entry "+ entry,t);
		}
		return res;

	}


	IPage readExternalBuffer(java.nio.ByteBuffer buffer, int numberEntries)
			throws JoftiException
	{

		IPage page = manager.doGetNewPage(buffer.limit());

		ByteBuffer pBuf = page.getBuffer();

		int[] pointers = page.getPointers();
		
		try {
			pBuf.clear();
			pBuf.put(buffer);
			pBuf.flip();
			pBuf.mark();

			// lets loop through the entries
			for (int i = 0; i < numberEntries; i++) {

				// first get the size of each entry
				pointers[i] = pBuf.position();
				int size = pBuf.getInt();
				// set the position of the entry
				pBuf.position(pBuf.position() + size);

			}
			pBuf.reset();

		} catch (Throwable e) {
			log.fatal("expected to read " + numberEntries + " pos at "
					+ buffer);
			throw new JoftiException("unable to read block ", e);
		}
		return page;

	}

	
	private void writeValue(ObjectOutputStream out, Object obj)
			throws IOException {
		Class clazz = obj.getClass();
		if (clazz == String.class) {
			out.writeByte(STRING);

			out.writeUTF((String) obj);

		} else if (clazz == Integer.class) {
			out.writeByte(INTEGER);
			out.writeInt(((Integer) obj).intValue());

		} else if (clazz == Long.class) {
			out.writeByte(LONG);
			out.writeLong(((Long) obj).longValue());
		} else {
			out.writeByte(OTHER);
			out.writeObject(obj);
		}
	}

	private Object readValue(ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		int valType = in.readByte();
		switch (valType) {
		case STRING:
			return in.readUTF();
		case INTEGER:
			return new Integer(in.readInt());

		case LONG:
			return new Long(in.readLong());
		default:
			return in.readObject();
		}

	}
}