package com.jofti.store;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogConfigurationException;
import org.apache.commons.logging.LogFactory;


import com.jofti.exception.InsufficientSpaceException;
import com.jofti.exception.JoftiException;

import edu.emory.mathcs.backport.java.util.concurrent.LinkedBlockingQueue;

/**
 * 
 * Manages files set up in Jofti. This manager sets up the files, allocates space in the files and 
 * keeps track of the current poaitions in each file.
 * <p>
 * This is based on some of the ideas found in HOWL.
 * </p>
 * 
 * @author xenephon
 *
 */
class FileManager {

	private static Log log = LogFactory.getLog(FileManager.class);

	// lock controlling access to block allocation
	private Object recordLock = new Object();


	/**
	 * set of FileStore objects associated with the physical files.
	 * 
	 * @see #open()
	 */
	FileStore[] fileSet = null;

	/**
	 * array of unalloctaed blocks fro each store.
	 * 
	 * @see #open()
	 */
	long[] unallocatedBlocks = null;

	/**
	 * array of positions for each store.
	 * 
	 */
	long[] filePositions = null;

	/**
	 * current store
	 * 
	 */	
	FileStore currentStore = null;

	// where the old blocks are put - should just be a queue of FilePositionHolders
	protected LinkedBlockingQueue removedBlockQueue = new LinkedBlockingQueue(); //this is unbounded

	//default max files
	private int maxFiles = 10;

	// max size of each file
	private long maxSize = 0;

	/**
	 * keys into properties
	 * 
	 */	
	protected String OVERFLOW_DIRECTORY = "overflow-directory";

	protected String fileName = "jofti";

	protected String extension = "str";

	protected String logFileDirectory = "../tmp";

	private Properties props = null;

	// block size to be used
	int blockSize = 0;

	// block size- (header size + footer size)
	int payloadSize = 0;


	FileManager() {

	}

	void init(int blockSize, String directory, String fileName, long size,
			int maxFiles) throws JoftiException 
	{

		//set up the properties we need

		this.blockSize = blockSize;
		this.logFileDirectory = directory;
		this.fileName = fileName;
		this.maxSize = size;
		this.maxFiles = maxFiles;

		// open all the files
		try {
			open();
		} catch (Exception e) {
			throw new JoftiException(e);
		}
		
		// set up the initial free blocks for each file
		unallocatedBlocks = new long[fileSet.length];
		filePositions = new long[fileSet.length];

		long initialBlocks = maxSize / blockSize;
		
		// set up the unallocated blocks
		for (int i = 0; i < fileSet.length; i++) {
			unallocatedBlocks[i] = initialBlocks;
		}

		//set up the payload size
		payloadSize = blockSize
				- (BlockBuffer.bufferHeaderSize + BlockBuffer.bufferFooterSize);

	}

	
	public synchronized void reset(){
//		 set up the initial free blocks for each file
		unallocatedBlocks = new long[fileSet.length];
		filePositions = new long[fileSet.length];

		long initialBlocks = maxSize / blockSize;
		
		// set up the unallocated blocks
		for (int i = 0; i < fileSet.length; i++) {
			unallocatedBlocks[i] = initialBlocks;
		}
		removedBlockQueue.clear();
	}
	/**
	 * Allocates an array of FilePositionHolder objects which encompasses the locations 
	 * of the page data across one or more files.
	 * <p>
	 * @param blocks - existing file positions (if any)
	 * @param size - total number of bytes to store
	 * @return
	 * @throws JoftiException
	 */
	public FilePositionHolder[] allocateBlocks(FilePositionHolder[] blocks,
			int size) throws JoftiException 
	{

		FilePositionHolder[] returnBlocks = null;

		// empty storage
		if (size == 0) {
			return blocks;
		}
		
		// see how many keys we need
		int keysNeeded = size / payloadSize;
		int remainder = size % payloadSize;

		// is there a bit left over?
		if (remainder != 0) {
			keysNeeded++;
		}
		
		if (log.isDebugEnabled()) {
			log.debug("payloadSize " + payloadSize + " number keys needed "
					+ keysNeeded + " for data size:" + size);
		}

		// we now know the number of keys needed - but we do not know what original ones we have?

		try {
			returnBlocks = allocateRecords(blocks, keysNeeded);
		} catch (Exception e) {
			throw new JoftiException(e);
		}

		return returnBlocks;

	}

	private FilePositionHolder[] allocateRecords(FilePositionHolder[] blocks,
			int number) throws InsufficientSpaceException, Exception 
	{

		int originalRecordNumber = 0;
		FilePositionHolder[] temp = null;

		if (blocks != null) {
			originalRecordNumber += blocks.length;
		}
		
		if (number == originalRecordNumber) {
			// return the same blocks
			temp = blocks;
		} else if (number < originalRecordNumber) {
			// otherwise we need to free some up
			
			temp = new FilePositionHolder[number];

			//free up blocks we do not need
			for (int i = 0; i < blocks.length; i++) {
				if (i < number) {
					temp[i] = blocks[i];
				} else {
					removedBlockQueue.put(blocks[i]);
				}

			}
			return temp;
		}

		// we need more than we have
		else if (number > originalRecordNumber) {

			// we need to add some more records here
			temp = new FilePositionHolder[number];
			// copy the existing records into the new array

			System.arraycopy(blocks, 0, temp, 0, blocks.length);

			for (int i = blocks.length; i < temp.length; i++) {
				// try and get from the freeQueue first

				temp[i] = getNextPosition();
			}

		}
		return temp;

	}

	public void removePosition(FilePositionHolder holder) throws JoftiException {
			removedBlockQueue.offer(holder);
	}

	public FilePositionHolder getNextPosition() throws JoftiException {
		FileStore file = null;

		//see if we can get one from the free queue first
		FilePositionHolder fileHolder = null;


		fileHolder = (FilePositionHolder) removedBlockQueue.poll();


		if (fileHolder != null) {
			return fileHolder;
		}

	
		file = currentStore;

		// store we started at
		int marker = file.fileId;
		long position = 0;

		do {
			synchronized (recordLock) {
				if (unallocatedBlocks[file.fileId] > 0) {
					position = filePositions[file.fileId];
					fileHolder = new FilePositionHolder(file.fileId, position);
					filePositions[file.fileId] = position + blockSize;
					unallocatedBlocks[file.fileId]--;
					
					//we rotate the store so we get even spread of entries in all stores
					if (file.fileId == unallocatedBlocks.length - 1) {
						currentStore = getFileStore(0);
					} else {
						currentStore = getFileStore(file.fileId + 1);
					}
					return fileHolder;
				} else {

					// ok we can't get as key here - try the next one
					file = getFileStore(file.fileId + 1);

					if (file != null) {
						currentStore = file;
						if (log.isInfoEnabled()) {
							log.info("Storing in:" + file.fileId
									+ " moving to next store");
						}
					}
				}
			}
			// loop around until we have run out of stores to try
		} while (file != null && file.fileId != marker);
		// we have dropped through with no file available

		throw new JoftiException("No space available in any file");
	}


	FileStore getFileStore(int id) {
		FileStore lf = null;
		int fsl = fileSet.length;

		if (id >= fsl) {
			return null;
		} else {
			return fileSet[id];
		}

	}

	
	/**
	 * open pool of FileStore(s).
	 * 
	 * @throws FileNotFoundException
	 * @throws LogConfigurationException
	 * @throws IOException
	 * @throws InvalidFileSetException
	 */
	void open() throws JoftiException, IOException, FileNotFoundException,
			JoftiException 
	{

		// get configuration information for log file names.
		String logDir = logFileDirectory;
		String logFileName = fileName;
		String logFileExt = extension;

		// make sure the directory exists
		File dir = new File(logDir);
		dir.mkdirs();

		int existingFiles = 0;

		// allocate the set of log files
		fileSet = new FileStore[maxFiles];
		for (int i = 0; i < maxFiles; ++i) {
			File name = new File(logDir + "/" + logFileName + "_" + (i + 1)
					+ "." + logFileExt);
			if (name.exists()) {
				name.delete();
				name.createNewFile();
			}

			try {
				fileSet[i] = new FileStore(name, i).open("rw");

			} catch (FileNotFoundException e) {
				while (--i >= 0) {
					fileSet[i].close();
					fileSet[i] = null;
				}

				throw e;
			}

		}
		currentStore = fileSet[0];

	}

	/**
	 * close the log files.
	 * 
	 * @throws IOException
	 * If FileChannel.close() encounters an error.
	 * @see java.nio.channels.FileChannel#close()
	 */
	void close() throws IOException, InterruptedException {

		if (fileSet == null)
			return;

		// close the log files
		for (int i = 0; i < fileSet.length; ++i) {
			if (fileSet[i] != null)
				fileSet[i].close();
		}

	}

}