
package com.jofti.store;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.exception.JoftiException;


class FileStore
{
  File file = null;
  
  /**
   * @see java.io.RandomAccessFile#RandomAccessFile(java.lang.String, java.lang.String)
   */
  String fileMode = "rw";
  
  /**
   * FileChannel associated with this FileStore.
   * 
   * <p>The FileChannel is private to guarantee that all calls to the
   * channel methods come through this FileStore.  
   */
  FileChannel channel = null;
  

  protected int fileId =0;
  
  
  private static Log log =  LogFactory.getLog(FileManager.class);
  
  
  /**
   * FileChannel.position() of last read or write.
   * 
   * <p>May be used to report the file position when IOException occurs. 
   */
  long nextPosition = 0;
  
  /**
   * indicates the file was created during the call to open()
   * @see #open(String filemode)
   */
  boolean newFile = true;
  
  /**
   * FileLock acquired when file is opened.
   */
  
  FileLock lock = null;
  
  Object recordLock = new Object();
  

  long writes =0;


  
  /**
   * construct an instance of FileStore for a given file name
   * @param file filename
   */
  FileStore(File file, int id)
  {
    this.file = file;
    this.fileId =id;
  }
  
  
 
	
  /**
   * open the file and get the associated nio FileChannel for the file.
   * 
   * <p>
   * If the file does not exist, then the newFile member is set true.
   * 
   * @param fileMode
   *            value passed to RandomAccessFile constructor.
   * @throws FileNotFoundException
   *             if the parent directory structure does not exist.
   * @see java.io.RandomAccessFile#RandomAccessFile(java.lang.String,
   *      java.lang.String)
   */
  FileStore open(String fileMode) throws JoftiException, FileNotFoundException
  {
    this.fileMode = fileMode;

    // remember whether the file existed or not
    newFile = !file.exists();
    
    // if it already existed, but length is zero, then it is still a new file
    if (!newFile) newFile = file.length() == 0;
    
    channel = new RandomAccessFile(file, fileMode).getChannel();
   
    try {
      lock = channel.tryLock();
    } catch (IOException e) {
      throw new JoftiException(e);
    }
    if (lock == null)
     log.info("Unable to obtain lock on " + file.getAbsolutePath());
    
    return this;
  }
  
  /**
   * Close the channel associated with this FileStore.
   * <p>Also releases the lock that is held on the file.
   * @return this FileStore
   * @throws IOException
   */
  FileStore close() throws IOException
  {
    if (channel.isOpen())
    {
      if (lock != null)
      {
        lock.release();
      }
      channel.close();
    }
    return this;
  }
  
  /**
   * Helper provides access to the FileChannel.write() method for
   * the FileChannel associated with this FileStore.
   * @param lb Reference to a LogBuffer object that is to be written.
   * @throws IOException
   */
  void write(BlockBuffer lb) throws IOException
  {
  	
  	channel.write(lb.buffer,lb.positionHolder.position);
  	
  	//  force 
  	if (writes %10000 == 0){
  		force(true);
  	}
  	writes++;
  }
  
  /**
   * 
   * @param forceMetadata as defined by FileChannel.force()
   * @throws IOException
   * @see FileChannel#force(boolean)
   */
  void force(boolean forceMetadata) throws IOException
  {
    channel.force(forceMetadata);
  }
  
  
}
