package com.jofti.store;

import java.nio.ByteBuffer;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.btree.BTree;
import com.jofti.btree.IPage;
import com.jofti.btree.LeafNodeEntry;
import com.jofti.core.IStoreKey;
import com.jofti.exception.JoftiException;
import com.jofti.util.PrimeFinder;

public class HashedStoreManager extends AbstractStoreManager
{

    private static Log       log    = LogFactory
                                            .getLog(AbstractStoreManager.class);

    protected StoreWrapper[] nodes  = null;

    protected Object[]       locks  = null;

    long                     stores = 0;

    public void init(Properties properties) throws JoftiException
    {
        if (log.isInfoEnabled()) {
            log.info("Initialising Hashed store manager");
        }
        super.init(properties);
        maxNodes = PrimeFinder.nextPrime(maxNodes);
        

        nodes = new StoreWrapper[maxNodes];
        locks = new Object[maxNodes];
        for (int i = maxNodes - 1; i >= 0; i--) {
            locks[i] = new Object();
        }
        if (log.isInfoEnabled()) {
            log.info("adjusted maxNodes to " + maxNodes);
            log.info("Initialised Hashed store manager");
        }
    }

    public void removeAll() throws JoftiException
    {
        for (int i = 0; i < nodes.length; i++) {
            StoreWrapper wrap = nodes[i];
            if (wrap != null) {
                CachedPage old = (CachedPage) wrap.page;
                old.releaseReference();
                if (!old.hasReferences()) {
                    pageManager.releasePage(old.page);
                }
                nodes[i] = null;
            }
        }
        doRemoveAll();
    }

    protected IPage getNewPage(int size)
    {
        IPage temp = doGetNewPage(size);

        // wrap the page
        CachedPage cPage = new CachedPage(temp);
        cPage.entries = new LeafNodeEntry[BTree.getMaxNodeSize()];
        // add a usage reference here
        cPage.acquireReference();
        return cPage;
    }

    public IStoreKey store(IStoreKey key, IPage page) throws JoftiException
    {
        StoreWrapper wrapper = null;

        // get the number of positions we need to store the data
        int limit = page.getBuffer().limit();
        FilePositionHolder[] array = allocatePositions(key.getFilePositions(),
                limit);
        key.setFilePositions(array);

        // make sure we are storing a copy here
        ByteBuffer buf = pageManager.acquireBuffer(limit);

        doStore(key, page.copyBuffer(buf));
        // release the buffer we have just stored
        pageManager.releaseBuffer(buf);

        int i = (int) (key.getId() % maxNodes);

        // add to cached pages
        synchronized (locks[i]) {
            // release our reference
            ((CachedPage) page).releaseReference();
            // get the old ref
            wrapper = nodes[i];
            // if previous is null
            if (wrapper == null) {
                StoreWrapper newWrapper = new StoreWrapper(key, page);
                nodes[i] = newWrapper;
                // we are replacing the entry here
            } else if (!wrapper.key.equals(key)) {

                CachedPage old = (CachedPage) wrapper.page;
                if (!old.hasReferences()) {
                    pageManager.releasePage(old.page);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("unable to release ref on write " + old);
                    }
                }
                StoreWrapper newWrapper = new StoreWrapper(key, page);
                nodes[i] = newWrapper;
            }
            // else the page should be attached to the wrapper anyway

        }

        // we do not care
        return null;
    }

    public void releasePage(IStoreKey key, IPage page)
    {

        int i = (int) (key.getId() % maxNodes);

        synchronized (locks[i]) {
            ((CachedPage) page).releaseReference();
        }
    }

    public StoreWrapper retrieve(IStoreKey key) throws JoftiException
    {

        StoreWrapper wrapper = null;

        int i = (int) (key.getId() % maxNodes);

        synchronized (locks[i]) {
            // get the previous entry
            wrapper = nodes[i];
            if (wrapper != null && wrapper.key.equals(key)) {
                ((CachedPage) wrapper.page).acquireReference();
                return wrapper;
            }
        }

        IPage page = doRetrieve(key);
        // create a new cached node
        CachedPage cPage = new CachedPage(page);
        cPage.entries = new LeafNodeEntry[BTree.getMaxNodeSize()];
        cPage.acquireReference();
        wrapper = new StoreWrapper(key, cPage);

        // otherwise replace the val

        synchronized (locks[i]) {
            StoreWrapper old = nodes[i];

            if (old != null && !old.key.equals(key)) {
                CachedPage oldPage = (CachedPage) old.page;
                if (!oldPage.hasReferences()) {
                    pageManager.releasePage(oldPage.page);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("unable to release due to ref "
                                + oldPage.page);
                    }
                }
            }

            nodes[i] = wrapper;
        }
        return wrapper;

    }

    public void remove(IStoreKey key, IPage page) throws JoftiException
    {
        releasePage(key, page);
        int i = (int) (key.getId() % maxNodes);

        StoreWrapper wrapper = null;

        synchronized (locks[i]) {

            wrapper = nodes[i];
            if (wrapper != null && wrapper.key.equals(key)) {
                CachedPage oldPage = (CachedPage) wrapper.page;
                oldPage.releaseReference();
                if (!oldPage.hasReferences()) {
                    pageManager.releasePage(oldPage.page);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("unable to release due to ref "
                                + oldPage.page);
                    }
                }

                nodes[i] = null;

            }

        }
        doRemove(key);

    }

}
