package com.jofti.store;

import java.nio.ByteBuffer;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jofti.btree.BTree;
import com.jofti.btree.IPage;
import com.jofti.btree.LeafNodeEntry;
import com.jofti.core.IStoreKey;
import com.jofti.exception.JoftiException;

public class LRUStoreManager extends AbstractStoreManager
{

    private static Log log        = LogFactory
                                          .getLog(AbstractStoreManager.class);

    protected Map      lruMap     = null;

    private Object     objectLock = new Object();

    public void init(Properties properties) throws JoftiException
    {

        if (log.isInfoEnabled()) {
            log.info("Initialising LRU store manager");
        }

        super.init(properties);

        lruMap = new LinkedHashMap() {
            protected final int limit = maxNodes;

            /*
             * (non-Javadoc)
             * 
             * @see java.util.LinkedHashMap#removeEldestEntry(java.util.Map.Entry)
             */
            protected boolean removeEldestEntry(Entry eldest)
            {
                // TODO Auto-generated method stub
                if (size() > limit) {
                    StoreWrapper wrapper = (StoreWrapper) eldest.getValue();
                    CachedPage page = (CachedPage) wrapper.page;
                    if (!page.hasReferences()) {
                        pageManager.releasePage(page.page);
                    } else {
                        log.debug("reference held - unable to release page "
                                + page.page);
                    }
                    return true;
                }

                return false;
            }
        };

        if (log.isInfoEnabled()) {
            log.info("Initialised LRu store manager");
        }
    }

    public IStoreKey store(IStoreKey key, IPage page) throws JoftiException
    {

        StoreWrapper wrapper = null;

        // get the number of positions we need to store the data
        int limit = page.getBuffer().limit();
        FilePositionHolder[] array = allocatePositions(key.getFilePositions(),
                limit);
        key.setFilePositions(array);

        // make sure we are storing a copy here
        ByteBuffer buf = pageManager.acquireBuffer(limit);

        doStore(key, page.copyBuffer(buf));
        // release the buffer we have just stored
        pageManager.releaseBuffer(buf);

        wrapper = new StoreWrapper(key, page);

        // doStore(key,obj);
        synchronized (objectLock) {
            ((CachedPage) page).releaseReference();
            lruMap.put(key, wrapper);
        }

        return null;
    }

    protected IPage getNewPage(int size)
    {
        IPage temp = doGetNewPage(size);

        // wrap the page
        CachedPage cPage = new CachedPage(temp);
        cPage.entries = new LeafNodeEntry[BTree.getMaxNodeSize()];
        // add a usage reference here
        cPage.acquireReference();
        return cPage;
    }

    public void releasePage(IStoreKey key, IPage page)
    {
     synchronized (objectLock) {
            ((CachedPage) page).releaseReference();
        }
    }

    public StoreWrapper retrieve(IStoreKey key) throws JoftiException
    {
        StoreWrapper obj = null;
        synchronized (objectLock) {
            obj = (StoreWrapper) lruMap.get(key);
            if (obj != null) {
                ((CachedPage) obj.page).acquireReference();
                return obj;
            }
        }

        // otherwise replace the val

        IPage page = doRetrieve(key);
        // got to get from store

        CachedPage cPage = new CachedPage(page);
        cPage.entries = new LeafNodeEntry[BTree.getMaxNodeSize()];
        cPage.acquireReference();

        obj = new StoreWrapper(key, cPage);
        synchronized (objectLock) {
            lruMap.put(key, obj);
        }
        return obj;

    }

    public void remove(IStoreKey key, IPage page) throws JoftiException
    {
        releasePage(key, page);
        synchronized (objectLock) {

            StoreWrapper wrapper = (StoreWrapper) lruMap.remove(key);
            CachedPage cPage = (CachedPage) wrapper.page;
            cPage.releaseReference();
            if (!cPage.hasReferences()) {
                pageManager.releasePage(cPage.page);
            }
        }
        doRemove(key);

    }

    public void removeAll() throws JoftiException
    {
        java.util.Iterator it = lruMap.entrySet().iterator();
        int size = lruMap.size();
        for (int i = 0; i < size; i++) {
            Map.Entry entry = (Map.Entry) it.next();
            StoreWrapper wrap = (StoreWrapper) entry.getValue();

            CachedPage old = (CachedPage) wrap.page;
            old.releaseReference();
            if (!old.hasReferences()) {
                pageManager.releasePage(old.page);
            }

        }
        lruMap.clear();
        doRemoveAll();
    }


}
