package com.jofti.util;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.NoSuchElementException;





import java.io.Serializable;
import java.lang.ref.SoftReference;




/**
 * 
 * This is a specific implementation of a Set (heavily based on the HashMap code) that provides for quicker and cached 
 * return of entries in the toArray method. </p>
 * Once a call to the toArray has been made, the resulting Object[] is cached for future calls. </p>
 * 
 * Any mutator method called on the set will clear the cached array. In addition, the cached array is held as a SoftReference so the 
 * Garbage Collector can clear the value if it needs to. </p>
 * The degenerate behaviour without the cache is around 50% of the Hashset implementation.
 *  
 * @author swoodcock
 *
 */
public class FastSet extends AbstractSet
 implements Cloneable, Serializable
{
	transient SoftReference lookupArray = null;
  
	  /**
     * The default initial capacity - MUST be a power of two.
     */
    static final int DEFAULT_INITIAL_CAPACITY = 2;

    /**
     * The maximum capacity, used if a higher value is implicitly specified
     * by either of the constructors with arguments.
     * MUST be a power of two <= 1<<30.
     */
    static final int MAXIMUM_CAPACITY = 1 << 30;

    /**
     * The load factor used when none specified in constructor.
     **/
    static final float DEFAULT_LOAD_FACTOR = 0.75f;

    /**
     * The table, resized as necessary. Length MUST Always be a power of two.
     */
     transient Entry[] table;

    /**
     * The number of key-value mappings contained in this identity hash map.
     */
     transient int size;
  
    /**
     * The next size value at which to resize (capacity * load factor).
     * @serial
     */
    int threshold;
  
    /**
     * The load factor for the hash table.
     *
     * @serial
     */
    float loadFactor;

    /**
     * The number of times this HashMap has been structurally modified
     * Structural modifications are those that change the number of mappings in
     * the HashMap or otherwise modify its internal structure (e.g.,
     * rehash).  This field is used to make iterators on Collection-views of
     * the HashMap fail-fast.  (See ConcurrentModificationException).
     */
    transient volatile int modCount;
 

  
    /**
     * Constructs an empty <tt>HashMap</tt> with the default initial
     * capacity and the default load factor (0.75).
     *
     * @throws IllegalArgumentException if the initial capacity is negative.
     */
    public FastSet() {
        this.loadFactor = DEFAULT_LOAD_FACTOR;
        threshold = (int)(DEFAULT_INITIAL_CAPACITY * DEFAULT_LOAD_FACTOR);
        table = new Entry[DEFAULT_INITIAL_CAPACITY];
    }
    
    /**
     * Value representing null keys inside tables.
     */
    static final Object NULL_KEY = new Object();
    
    static Object maskNull(Object key) {
        return (key == null ? NULL_KEY : key);
    }

    /**
     * Returns key represented by specified internal representation.
     */
    static Object unmaskNull(Object key) {
        return (key == NULL_KEY ? null : key);
    }
    
    static class Entry {
        Object value;
        final int hash;
        Entry next;

        /**
         * Create new entry.
         */
        Entry(int h, Object k,  Entry n) { 
            next = n;
            value = k;
            hash = h;
        }

        public Object getValue() {
            return unmaskNull(value);
        }
    
    
        public Object setValue(Object newValue) {
            Object oldValue = value;
            value = newValue;
            return oldValue;
        }
    
        public boolean equals(Object o) {
            if (!(o instanceof Map.Entry))
                return false;
            Entry e = (Entry)o;
            Object k1 = value;
            Object k2 = e.value;
            if (k1 == k2 || (k1 != null && k1.equals(k2))) { 
                    return true;
            }
            return false;
        }
    
        public int hashCode() {
            return (value==NULL_KEY ? 0 : value.hashCode());
        }
    
        public String toString() {
            return  getValue().toString();
        }

        /**
         * This method is invoked whenever the value in an entry is
         * overwritten by an invocation of put(k,v) for a key k that's already
         * in the HashMap.
         */
        void recordAccess(HashMap m) {
        }

        /**
         * This method is invoked whenever the entry is
         * removed from the table.
         */
        void recordRemoval(HashMap m) {
        }
    }

    public Iterator iterator() {
    	return new KeyIterator();
        }
    
    public int size() {
    	return size;
        }
    public boolean isEmpty() {
        return size == 0;
    }
    
    
    static int hash(Object x) {
        int h = x.hashCode();

        h += ~(h << 9);
        h ^=  (h >>> 14);
        h +=  (h << 4);
        h ^=  (h >>> 10);
        return h;
    }

    /** 
     * Check for equality of non-null reference x and possibly-null y. 
     */
    static boolean eq(Object x, Object y) {
        return x == y || x.equals(y);
    }

    /**
     * Returns index for hash code h. 
     */
    static int indexFor(int h, int length) {
        return h & (length-1);
    }
    
    public boolean contains(Object o) {
    	

    	        Object k = maskNull(o);
    	        int hash = hash(k);
    	        int i = indexFor(hash, table.length);
    	        Entry e = table[i]; 
    	        while (e != null) {
    	            if (e.hash == hash && eq(k, e.value)) 
    	                return true;
    	            e = e.next;
    	        }
    	        return false;
    	    }

    private void clearCache(){
        lookupArray  = new SoftReference(null);

        
    }
    public boolean add(Object o) {
	        // set toArray cache to be null
    		clearCache();
            
    	        Object k = maskNull(o);
    	        int hash = hash(k);
    	        int i = indexFor(hash, table.length);

    	        for (Entry e = table[i]; e != null; e = e.next) {
    	            if (e.hash == hash && eq(k, e.value)) {
    	                Object oldValue = e.value;
    	                e.value = k;
    	               
    	                return oldValue ==null;
    	            }
    	        }

    	        modCount++;
    	        addEntry(hash, k, i);
    	        return false;
    	    

        }
    
    public boolean remove(Object o) {
    	
    	  Entry e = removeEntryForKey(o);
          return (e == null ? false : true);
        }
    
   
    /**
     * Removes and returns the entry associated with the specified key
     * in the HashMap.  Returns null if the HashMap contains no mapping
     * for this key.
     */
    Entry removeEntryForKey(Object value) {
        Object k = maskNull(value);
        int hash = hash(k);
        int i = indexFor(hash, table.length);
        Entry prev = table[i];
        Entry e = prev;

        while (e != null) {
            Entry next = e.next;
            if (e.hash == hash && eq(k, e.value)) {
                modCount++;
                size--;
                if (prev == e) 
                    table[i] = next;
                else
                    prev.next = next;
                return e;
            }
            prev = e;
            e = next;
        }
       if (e != null){
    	   clearCache();
       }
        return e;
    }

    public void clear() {
        // set toArray cache to be null
    	clearCache();
        modCount++;
        Entry tab[] = table;
        for (int i = 0; i < tab.length; i++) 
            tab[i] = null;
        size = 0;
    }
    public boolean removeAll(Collection c) {
    	boolean modified =false;
    clearCache();
    if (size() > c.size()) {
        for (Iterator i = c.iterator(); i.hasNext(); )
            modified |= remove(i.next());
    } else {
        for (Iterator i = iterator(); i.hasNext(); ) {
            if(c.contains(i.next())) {
                i.remove();
                modified = true;
            }
        }
    }
    return modified;
    
    }
    /**
     * Add a new entry with the specified key, value and hash code to
     * the specified bucket.  It is the responsibility of this 
     * method to resize the table if appropriate.
     *
     * Subclass overrides this to alter the behavior of put method.
     */
    void addEntry(int hash, Object value,  int bucketIndex) {
        table[bucketIndex] = new Entry(hash, value, table[bucketIndex]);
        if (size++ >= threshold) 
            resize(2 * table.length);
    }
    
    void resize(int newCapacity) {
        Entry[] oldTable = table;
        int oldCapacity = oldTable.length;
        if (oldCapacity == MAXIMUM_CAPACITY) {
            threshold = Integer.MAX_VALUE;
            return;
        }

        Entry[] newTable = new Entry[newCapacity];
        transfer(newTable);
        table = newTable;
        threshold = (int)(newCapacity * loadFactor);
    }
    
    /** 
     * Transfer all entries from current table to newTable.
     */
    void transfer(Entry[] newTable) {
        Entry[] src = table;
        int newCapacity = newTable.length;
        for (int j = 0; j < src.length; j++) {
            Entry e = src[j];
            if (e != null) {
                src[j] = null;
                do {
                    Entry next = e.next;
                    int i = indexFor(e.hash, newCapacity);  
                    e.next = newTable[i];
                    newTable[i] = e;
                    e = next;
                } while (e != null);
            }
        }
    }
    
    
    
    
    public Object[] toArray() {
        Object[] result = null;
        if (lookupArray != null){
            result =(Object[])lookupArray.get();
        }
   		
    		if (result ==null){
    			result= new Object[size];
		    	int resultCounter =0;
		    	Entry n = null;
		    	int length = table.length-1;
		        for (int i = length; i >= 0 ; i--){
		        	n = table[i];
		        	while (n != null){
		        		result[resultCounter++]= n.value;
		        		n = n.next;
		        	}
		          
		        }
		        lookupArray = new SoftReference(result);
    		}
        return result;
        }
    
   
    
    
    public boolean containsAll(Collection c) {
    	Iterator e = c.iterator();
    	while (e.hasNext())
    	    if(!contains(e.next()))
    		return false;

    	return true;
        }

    private class KeyIterator extends HashIterator {
        public Object next() {
            return nextEntry().getValue();
        }
    }
    
    private abstract class HashIterator implements Iterator {
        Entry next;                  // next entry to return
        int expectedModCount;        // For fast-fail 
        int index;                   // current slot 
        Entry current;               // current entry

        HashIterator() {
            expectedModCount = modCount;
            Entry[] t = table;
            if (t ==null){
    
            }
            int i = t.length;
            Entry n = null;
            if (size != 0) { // advance to first entry
                while (i > 0 && (n = t[--i]) == null)
                    ;
            }
            next = n;
            index = i;
        }

        public boolean hasNext() {
            return next != null;
        }

        Entry nextEntry() { 
            if (modCount != expectedModCount)
                throw new ConcurrentModificationException();
            Entry e = next;
            if (e == null) 
                throw new NoSuchElementException();
                
            Entry n = e.next;
            Entry[] t = table;
            int i = index;
            while (n == null && i > 0)
                n = t[--i];
            index = i;
            next = n;
            return current = e;
        }

        public void remove() {
        	
            if (current == null)
                throw new IllegalStateException();
            if (modCount != expectedModCount)
                throw new ConcurrentModificationException();
            // set toArray cache to be null
           clearCache();
            Object k = current.value;
            current = null;
            removeEntryForKey(k);
            expectedModCount = modCount;
        }

    }
    
    private void writeObject(java.io.ObjectOutputStream s)
    throws java.io.IOException {
// Write out any hidden serialization magic
    	s.defaultWriteObject();

    // Write out HashMap capacity and load factor
    s.writeFloat(loadFactor);

    // Write out size
    s.writeInt(size);

// Write out all elements in the proper order.
for (Iterator i=new KeyIterator(); i.hasNext(); )
        s.writeObject(i.next());
}

/**
 * Reconstitute the <tt>HashSet</tt> instance from a stream (that is,
 * deserialize it).
 */
private void readObject(java.io.ObjectInputStream s)
    throws java.io.IOException, ClassNotFoundException {
// Read in any hidden serialization magic
s.defaultReadObject();

    // Read in HashMap capacity and load factor and create backing HashMap
    float loadFactor = s.readFloat();
    
    this.loadFactor = DEFAULT_LOAD_FACTOR;
    threshold = (int)(DEFAULT_INITIAL_CAPACITY * DEFAULT_LOAD_FACTOR);
    table = new Entry[DEFAULT_INITIAL_CAPACITY];
    
  
    // Read in size
    int size = s.readInt();

// Read in all elements in the proper order.
for (int i=0; i<size; i++) {
        Object e = s.readObject();
        add(e);
    }
}
    
}


