import moment from 'moment';

let timezoneOffset = 0;

export default {

    /**
     * Format the date.
     *
     * @param {Date|string} date The date/time to format, eg '19:45' or '2020-01-07T17:45:00.000Z' or Date object
     * @param {string} format Moment-js date format
     * @return {string} The formatted date
     */
    dateFormat: (date, format) => {
        if (!format) {
            throw new Error(`Format is required "${format}"`);
        }
        if (!date) {
            return '';
        }

        let result;

        // Detect time format; eg 19:45
        if (date.length && /^[0-2][0-9]:[0-5][0-9]$/g.test(date)) {
            // We accept time as string for now.
            const momentObject = moment(date, 'HH:mm');
            momentObject.add(timezoneOffset, 'minutes');
            result = momentObject.format(format);
        } else {
            // Let momentJS figure out the date format; eg 2020-01-07T17:45:00.000Z
            const momentObject = moment(date);
            momentObject.add(timezoneOffset, 'minutes');
            result = momentObject.format(format);
        }

        // Date appears to be invalid.
        if (!result || result === 'Invalid date') {
            throw Error(`Invalid date "${date}"`);
        }

        return result;
    },

    /**
     * Set the global timezone offset for displaying time.
     *
     * @param {number} customTimezoneOffset Number of minutes to add to all times
     */
    setTimezoneOffset(customTimezoneOffset) {
        timezoneOffset = customTimezoneOffset;
    }
};
