import Vue from "vue";

/**
 * Finds and highlights arbitrary text in a <mark>.
 * Use v-html in order to load the HTML. Example:
 * A) <span :inner-html.prop="bsn || bsnFromTitle | highlight(searchQuery)"></span>
 * B) {{bsn || bsnFromTitle | highlight(searchQuery)}}
 *
 * @param { string|null } value The string value that should be looked upon for highlighting.
 * @param { string } toHighlight The string value of the text that should be highlighted.
 *
 * @returns { string|null } The value text, with a possible <mark> tag. If none can be found a default message will be returned.
 */
export const highlight = function (value, toHighlight) {
    if (!value || !toHighlight) {
        return value;
    }

    const regExp = new RegExp('(' + escapeRegExp(toHighlight.trim()) + ')', 'i');
    return value.replace(regExp, '<mark class="highlightedText">$1</mark>');
}

// Add the function to the highlight filter.
Vue.filter('highlight', highlight);

/**
 * Escape text to be used in Regular Expression.
 * https://stackoverflow.com/questions/3446170/escape-string-for-use-in-javascript-regex
 *
 * @param {string} value The value to escape
 * @return {string} Escaped value
 */
function escapeRegExp(value) {
    return value.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
}