// eslint-disable-next-line
import {AxiosError} from 'axios'; // jshint ignore:line

export default {

    /**
     * Extract readable error from Axios error response.
     *
     * @param {AxiosError|Error|null|undefined} errorResponse Axios error response
     * @return {string} readable error message
     */
    extractErrorMessage(errorResponse) {
        if (!errorResponse) {
            return '';
        }
        let errorMessage = '';
        if (errorResponse.response) {
            if (errorResponse.response.status === 401) {
                // noinspection SpellCheckingInspection,JSUnresolvedFunction
                errorMessage = "Invalid username or password";
            } else if (errorResponse.response.data.errorDetails || errorResponse.response.data.errorMessage) {
                errorMessage = errorResponse.response.data.errorDetails || errorResponse.response.data.errorMessage;
            } else {
                errorMessage = JSON.stringify(errorResponse.response.data);
            }
        } else {
            errorMessage = errorResponse.message || '' + errorResponse;
        }
        return errorMessage;
    },

    /**
     * Parse the query parameters.
     * https://stackoverflow.com/questions/8486099/how-do-i-parse-a-url-query-parameters-in-javascript
     *
     * @param url Search query (should start with ?)
     * @return {{}} key value map
     */
    parseQueryParameters(url) {
        const query = url ? url.substr(1) : '';
        const result = {};
        query.split('&').forEach(function (part) {
            const item = part.split('=');
            if (item.length >= 2) {
                result[item[0]] = decodeURIComponent(item[1]);
            }
        });
        return result;
    },

    /**
     * Add parameter to given URL.
     * eg. /casemanager -> /casemanager?uid=wpsadmin
     * eg. /casemanager?s=3&uid=wpsadmin
     *
     * @param {string} url The URL to add given parameter to
     * @param {string} parameterName The name of the parameter to add
     * @param {string} parameterValue The value of the parameter to add
     * @return {string} the new URL
     */
    addUrlParameter(url, parameterName, parameterValue) {
        if (!url || typeof url !== 'string') {
            throw new Error(`Given URL is empty or not a string: "${ url }"`);
        }
        const containsQuestionMark = (url.indexOf('?') !== -1);
        return (containsQuestionMark ? `${ url }&` : `${ url }?`) + `${ parameterName }=${ encodeURIComponent(parameterValue) }`;
    }
};


