import DateFormatFilter from '../../../src/components/DateFormatFilter';

describe('should format the given date', () => {

    afterEach(() => {
        // Reset time zone offset.
        DateFormatFilter.setTimezoneOffset(0);
    });

    it('should format 19:00 to 7:00 pm', () => {
        expect(DateFormatFilter.dateFormat('19:00', 'h:mm a', 'en-US')).toBe('7:00 pm');
    });

    it('should format 19:00 to 19:00', () => {
        expect(DateFormatFilter.dateFormat('19:00', 'HH:mm', 'nl-NL')).toBe('19:00');
    });

    it('should format 2020-01-07T17:45:00.000Z to 6:45 pm - string', () => {
        // 2020-01-07T17:45:00.000Z
        const date = new Date();
        date.setFullYear(2020);
        date.setMonth(0);
        date.setDate(7);
        date.setHours(18);
        date.setMinutes(45);
        const dateAsString = `${date}`;
        expect(DateFormatFilter.dateFormat(dateAsString, 'h:mm a', 'en-US')).toBe('6:45 pm');
    });

    it('should format 2020-01-07T17:45:00.000Z to 6:45 pm - date', () => {
        // 2020-01-07T17:45:00.000Z
        const date = new Date();
        date.setFullYear(2020);
        date.setMonth(0);
        date.setDate(7);
        date.setHours(18);
        date.setMinutes(45);
        expect(DateFormatFilter.dateFormat(date, 'h:mm a', 'en-US')).toBe('6:45 pm');
    });

    it('should format 2020-01-07T17:45:00.000Z to 18:45', () => {
        // 2020-01-07T17:45:00.000Z
        const timezoneOffset = new Date().getTimezoneOffset();
        if(timezoneOffset === -60 /* wintertime */ || timezoneOffset === -120 /* summertime */) {
            expect(DateFormatFilter.dateFormat('2020-01-07T17:45:00.000Z', 'HH:mm', 'nl-NL')).toBe('18:45');
        } else if(timezoneOffset === 0) {
            expect(DateFormatFilter.dateFormat('2020-01-07T17:45:00.000Z', 'HH:mm', 'nl-NL')).toBe('17:45');
        } else {
            fail('Unknown timezone offset: ' + timezoneOffset + ', cannot test date');
        }
    });

    it('should format 2020-01-07T17:45:00.000Z with 5 hours offset to 23:45', () => {
        // 2020-01-07T17:45:00.000Z
        DateFormatFilter.setTimezoneOffset(300);
        const timezoneOffset = new Date().getTimezoneOffset();
        if(timezoneOffset === -60 /* wintertime */ || timezoneOffset === -120 /* summertime */) {
            expect(DateFormatFilter.dateFormat('2020-01-07T17:45:00.000Z', 'HH:mm', 'nl-NL')).toBe('23:45');
        } else if(timezoneOffset === 0) {
            expect(DateFormatFilter.dateFormat('2020-01-07T17:45:00.000Z', 'HH:mm', 'nl-NL')).toBe('22:45');
        } else {
            fail('Unknown timezone offset: ' + timezoneOffset + ', cannot test date');
        }
    });

    it('should throw an error in case of an invalid date', () => {
        try {
            DateFormatFilter.dateFormat('whatever', 'HH:mm', 'nl-NL');
            fail('Should have thrown an error');
        } catch (e) {
            // This is ok.
            expect(e.message).toBe('Invalid date "whatever"');
        }
    });

    it('should display nothing in case of an undefined date', () => {
        const result = DateFormatFilter.dateFormat(undefined, 'HH:mm', 'nl-NL');
        expect(result).toBe('');
    });

    it('should throw an error in case of an undefined format', () => {
        try {
            DateFormatFilter.dateFormat('0:00', undefined, 'nl-NL');
            fail('Should have thrown an error');
        } catch (e) {
            // This is ok.
            expect(e.message).toBe('Format is required "undefined"');
        }
    });

    it('should throw an error in case of an undefined locale', () => {
        try {
            DateFormatFilter.dateFormat('0:00', 'HH:mm');
            fail('Should have thrown an error');
        } catch (e) {
            // This is ok.
            expect(e.message).toBe('Locale is required "undefined"');
        }
    });

    it('should throw an error in case of invalid locale', () => {
        try {
            DateFormatFilter.dateFormat('0:00', 'HH:mm', 'NL123');
            fail('Should have thrown an error');
        } catch (e) {
            // This is ok.
            expect(e.message).toBe('Locale is invalid "NL123"');
        }
    });

});
