import Vue from 'vue';
import {mount} from '@vue/test-utils';
import VueI18n from 'vue-i18n';
import messages from '../../../src/messages';
import Login from '../../../src/pages/Login';
import axios from 'axios';

// Mock Axios.
// https://jestjs.io/docs/en/mock-functions.html
// https://stackoverflow.com/questions/45016033/how-do-i-test-axios-in-jest#45017940
jest.mock('axios');

describe('Login component / page', () => {
    let i18n;

    beforeEach(() => {
        Vue.use(VueI18n);
        i18n = new VueI18n({
            locale: 'nl-NL',
            messages
        });
    });

    it('should show the login form', () => {
        const wrapper = mount(Login, {
            i18n,
            propsData: {}
        });

        expect(wrapper.contains('form input#username')).toBe(true);
        expect(wrapper.contains('form input#password')).toBe(true);
        expect(wrapper.contains('form button[type=submit]')).toBe(true);
    });

    it('should fill the username if stored in local storage', () => {
        // Mock local storage.
        window.localStorage.setItem('username', 'Rob');
        const wrapper = mount(Login, {
            i18n,
            propsData: {}
        });

        expect(wrapper.contains('form input#username')).toBe(true);
        expect(wrapper.find('form input#username').element.value).toBe('Rob');
        expect(wrapper.contains('form input#password')).toBe(true);
        expect(wrapper.contains('form button[type=submit]')).toBe(true);
    });

    it('should post the form on submit', () => {
        // Mock local storage.
        const wrapper = mount(Login, {
            i18n,
            propsData: {},
            // Need to be attached to the DOM.
            // https://stackoverflow.com/questions/53382235/trigger-form-submit-on-button-click-in-vue-unit-test
            // https://vue-test-utils.vuejs.org/api/options.html#attachtodocument
            attachToDocument: true
        });

        axios.post.mockResolvedValue(true);
        axios.post.mock.calls.length = 0;

        wrapper.setData({username: 'Rob', password: 'secret'});
        wrapper.find('form button[type=submit]').element.click();
        expect(axios.post.mock.calls.length).toBe(1);
        expect(axios.post.mock.calls[0][0]).toBe('/api/login');
        expect(axios.post.mock.calls[0][1]).toBe('username=Rob&password=secret');
        expect(axios.post.mock.calls[0][2]).toStrictEqual({'headers': {'Content-Type': 'application/x-www-form-urlencoded'}});

        // Required when DOM attached!
        wrapper.destroy();
    });

});
