// 'Request' package to handle the https communication.
// https://www.npmjs.com/package/request
// Execute http requests.
const request = require('request');
const packageJson = require('../package.json');
const config = require('../config.js').config;
const networkServices = require('./network-services');

// User agent is based on package name and version and includes the email address.
/** @namespace packageJson.name */
/** @namespace packageJson.version */
const USER_AGENT_STRING = `${packageJson.name}/${packageJson.version}`;

// Possibility to keep the cookies between requests.
let cookieJar;

const debugPrefix = '\x1B[36mDEBUG\x1B[0m: ';
const verbose = true;

// We enable cookies by default, so they're also used in subsequent requests.
// https://github.com/request/request#readme
cookieJar = request.jar();
// https://github.com/request/request#requestdefaultsoptions
const requestInstance = request.defaults({
    jar: cookieJar,
    // Ignore(allow) self-signed certificates.
    rejectUnauthorized: false,
});

/**
 * Create the server session and login.
 * crumb=`curl -c $COOKIE_PATH --user rob:password "https://git.juurlink.org/jenkins/crumbIssuer/api/xml?xpath=concat(//crumbRequestField,%22:%22,//crumb)"`
 * https://stackoverflow.com/questions/23497819/trigger-parameterized-build-with-curl-and-crumb
 * https://wiki.jenkins.io/display/JENKINS/Remote+access+API
 *
 * @param {string} username
 * @param {string} password
 * @return {Promise<string|Error>} resolved with "crumb" in case of success, reject with error otherwise
 */
exports.getCrumb = function(username, password) {
    return new Promise((resolveFn, rejectFn) => {

        if (!username || !username.trim() || !password || !password.trim()) {
            throw Error('Username and password are both required');
        }

        let url = config.JENKINS_URL + 'crumbIssuer/api/xml?xpath=concat(//crumbRequestField,%22:%22,//crumb)';
        url = networkServices.addCredentialsToUrl(url, username, password);

        const parsedUrl = networkServices.parseUrl(url);

        const headers = {
            'Host': parsedUrl.host,
            'User-Agent': USER_AGENT_STRING,
            'Accept': '*/*',
            'Accept-Language': 'en-US',
            'Origin': parsedUrl.origin,
            'Referer': parsedUrl.origin
        };

        if (verbose) {
            console.log(`${debugPrefix}Request url: "${url}"`);
            console.log(`${debugPrefix}Request headers: "${JSON.stringify(headers, null, 4)}"`);
        }

        requestInstance.get({
            jar: cookieJar,
            // Ignore(allow) self-signed certificates.
            rejectUnauthorized: false,
            url: url,
            headers: headers

        }, (error, response, data) => {
            if (error || response.statusCode >= 400) {
                console.log(`Accessing url ${url} failed`, `${error ? error : ''}${response ? response.statusCode : ''}`);
                rejectFn(`${error ? error : ''}${response.statusCode}`.trim());
                return;
            }

            // https://github.com/salesforce/tough-cookie#getcookiescurrenturl-options-cberrcookies
            const cookies = cookieJar.getCookies(url, {allPaths: true});
            if (verbose) {
                console.log(`${debugPrefix}Login statusCode: ${response.statusCode}`);
                console.log(`${debugPrefix}Cookies Jar: ${JSON.stringify(cookies, null, 2)}`);
            }
            if (cookies && cookies.length) {
                console.log('==== cookies: ', cookies)
            }
            resolveFn(data);
        });
    });
};

/**
 * Start given Jenkins job.
 * https://stackoverflow.com/questions/23497819/trigger-parameterized-build-with-curl-and-crumb
 * https://wiki.jenkins.io/display/JENKINS/Remote+access+API
 *
 * @param {string} crumb Variable used for CSRF protection
 * @param {string} jobName Name of the Jenkins job to start
 * @param {string} assetName Value of assetName build parameter
 * @param {string} username Username
 * @param {string} password Password
 * @return {Promise<string|Error>} resolved in case of success, reject with error otherwise
 */
exports.startBuild = function(crumb, jobName, assetName, username, password) {
    return new Promise((resolveFn, rejectFn) => {

        // Fast fail.
        if (!crumb || !crumb.trim() || crumb.indexOf(':') === -1) {
            throw Error('Invalid crumb: ' + crumb);
        }
        if (!jobName || !jobName.trim()) {
            throw Error('Invalid jobName: ' + jobName);
        }
        if (!assetName || !assetName.trim()) {
            throw Error('Invalid assetName: ' + assetName);
        }
        if (!username || !username.trim() || !password || !password.trim()) {
            throw Error('Username and password are both required');
        }

        let url = config.JENKINS_URL + 'job/' + jobName + '/buildWithParameters?assetName=' + encodeURIComponent(assetName);
        url = networkServices.addCredentialsToUrl(url, username, password);

        const parsedUrl = networkServices.parseUrl(url);

        const headers = {
            'Host': parsedUrl.host,
            'User-Agent': USER_AGENT_STRING,
            'Accept': '*/*',
            'Accept-Language': 'en-US',
            'Origin': parsedUrl.origin,
            'Referer': parsedUrl.origin
        };

        // Add Jenkins crumb header.
        const [crumbHeader, crumbValue] = crumb.split(':');
        headers[crumbHeader] = crumbValue;

        if (verbose) {
            console.log(`${debugPrefix}Request url: "${url}"`);
            console.log(`${debugPrefix}Request headers: "${JSON.stringify(headers, null, 4)}"`);
        }

        requestInstance.post({
            jar: cookieJar,
            // Ignore(allow) self-signed certificates.
            rejectUnauthorized: false,
            url: url,
            headers: headers

        }, (error, response) => {
            if (error || response.statusCode >= 400) {
                console.log(`Accessing url ${url} failed`, `${error ? error : ''}${response.statusCode}`);
                rejectFn(`${error ? error : ''}${response.statusCode}`.trim());
                return;
            }

            // https://github.com/salesforce/tough-cookie#getcookiescurrenturl-options-cberrcookies
            const cookies = cookieJar.getCookies(url, {allPaths: true});
            if (verbose) {
                console.log(`${debugPrefix}Login statusCode: ${response.statusCode}`);
                console.log(`${debugPrefix}Cookies Jar: ${JSON.stringify(cookies, null, 2)}`);
            }
            if (cookies && cookies.length) {
                console.log('==== cookies: ', cookies)
            }
            resolveFn();
        });
    });
};
