/**
 * Add basic authentication credentials to the given URL.
 *
 * @param {string} username
 * @param {string} password
 * @param {string} url URL to add credentials to
 * @return {string} URL with credentials
 */
exports.addCredentialsToUrl = function (url, username, password) {
    return url.replace(/^(https?:\/\/)/i, `$1${username}:${password}@`);
};

/**
 * Trim excessive whitespace within the string and trim leading and trailing whitespace completely.
 *
 * @param {String} text String to trim
 * @return {String} Trimmed string or empty string in case the given parameter was not truthy
 */
exports.trimExcessiveWhitespace = function (text) {
    return (text || '').replace(/\s+/g, ' ').trim();
};

/**
 * Trim start slash, if available.
 *
 * @param {string} url URL to strip / from
 * @returns {string} URL with starting / removed
 */
exports.trimStartSlash = function (url) {
    return (url || '').replace(/^\/+/g, '');
};

/**
 * Trim start slash, if available.
 *
 * @param {string} url URL to strip / from
 * @returns {string} URL with starting / removed
 */
exports.trimEndSlash = function (url) {
    return (url || '').replace(/\/+$/g, '');
};

/**
 * Parse given URL.
 * https://stackoverflow.com/questions/736513/how-do-i-parse-a-url-into-hostname-and-path-in-javascript
 *
 * @param {string} url
 * @returns {boolean|*|Promise<Response | undefined>|RegExpMatchArray|{hostname: (*|string|string), protocol: (*|string|string), search: (*|string|string), password: (*|string|string), port: (*|string|string), origin: (*|string|string), host: (*|string|string), href: (*|string|string), hash: (*|string|string), pathname: (*|string), username: (*|string|string)}}
 */
exports.parseUrl = function (url) {
    // eslint-disable-next-line no-useless-escape
    const m = url.match(/^(([^:\/?#]+:)?(?:\/\/((?:([^\/?#:]*):([^\/?#:]*)@)?([^\/?#:]*)(?::([^\/?#:]*))?)))?([^?#]*)(\?[^#]*)?(#.*)?$/),
        r = {
            hash: m[10] || '',                   // #asd
            host: m[3] || '',                    // localhost:257
            hostname: m[6] || '',                // localhost
            href: m[0] || '',                    // http://username:password@localhost:257/deploy/?asd=asd#asd
            origin: m[1] || '',                  // http://username:password@localhost:257
            pathname: m[8] || (m[1] ? '/' : ''), // /deploy/
            port: m[7] || '',                    // 257
            protocol: m[2] || '',                // http:
            search: m[9] || '',                  // ?asd=asd
            username: m[4] || '',                // username
            password: m[5] || ''                 // password
        };
    if (r.protocol.length === 2) {
        r.protocol = 'file:///' + r.protocol.toUpperCase();
        r.origin = r.protocol + '//' + r.host;
    }
    r.href = r.origin + r.pathname + r.search + r.hash;
    const result = m && r;
    if (result && result.host && result.host.indexOf('@') !== -1) {
        result.host = result.host.substr(result.host.indexOf('@') + 1);
    }

    return result;
};
