// 'Request' package to handle the https communication.
// https://www.npmjs.com/package/request
// Execute http requests.
const request = require('request');
const packageJson = require('../package.json');
const networkServices = require('./network-services');
const config = require('../config.js').config;

// Start Jenkins build with parameters.
// https://wiki.jenkins.io/display/JENKINS/Parameterized+Build
// User agent is based on package name and version and includes the email address.
/** @namespace packageJson.name */
/** @namespace packageJson.version */
const USER_AGENT_STRING = `${packageJson.name}/${packageJson.version}`;

// Possibility to keep the cookies between requests.
let cookieJar;

const debugPrefix = '\x1B[36mDEBUG\x1B[0m: ';
const verbose = true;

// We enable cookies by default, so they're also used in subsequent requests.
// https://github.com/request/request#readme
cookieJar = request.jar();
// https://github.com/request/request#requestdefaultsoptions
const requestInstance = request.defaults({
    jar: cookieJar,
    // Ignore(allow) self-signed certificates.
    rejectUnauthorized: false,
});

/**
 * Create the server session and login.
 *
 * @param {string} username The username
 * @param {string} password The password
 * @return {Promise<void|Error>} resolved in case of success, reject with error otherwise
 */
exports.login = function (username, password) {
    return new Promise((resolveFn, rejectFn) => {

        if (!username || !username.trim() || !password || !password.trim()) {
            throw Error('Username and password are both required');
        }

        const url = `${config.NEXUS_URL}service/rapture/session`;
        const parsedUrl = networkServices.parseUrl(url);

        // Call the real service.
        const headers = {
            'Host': parsedUrl.host,
            'User-Agent': USER_AGENT_STRING,
            'Accept': 'text/html',
            'Accept-Language': 'en-US',
            'Content-Type': 'application/x-www-form-urlencoded',
            'Origin': parsedUrl.origin,
            'Referer': parsedUrl.origin
        };

        if (verbose) {
            console.log(`${debugPrefix}Request url: "${url}"`);
            console.log(`${debugPrefix}Request headers: "${JSON.stringify(headers, null, 4)}"`);
        }

        requestInstance.post({
            url: url,
            headers: headers,
            form: {
                username: Buffer.from(username).toString('base64'),
                password: Buffer.from(password).toString('base64')
            }
        }, (error, response) => {
            if (error || response.statusCode >= 400) {
                console.log(`Accessing url ${url} failed`, `${error ? error : ''}${response.statusCode}`);
                rejectFn(`${error ? error : ''}${response.statusCode}`.trim());
                return;
            }

            // https://github.com/salesforce/tough-cookie#getcookiescurrenturl-options-cberrcookies
            const cookies = cookieJar.getCookies(url, {allPaths: true});
            if (verbose) {
                console.log(`${debugPrefix}Login statusCode: ${response.statusCode}`);
                console.log(`${debugPrefix}Cookies Jar: ${JSON.stringify(cookies, null, 2)}`);
            }
            if (cookies && cookies.length) {
                console.log('==== cookies: ', cookies)
            }
            resolveFn();
        });
    });
};

/**
 * Search for assets.
 *
 * @param {string|undefined} searchTerm Where to search for
 * @return {Promise<{lastUpdated:string,name:string}[]|Error>}
 */
exports.search = function (searchTerm) {

    const repositoryName = config.NEXUS_REPOSITORY_NAME;
    return new Promise((resolveFn, rejectFn) => {

        const json = {
            "action": "coreui_Component",
            "method": "readAssets",
            "data": [{
                "page": 1,
                "start": 0,
                "limit": 10000,
                "sort": [{"property": "name", "direction": "ASC"}],
                "filter": [
                    {"property": "repositoryName", "value": repositoryName},
                    {"property": "filter", "value": searchTerm || ''}]
            }],
            "type": "rpc",
            "tid": 55
        };

        const url = "http://localhost:8081/service/extdirect";
        const parsedUrl = networkServices.parseUrl(url);

        // Call the real service.
        const headers = {
            'Host': parsedUrl.host,
            'User-Agent': USER_AGENT_STRING,
            'Pragma': 'no-cache',
            'Accept': '*/*',
            'Accept-Language': 'en-US',
            'Content-Type': 'application/json',
            'Origin': parsedUrl.origin,
            'Referer': parsedUrl.origin
        };

        if (verbose) {
            console.log(`${debugPrefix}Request url: "${url}"`);
            console.log(`${debugPrefix}Request headers: "${JSON.stringify(headers, null, 4)}"`);
        }

        requestInstance.post({
            url: url,
            headers: headers,
            json
        }, (error, response, data) => {
            if (error || response.statusCode >= 400) {
                console.log(`Accessing url ${url} failed`, `${error ? error : ''}${response.statusCode}`);
                rejectFn(`${error ? error : ''}${response.statusCode}`.trim());
                return;
            }
            if (verbose) {
                console.log(`${debugPrefix}Search statusCode: ${response.statusCode}`);
            }

            /*
            {
                "id": "1c787a03db8209eb8ccb206012a1e42b",
                "name": "nexus-scraper/-/nexus-scraper-1.0.0.tgz",
                "format": "npm",
                "contentType": "application/x-tgz",
                "size": 2863,
                "repositoryName": "test",
                "lastUpdated": "2020-04-02T08:31:49.211Z",
                "lastAccessed": "2020-04-02T08:31:49.202Z",
                "blobRef": "test@F19C361C-42F627A6-BF68741B-5FE433F2-2D09C481:2511b603-e053-4c38-9ada-ba36d744c625",
                "componentId": "7f6379d32f8dd78f21bfae9de8d56af9"
            }
             */

            /** @type {{lastUpdated:string,name:string,componentId:string}[]} */
            const rawItems = data.result.data;
            let items = rawItems.map(item => {
                return {
                    date: item.lastUpdated,
                    longName: item.name,
                    name: item.name.split('/').pop()
                }
            });

            // Only keep the newest version from every unique name.
            items = items.filter(item => item.name.endsWith('.tgz'));
            items = sortByFullNameAndVersion(items);
            items = filterKeepUnique(items);

            resolveFn(items);
        });
    });
};


exports.login('admin', 'admin123')
    .then(() => {
        console.log('==== Login SUCCESS');

        exports.search('.tgz')
            .then(items => {
                console.log('==== Search SUCCESS: ', JSON.stringify(items, null, 2));
            });
    });


/**
 * Sort the items, newest first.
 * Array will be modified!
 *
 * @param {{date:string,name:string}[]} items
 * @return {{date:string,name:string}[]}
 */
function sortByFullNameAndVersion(items) {
    items.sort((a, b) => {
        const dateA = a.date;
        const dateB = b.date;
        if (dateA < dateB) {
            return -1;
        }
        if (dateA > dateB) {
            return 1;
        }
        // Must be equal
        return 0;
    });
    return items;
}

/**
 * @param {{date:string,name:string}[]} items
 * @return {{date:string,name:string}[]}
 */
function filterKeepUnique(items) {
    const unique = [];
    return items.filter(item => {
        const isNew = unique.indexOf(item.name) === -1;
        if (isNew) {
            unique.push(item.name);
        }
        return isNew;
    });
}

// Make private functions testable.
exports._sortByFullNameAndVersion = sortByFullNameAndVersion;
exports._filterKeepUnique = filterKeepUnique;