const pjson = require('../package.json');
const http = require('http');
const pathFn = require('path');
const querystring = require('querystring');
const url = require('fast-url-parser');
const cookie = require('cookie');
const uuidv1 = require('uuid/v1');
const LruCache = require('lru-cache');
const jenkinsServices = require('./jenkins-services');
const networkServices = require('./network-services');
const nexusServices = require('./nexus-services');
// handle static content.
const handler = require('serve-handler');
const config = require('../config').config;

/** OpenShift expects the application to run at port 8080. */
const HTTP_PORT = 8080;

const caching = false;

// Holds users memory session data. Max n active sessions.
const SESSION_CACHE = new LruCache({max: 25});

const COOKIE_SESSION_ID_NAME = 'SESSIONID';
const TIME_ONE_WEEK_SECONDS = 60 * 60 * 24 * 7;
const FORM_URLENCODED = 'application/x-www-form-urlencoded';

/*
 * (http[s]) Web related service functions.
 */

// https://blog.risingstack.com/your-first-node-js-http-server/

// Keep the software version of the running instance.
const version = pjson.version;

/**
 * Start webservice to show all current orders.
 * @returns {void}
 */
exports.startServer = () => {
    const server = http.createServer({}, requestHandler);
    server.listen(HTTP_PORT, (err) => {
        if (err) {
            return console.error('Error starting web server: ', err);
        }
        console.log(`Server is listening on http://localhost:${HTTP_PORT}`);
    });
};

/**
 * Handle the web server request.
 *
 * @param request HTTP Request
 * @param response HTTP response
 */
function requestHandler(request, response) {

    // Get the path from the url (eg. / for index page).
    const path = extractRelativePath(request.url);

    // Handle API calls.
    if (path.indexOf('/api/') === 0) {
        const session = createSessionIfNotExists(request, response);

        if (!performAuthentication(session, request, response)) {
            return;
        }

        if (path === '/api/login') {
            handleLogin(session, request, response);
            return;
        }
        if (path === '/api/logout') {
            handleLogout(session, request, response);
            return;
        }
        if (path === '/api/version') {
            handleVersion(request, response);
            return;
        }
        if (path === '/api/assets') {
            handleAssets(session, request, response);
            return;
        }
        if (path === '/api/startBuild') {
            handleStartBuild(session, request, response);
            return;
        }
        response.writeHead(404, {'Content-Type': 'application/json'});
        response.end('{"errorCode":404, "errorMessage":"Not Found"}');
        return;
    }

    // https://nodejs.org/api/http.html#http_class_http_serverresponse
    // Serve image, css etc.
    return serveAsset(request, response);
}

/**
 * Get session.
 *
 * @param request Server request
 * @return {Object|undefined} server side session object or undefined if not found
 */
function getSession(request) {
    // Parse the cookies on the request
    const cookies = cookie.parse(request.headers.cookie || '');
    const sessionId = cookies[COOKIE_SESSION_ID_NAME];
    const session = SESSION_CACHE.get(sessionId);
    // Add the session id to the session object.
    if (session) {
        session.id = sessionId;
    }
    return session;
}

/**
 * Ensures the session exists.
 * If not, start new session.
 * If it already exists, update the expiration date of the cookie.
 *
 * @param request Server request
 * @param response Server response
 * @return {Object} server side session object
 */
function createSessionIfNotExists(request, response) {
    let session = getSession(request);
    if (!session) {
        // Create new server side session.
        const sessionId = uuidv1();
        session = {
            id: sessionId,
            loggedIn: false
        };
    }

    // Create or update the server cookie.
    response.setHeader('Set-Cookie', cookie.serialize(COOKIE_SESSION_ID_NAME, session.id, {
        // If httpOnly is true, the cookie cannot be accessed through client side script.
        path: '/',
        httpOnly: true,
        maxAge: TIME_ONE_WEEK_SECONDS
    }));
    SESSION_CACHE.set(session.id, session);
    return session;
}

/**
 * Show the asset.
 *
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 * @return void
 */
function serveAsset(request, response) {

    // Remove set-cookie header, not necessary for static content.
    response.removeHeader('Set-Cookie');

    // Serve-handler documentation: https://www.npmjs.com/package/serve-handler
    const vuePath = pathFn.join(__dirname, '..', 'server_app', 'dist');

    // Only cache in production.
    const headers = [];
    if (caching) {
        headers.push({
            source: '**/*.@(jpg|jpeg|gif|png|ico|svg|css|js)',
            headers: [{
                key: 'Cache-Control',
                value: 'public, max-age=31536000'
            }]
        });
    }
    handler(request, response, {
        public: vuePath,
        directoryListing: false,
        headers: headers
    });
}

/**
 * handle the login POST request.
 *
 * @param {Object} session We need a reference to set "logged in" to true in case of successful login
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 * @return {void}
 */
function handleLogin(session, request, response) {

    if (request.method !== 'POST') {
        response.writeHead(405, {'Content-Type': 'application/json'});
        response.end('{"errorCode":405, ' +
            '"errorMessage":"Method Not Allowed", ' +
            '"errorDetails":"Only POST request allowed for login API call"}');
        return;
    }
    // Process login.
    parsePostRequest(request, result => {
        if (result) {
            checkCredentials(result.username, result.password)
                .then(() => {
                    session.username = result.username;
                    session.password = result.password;
                    session.loggedIn = true;
                    response.writeHead(200, {'Content-Type': 'application/json'});
                    response.end('{"statusCode":200, "statusMessage":"Ok"}');
                })
                .catch(error => {
                    session.loggedIn = false;
                    response.writeHead(401, {'Content-Type': 'application/json'});
                    response.end(JSON.stringify({
                        errorCode: 401,
                        errorMessage: "Unauthorized",
                        errorDetails: `Error: ${error}`
                    }));
                });
            return;
        }
        session.loggedIn = false;
        response.writeHead(401, {'Content-Type': 'application/json'});
        response.end('{"errorCode":401, ' +
            '"errorMessage":"Unauthorized", ' +
            '"errorDetails":"Invalid username and or password"}');
    });
}

/**
 * handle the logout GET request.
 *
 * @param {Object} sessionObject We need a reference to invalidate the session
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 * @return {void}
 */
function handleLogout(sessionObject, request, response) {
    sessionObject.loggedIn = false;
    response.writeHead(200, {'Content-Type': 'application/json'});
    response.end('{"statusCode":200, "statusMessage":"Ok"}');
}

/**
 * Return software version.
 *
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 */
function handleVersion(request, response) {
    response.writeHead(200, {'Content-Type': 'application/json'});
    response.end(JSON.stringify(version));
}

/**
 * Receive the list of assets.
 *
 * @param {Object} session The session contains the username and password
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 */
function handleAssets(session, request, response) {

    const username = session.username;
    const password = session.username;

    nexusServices.search('')
        .then(assets => {
            response.writeHead(200, {'Content-Type': 'application/json'});
            response.end(JSON.stringify({assets: assets}, null, 2));
        })
        .catch(error => {
            // Return the raw error.
            response.writeHead(500, {'Content-Type': 'text/plain'});
            response.end(`${error}`);
        });
}

function sendServerError(response, error) {
    // Just return the raw service error.
    response.writeHead(500, {'Content-Type': 'application/json'});
    response.end(JSON.stringify({
        errorCode: 500,
        errorMessage: "Server Error",
        errorDetails: `${error.substr ? error : JSON.stringify(error)}`
    }, null, 2));
}

/**
 * Start the Jenkins build.
 *
 * @param {Object} session The session contains the username and password
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 */
function handleStartBuild(session, request, response) {

    const username = session.username;
    const password = session.password;

    const queryParams = networkServices.parseUrl(request.url).search;
    const parsedQueryParams = querystring.parse(queryParams ? queryParams.substr(1) /* skip the ? */ : '');
    const assetName = parsedQueryParams.assetName;

    if (!assetName) {
        sendServerError(response, 'No asset name provided');
        return;
    }

    jenkinsServices.getCrumb(username, password)
        .then(crumb => {
            return jenkinsServices.startBuild(crumb, config.JENKINS_JOB, assetName, username, password)
                .then(() => {
                    const jenkinsUrl = config.JENKINS_URL + 'job/' + config.JENKINS_JOB;

                    // This doesn't work.
                    // const jenkinsUrlWithCredentials = networkServices.addCredentialsToUrl(jenkinsUrl, username, password);

                    response.writeHead(200, {'Content-Type': 'text/plain'});
                    response.end(jenkinsUrlWithCredentials);
                })
        })
        .catch(error => {
            sendServerError(response, error);
        });
}

/**
 * Remove query parameters from URL.
 *
 * @param {string} requestUrl Full url with query parameters
 * @return {string} Relative path of the resource
 */
function extractRelativePath(requestUrl) {
    return decodeURIComponent(url.parse(requestUrl).pathname);
}

/**
 * Test if the user is logged in.
 * responds with 401
 *
 * @param session Server side session object with login state
 * @param {module:http.ClientRequest} request
 * @param {module:http.ServerResponse} response
 *
 * @returns {boolean} true if user is logged in, false otherwise
 */
function performAuthentication(session, request, response) {
    if (session.loggedIn) {
        return true;
    }
    // All URLs which start with "/api/login" are accessible without authentication.
    if (request.url && (request.url.indexOf('/api/login') === 0 || request.url.indexOf('/api/version') === 0)) {
        return true;
    }

    // Access Denied.
    response.writeHead(401, {'Content-Type': 'application/json'});
    response.end('{"errorCode":401, "errorMessage":"Unauthorized"}');
    return false;
}

/**
 * Basic function to validate credentials.
 * (We use Jenkins to check the credentials).
 *
 * @param {string|undefined} username Username
 * @param {string|undefined} password Password to test
 * @return {Promise<boolean|string>} Promise which resolves if valid or rejects with error otherwise
 */
function checkCredentials(username, password) {
    return jenkinsServices.getCrumb(username, password);
}

/**
 * Parse the POST request.
 * Source: https://itnext.io/how-to-handle-the-post-request-body-in-node-js-without-using-a-framework-cd2038b93190
 *
 * @param {module:http.ClientRequest} request
 * @param callback Callback will be called with the POST parameters as key/values
 */
function parsePostRequest(request, callback) {
    if (request.headers['content-type'] !== FORM_URLENCODED) {
        callback(null);
        return;
    }
    let body = '';
    request.on('data', chunk => {
        body += chunk.toString();
    });
    request.on('end', () => {
        callback(querystring.parse(body));
    });
}
